﻿using Curse;
using Curse.WAR;
using System;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;
namespace WARDataService
{

    public static partial class DBNPCs
    {        
        private const Byte NUM_BONUSES = 10;
        private const Byte NUM_CRAFTING_BONUSES = 25;

        private static Dictionary<CustomKey, Int32> sNPCNameToId = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());                

        private static Dictionary<CustomKey, DateTime> sHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sNPCConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sNPCLocationConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sNPCLootConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static String sUpdateQuery = null;
        private static String sInsertQuery = null;

        private static String[] sUpdateIgnoredColumns = { "id", "posted", "posted_id", "version_id", "name", "times_killed", "times_looted",  "times_butchered", "times_scavenged"};
        private static String[] sIgnoreInsertColumns = {"id"};
        private static Dictionary<String, String> sUpdateCustomColumns = new Dictionary<string, string>();

        private static String sNPCLocationQueryFormat = "UPDATE npc_location SET " +
                                                        "total=total+1," +
                                                        "updated_id={0}," +
                                                        "updated=GETUTCDATE()" +
                                                        " WHERE npc_id={1} AND pos_x_key=ROUND({4}/100,1) AND pos_y_key=ROUND({5}/100,1)" +
                                                        " IF @@ROWCOUNT = 0 " +
                                                        "INSERT INTO npc_location(" +
                                                        "npc_id,version_id,zone_id," +
                                                        "pos_x,pos_y,pos_x_key,pos_y_key,total,updated_id,posted_id) VALUES(" +
                                                        "{1},{2},{3},{4},{5},ROUND({4}/100,1),ROUND({5}/100,1),1,{0},{0});";

        private static String sNPCLootQueryFormat = "UPDATE npc_loot SET " +
                                                    "total=total+{4}," +                                                   
                                                    "updated_id={5}," +
                                                    "updated=GETUTCDATE()" +
                                                    " WHERE npc_id={1} AND item_id={3} and type={0}" +
                                                    " IF @@ROWCOUNT = 0 " +
                                                    "INSERT INTO npc_loot(type," +
                                                    "npc_id,version_id,item_id," +
                                                    "total,posted_id,updated_id) VALUES(" +
                                                    "{0},{1},{2},{3},{4},{5},{5});";

        private static String sNPCCoinQueryFormat = "UPDATE npc_coin SET " +
                                                   "amount=amount+{1}," +
                                                   "updated_id={3}," +
                                                   "updated=GETUTCDATE()" +
                                                   " WHERE npc_id={0}" +
                                                   " IF @@ROWCOUNT = 0 " +
                                                   "INSERT INTO npc_coin(" +
                                                   "npc_id,amount,version_id," +
                                                   "posted_id,updated_id) VALUES(" +
                                                   "{0},{1},{2},{3},{3});";

        private static String sNpcStatsQueryFormat = "update npc set " +
                                                "times_killed=(times_killed+{0})," +
                                                "times_looted=(times_looted+{1})," +
                                                "times_butchered=(times_butchered+{2})," +
                                                "times_scavenged=(times_scavenged+{3}) where id = {4}";

        private static String sTempTableCreation = "SELECT TOP 0 * INTO #npc FROM npc;ALTER TABLE #npc DROP COLUMN id;ALTER TABLE #npc ADD id int;";
        
        private static DataTable sNpcUpdateSchema = null;
        private static DataTable sNpcInsertSchema = null;

        private static DataTable sNpcMerchandiseSchema = null;

        public static void Initialize()
        {            
            sHistory.Clear();
            sNPCNameToId.Clear();                                
           
            foreach (String locale in DB.sLocalizations)
            {
                DB.PopulateNameToID(sNPCNameToId, "npc", locale);
            }
            DB.LoadSchema(ref sNpcUpdateSchema, "#npc", sTempTableCreation);
            DB.LoadSchema(ref sNpcInsertSchema, "npc");
            DB.LoadSchema(ref sNpcMerchandiseSchema, "npc_merchandise");

            sInsertQuery = DB.GetInsertSQL(sNpcInsertSchema, sIgnoreInsertColumns);

            sUpdateCustomColumns.Add("min_level", "min_level=dbo.Lowest(npc.min_level,tmp.min_level)");
            sUpdateCustomColumns.Add("max_level", "max_level=dbo.Highest(npc.max_level,tmp.max_level)");

            sUpdateCustomColumns.Add("subtype", "subtype=COALESCE(tmp.subtype,tmp.subtype,npc.subtype)");
            sUpdateCustomColumns.Add("species", "species=COALESCE(tmp.species,tmp.species,npc.species,tmp.species)");

            sUpdateCustomColumns.Add("map_pin_type", "map_pin_type=npc.map_pin_type|tmp.map_pin_type");
            sUpdateCustomColumns.Add("title", "title=COALESCE(tmp.title,tmp.title,npc.title)");
            
            sUpdateQuery = "update npc set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sNpcUpdateSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM npc,#npc tmp" +
                " WHERE npc.id=tmp.id;";

        }

        public static Int32 GetNpcId(String pNpcName, String pLocale)
        {
            return DB.GetIdFromName(sNPCNameToId, pNpcName, pLocale);
        }

        public static void AddNpc(String pNpcName, String pLocale, Int32 pNpcId)
        {
            DB.AddNameFromId(sNPCNameToId, pLocale, pNpcName, pNpcId);  
        }


        private static void SetNPCLocationQuery(Int32 pNpcId, Update pUpdate, NPC pNpc, StringBuilder pBuilder)
        {
            // Locations
            foreach (EntityLocation location in pNpc.Locations)
            {

                if (location.X == 0 && location.Y == 0 && location.Zone == 0)
                {
                    continue;
                }

                CustomKey key = new CustomKey(pNpcId,
                                            location.Zone,
                                            location.X,
                                            location.Y);

                if (!DB.IsEntityConfirmed(sNPCLocationConfirmations,
                                key,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }

                pBuilder.AppendFormat(sNPCLocationQueryFormat,
                    pUpdate.UserId,
                    pNpcId,
                    pUpdate.ClientVersion.Value,
                    location.Zone,
                    location.X,
                    location.Y);
            }

        }

        private static void SetNPCStatsQuery(Int32 pNpcId, NPC pNpc, StringBuilder pBuilder)
        {

            Int32 timesKilled = pNpc.KillCount >= pNpc.LootCount ? pNpc.KillCount : pNpc.LootCount;
            pBuilder.AppendFormat(sNpcStatsQueryFormat, timesKilled, pNpc.LootCount, pNpc.ButcherCount, pNpc.ScavengeCount, pNpcId);
        }

        private static void SetNPCLootQuery(Byte pLootType, Int32 pNpcId, Update pUpdate, PackableList<EntityLoot> pNpcLoot, StringBuilder pBuilder)
        {         
            foreach (EntityLoot loot in pNpcLoot)
            {

                CustomKey key = new CustomKey(pNpcId,
                                            loot.Id);

                // Forget about confirmations for now. Let's get some data!
                /*
                if (!DB.IsEntityConfirmed(sNPCLootConfirmations,
                                key,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }
                */
                
                pBuilder.AppendFormat(sNPCLootQueryFormat,
                    pLootType,
                    pNpcId,                    
                    pUpdate.ClientVersion.Value,                    
                    loot.Id,
                    loot.Quantity,                    
                    pUpdate.UserId);
            }

        }
        
        private static void SetNPCCoinQuery(Update pUpdate, NPC pNpc, Int32 pNpcId, StringBuilder pBuilder)
        {

            pBuilder.AppendFormat(sNPCCoinQueryFormat,
                pNpcId,
                pNpc.MoneyCount,                
                pUpdate.ClientVersion.Value,                
                pUpdate.UserId);          

        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.NPCs.Count == 0)
            {
                return;
            }

            StringBuilder sql = new StringBuilder();
            DataRow dr = null;
            DataTable dtInsert = sNpcUpdateSchema.Clone();
            DataTable dtMerchandiseInsert = sNpcMerchandiseSchema.Clone();
            DataTable dtUpdate = sNpcUpdateSchema.Clone();

            dtInsert.BeginLoadData();
            dtUpdate.BeginLoadData();
            dtMerchandiseInsert.BeginLoadData();

            CustomKey key;
            String locale = pUpdate.Language.ToString();         
            SqlCommand cmd = pConn.CreateCommand();                      


            bool isNewRow = false;
            Int32 npcId = 0;
            String deleteMerchandiseIds = "";            
            foreach (NPC npc in pUpdate.NPCs)
            {
                if (npc.Name.Flag == null)
                {
                    continue;
                }

                npcId = GetNpcId(npc.Name.Value, locale);
                if (npcId > 0)
                {
                    if (!DB.ReadyForDB(sHistory,
                                sNPCConfirmations,
                                npcId,
                                Config.Instance.NpcExpiration,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                (ELocale)pUpdate.Language,
                                pUpdate.IsTrustedUser))
                    {
                        continue;
                    }
                    isNewRow = false;                        
                    dr = dtUpdate.NewRow();
                    dr["id"] = npcId;
                }                                        
                else
                {
                    isNewRow = true;
                    dr = dtInsert.NewRow();
                }
                                                       
                dr["version_id"] = pUpdate.ClientVersion.Value;
                dr["name_" + locale] = npc.Name.Value;
                dr["map_pin_type"] = Utility.GetBigBitValue(npc.MapPinType);

                if (isNewRow || npc.Title > 0)
                {
                    dr["title"] = npc.Title;
                }
                else
                {
                    dr["title"] = System.DBNull.Value;
                }
                dr["type"] = npc.Type;
                dr["tier"] = npc.Tier;
                dr["min_level"] = npc.MinLevel;
                dr["max_level"] = npc.MaxLevel;
                if (isNewRow || npc.SubType > 0)
                {
                    dr["subtype"] = npc.SubType;
                }
                if (isNewRow || npc.Species > 0)
                {
                    dr["species"] = npc.Species;
                }                

                // Take the higher of kill count and loot count
                dr["times_killed"] = 0;
                dr["times_looted"] = 0;
                dr["times_butchered"] = 0;
                dr["times_scavenged"] = 0;
                
                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {
                    //dtInsert.Rows.Add(dr);
                    cmd.CommandText = DB.GetFilledInsertSQL(sInsertQuery, sNpcInsertSchema, dr, sIgnoreInsertColumns, locale);
                    npcId = (Int32)cmd.ExecuteScalar();
                    AddNpc(npc.Name.Value, locale, npcId);                                               
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }

                // NPC Locations
                SetNPCLocationQuery(npcId, pUpdate, npc, sql);
              
                if (npc.Merchandise.Count > 0)
                {
                    deleteMerchandiseIds += "," + npcId;

                    foreach (NPCMerchandise merchandise in npc.Merchandise)
                    {
                        DataRow merchandiseRow = dtMerchandiseInsert.NewRow();
                        merchandiseRow["npc_id"] = npcId;
                        merchandiseRow["item_id"] = merchandise.Id;
                        dtMerchandiseInsert.Rows.Add(merchandiseRow);
                    }                   
                }
            }

            foreach (NPC npc in pUpdate.NPCs)
            {
                if (npc.Name.Flag == null)
                {
                    continue;
                }

                npcId = GetNpcId(npc.Name.Value, locale);

                // Move kill counters here:

                if (npcId > 0)
                {

                    SetNPCStatsQuery(npcId, npc, sql);

                    // NPC Regular Loot
                    SetNPCLootQuery((Byte)ELootType.Loot, npcId, pUpdate, npc.Loots, sql);

                    // NPC Scavenged Loot
                    SetNPCLootQuery((Byte)ELootType.Scavenging, npcId, pUpdate, npc.ScavengedLoots, sql);

                    // NPC Butchered Loot
                    SetNPCLootQuery((Byte)ELootType.Butchering, npcId, pUpdate, npc.ButcheredLoots, sql);

                    // Quest Loot
                    SetNPCLootQuery((Byte)ELootType.Quest, npcId, pUpdate, npc.QuestLoots, sql);

                    //Coin
                    if (npc.MoneyCount > 0)
                    {
                        SetNPCCoinQuery(pUpdate, npc, npcId, sql);
                    }
                }

            }

            if (dtMerchandiseInsert.Rows.Count > 0)
            {
                cmd.CommandText = "delete from npc_merchandise where npc_id in(" + deleteMerchandiseIds.Substring(1) + ");";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                     SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                     null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "npc_merchandise";
                    bulk.WriteToServer(dtMerchandiseInsert);
                    bulk.Close();
                }
            }

            
            
            if (dtUpdate.Rows.Count > 0)
            {

                cmd.CommandText = sTempTableCreation;
                cmd.ExecuteNonQuery();

                using (SqlBulkCopy bulkUpdate = new SqlBulkCopy(pConn,
                                                   SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                   null))
                {
                    bulkUpdate.BatchSize = 5000;
                    bulkUpdate.DestinationTableName = "#npc";
                    bulkUpdate.WriteToServer(dtUpdate);
                    bulkUpdate.Close();
                }
                                
                cmd.CommandText = String.Format(sUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #npc;";
                cmd.ExecuteNonQuery();
            }

            if (sql.Length > 0)
            {
                cmd.CommandText = sql.ToString();
                cmd.ExecuteNonQuery();
            }
            


        }
    }
    
}
