﻿using Curse;
using Curse.WAR;
using System;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Text;
namespace WARDataService
{
    public static partial class DBQuests
    {
        private const Byte NUM_CONDITIONS = 6;
        private const Byte NUM_REWARDS = 6;
        private const Byte NUM_ZONES = 5;                

        private static HashSet<CustomKey> sKnownQuests =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static String sUpdateQuery = null;

        private static String[] sUpdateIgnoredColumns = { "id", "version_id", "posted", "posted_id" };
        private static Dictionary<String, String> sUpdateCustomColumns = new Dictionary<string, string>();

        private static String sQuestRewardQueryFormat = "UPDATE quest_reward SET " +
                                            "careers=careers|{0}," +
                                            "updated_id={1}," +
                                            "updated=GETUTCDATE()" +
                                            " WHERE quest_id={2} AND item_id={3} and type={4}" +
                                            " IF @@ROWCOUNT = 0 " +
                                            "INSERT INTO quest_reward(quest_id,item_id,type," +
                                            "careers,updated_id,posted_id) VALUES(" +
                                            "{2},{3},{4},{0},{1},{1});";

        private static DataTable sTableSchema = null;

        public static void Initialize()
        {
            sKnownQuests.Clear();
            sHistory.Clear();
            sUpdateCustomColumns.Clear();            

            // Race to Side Lookups


            sUpdateCustomColumns.Add("reward_xp", "quest.reward_xp=COALESCE(tmp.reward_xp,tmp.reward_xp,quest.reward_xp)");
            sUpdateCustomColumns.Add("reward_coin", "quest.reward_coin=COALESCE(tmp.reward_coin,tmp.reward_coin,quest.reward_coin)");
            
            sUpdateCustomColumns.Add("starting_npc", "quest.starting_npc=COALESCE(tmp.starting_npc,tmp.starting_npc,quest.starting_npc)");
            sUpdateCustomColumns.Add("finishing_npc", "quest.finishing_npc=COALESCE(tmp.finishing_npc,tmp.finishing_npc,quest.finishing_npc)");

            sUpdateCustomColumns.Add("starting_gameobject", "quest.starting_gameobject=COALESCE(tmp.starting_gameobject,tmp.starting_gameobject,quest.starting_gameobject)");
            sUpdateCustomColumns.Add("finishing_gameobject", "quest.finishing_gameobject=COALESCE(tmp.finishing_gameobject,tmp.finishing_gameobject,quest.finishing_gameobject)");

            sUpdateCustomColumns.Add("previous_quest", "quest.previous_quest=COALESCE(tmp.previous_quest,tmp.previous_quest,quest.previous_quest)");
            sUpdateCustomColumns.Add("area", "quest.area=COALESCE(tmp.area,tmp.area,quest.area)");

            sUpdateCustomColumns.Add("journal_description_{0}", "quest.journal_description_{0}=COALESCE(tmp.journal_description_{0},tmp.journal_description_{0},quest.journal_description_{0})");
            sUpdateCustomColumns.Add("starting_description_{0}", "quest.starting_description_{0}=COALESCE(tmp.starting_description_{0},tmp.starting_description_{0},quest.starting_description_{0})");
            
            sUpdateCustomColumns.Add("level", "quest.level=dbo.Lowest(quest.level,tmp.level)");

            sUpdateCustomColumns.Add("given_to_races", "quest.given_to_races=quest.given_to_races|tmp.given_to_races");
            sUpdateCustomColumns.Add("given_to_careers", "quest.given_to_careers=quest.given_to_careers|tmp.given_to_careers");

            for (int i = 0; i < NUM_CONDITIONS; i++)
            {
                sUpdateCustomColumns.Add("condition_name" + i, "quest.condition_name" + i + "_{0}=COALESCE(tmp.condition_name" + i + "_{0},tmp.condition_name" + i + "_{0},quest.condition_name" + i + "_{0})");
                sUpdateCustomColumns.Add("condition_count" + i, "quest.condition_count" + i + "=COALESCE(tmp.condition_count" + i + ",tmp.condition_count" + i + ",quest.condition_count" + i + ")");
                sUpdateCustomColumns.Add("condition_type" + i, "quest.condition_type" + i + "=COALESCE(tmp.condition_type" + i + ",tmp.condition_type" + i + ",quest.condition_type" + i + ")");
                sUpdateCustomColumns.Add("condition_id" + i, "quest.condition_id" + i + "=COALESCE(tmp.condition_id" + i + ",tmp.condition_id" + i + ",quest.condition_id" + i + ")");
            }
            
            foreach (Int32 version in Config.Instance.Versions)
            {
                DB.PopulateId(sKnownQuests, "quest", version);
            }
            DB.LoadSchema(ref sTableSchema, "quest");

            sUpdateQuery = "update quest set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sTableSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM quest,#quest tmp" +
                " WHERE quest.id=tmp.id AND quest.version_id=tmp.version_id;";

        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.Quests.Count == 0)
            {
                return;
            }

            DataRow dr = null;
            DataTable dtInsert = sTableSchema.Clone();
            DataTable dtUpdate = sTableSchema.Clone();

            dtInsert.BeginLoadData();
            dtUpdate.BeginLoadData();

            CustomKey key;
            String locale = pUpdate.Language.ToString();
           
            SqlBulkCopy bulkInsert = new SqlBulkCopy(pConn,
                                                    SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                    null);

            SqlBulkCopy bulkUpdate = new SqlBulkCopy(pConn,
                                                     SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                     null);

            bulkInsert.BatchSize = 5000;
            bulkUpdate.BatchSize = 5000;
            bulkInsert.DestinationTableName = "quest";
            bulkUpdate.DestinationTableName = "#quest";


            bool isNewRow = false;
            foreach (Quest quest in pUpdate.Quests)
            {
                if (!DB.ReadyForDB(sHistory,
                                sConfirmations,
                                quest.Id,
                                Config.Instance.QuestExpiration,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                (ELocale)pUpdate.Language,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }

                Int32 startingNPC = DBNPCs.GetNpcId(quest.StartingNPC.Value, locale);
                Int32 finishingNPC = DBNPCs.GetNpcId(quest.FinishingNPC.Value, locale);

                Int32 startingGameObject = DBGameObjects.GetGameObjectId(quest.StartingGameObject.Value, locale);
                Int32 finishingGameObject = DBGameObjects.GetGameObjectId(quest.FinishingGameObject.Value, locale);

                key = new CustomKey(pUpdate.ClientVersion.Value,
                                    quest.Id);

                isNewRow = !sKnownQuests.Contains(key);
                if (isNewRow)
                {
                    dr = dtInsert.NewRow();
                }
                else
                {
                    dr = dtUpdate.NewRow();
                }

                dr["id"] = quest.Id;
                dr["version_id"] = pUpdate.ClientVersion.Value;
                dr["name_" + locale] = quest.Name.Value;
                if (quest.JournalDescription.Length > 0)
                {
                    dr["journal_description_" + locale] = quest.JournalDescription;
                }
                if (quest.StartingDescription.Length > 0)
                {
                    dr["starting_description_" + locale] = quest.StartingDescription;
                }
                dr["level"] = quest.Level;
                dr["timer"] = quest.Timer;                

                Int32 raceBits = 0;
                Utility.AddBitValue(ref raceBits, pUpdate.Players[0].Race);

                Int32 careerBits = 0;
                Utility.AddBitValue(ref careerBits, pUpdate.Players[0].Career);

                dr["given_to_races"] = raceBits;
                dr["given_to_careers"] = careerBits;

                if (quest.Area > 0)
                {
                    dr["area"] = quest.Area;
                }

                if (quest.RewardXP > 0 || quest.RewardCoin > 0)
                {
                    dr["reward_xp"] = quest.RewardXP;
                    dr["reward_coin"] = quest.RewardCoin;
                }

                if (startingNPC > 0)
                {
                    dr["starting_npc"] = startingNPC;
                    dr["starting_gameobject"] = 0;
                }
                else if (startingGameObject > 0)
                {
                    dr["starting_npc"] = 0;
                    dr["starting_gameobject"] = startingGameObject;
                }

                if (finishingNPC > 0)
                {
                    dr["finishing_npc"] = finishingNPC;
                    dr["finishing_gameobject"] = 0;
                }
                else if (finishingGameObject > 0)
                {
                    dr["finishing_npc"] = 0;
                    dr["finishing_gameobject"] = finishingGameObject;
                }

                if (quest.PreviousQuest > 0)
                {
                    dr["previous_quest"] = quest.PreviousQuest;
                }

                // Add 1 to type, since it is 0 based.
                for(Byte i = 0;i<quest.Types.Count;i++)
                {
                    ++quest.Types[i];
                }
                Utility.SetBitField(quest.Types, dr, "types");
                

                //Given Rewards

                /*
                 * 
                for (Int32 index = 0; index < NUM_REWARDS; ++index)
                {
                    if (index < quest.RewardGivenItems.Count)
                    {
                        dr["given_item" + index] = quest.RewardGivenItems[index];
                    }
                    else
                    {
                        dr["given_item" + index] = 0;                            
                    }
                }

                //Choice Rewards

                
                for (Int32 index = 0; index < NUM_REWARDS; ++index)
                {
                    if (index < quest.RewardChoiceItems.Count)
                    {
                        dr["choice_item" + index] = quest.RewardChoiceItems[index];
                    }
                    else
                    {
                        dr["choice_item" + index] = 0;
                    }
                }
                 *  */

                //Zones
                for (Int32 index = 0; index < NUM_ZONES; ++index)
                {
                    if (index < quest.Zones.Count)
                    {
                        dr["zone" + index] = quest.Zones[index];
                    }
                    else
                    {
                        dr["zone" + index] = 0;
                    }
                }
               

                //bool setConditions = Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i)

                bool setConditions = !quest.Conditions.Exists(delegate(QuestCondition x) { return (x.Name.Length == 0); });


                quest.Conditions.Sort(delegate (QuestCondition x, QuestCondition y)
                {
                    return x.Name.CompareTo(y.Name);
                });

                //Conditions
                if (isNewRow || setConditions)
                {
                    for (Int32 index = 0; index < NUM_CONDITIONS; ++index)
                    {
                        if (setConditions && index < quest.Conditions.Count)
                        {
                            dr["condition_name" + index + "_" + locale] = quest.Conditions[index].Name;
                            switch (quest.Conditions[index].ActionType)
                            {
                                case (Byte)EQuestConditionType.CreatureKill:
                                case (Byte)EQuestConditionType.CreatureInteraction:
                                    quest.Conditions[index].ActionId = DBNPCs.GetNpcId(quest.Conditions[index].ActionName.Value, locale);
                                    break;
                                case (Byte)EQuestConditionType.CollectGameObject:
                                    quest.Conditions[index].ActionId = DBGameObjects.GetGameObjectId(quest.Conditions[index].ActionName.Value, locale);
                                    break;                                
                            }
                            if (quest.Conditions[index].ActionId <= 0)
                            {
                                dr["condition_id" + index] = System.DBNull.Value;
                                dr["condition_type"+index] = System.DBNull.Value;
                                
                            }
                            else
                            {
                                dr["condition_id" + index] = quest.Conditions[index].ActionId;
                                dr["condition_type" + index] = quest.Conditions[index].ActionType;
                            }
                            Utility.SetNullIfZero(dr, "condition_count" + index, quest.Conditions[index].Count);                            
                        }
                        else
                        {
                            dr["condition_name" + index + "_" + locale] = System.DBNull.Value;
                            dr["condition_count" + index] = System.DBNull.Value;
                            dr["condition_type" + index] = System.DBNull.Value;
                            dr["condition_id" + index] = System.DBNull.Value;
                        }
                    }
                }

                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {
                    dtInsert.Rows.Add(dr);
                    sKnownQuests.Add(key);
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }
                
            }

            SqlCommand cmd = pConn.CreateCommand();
            StringBuilder sql = new StringBuilder(1000);
            foreach (Quest quest in pUpdate.Quests)
            {
                foreach(Int32 itemId in quest.RewardChoiceItems)
                {
                    sql.AppendFormat(sQuestRewardQueryFormat,
                            Utility.GetBitValue(pUpdate.Players[0].Career),
                            pUpdate.UserId,
                            quest.Id,
                            itemId,
                            (Byte)EQuestRewardType.Choice);                    
                }
                foreach (Int32 itemId in quest.RewardGivenItems )
                {
                    sql.AppendFormat(sQuestRewardQueryFormat,
                           Utility.GetBitValue(pUpdate.Players[0].Career),
                           pUpdate.UserId,
                           quest.Id,
                           itemId,
                           (Byte)EQuestRewardType.Given );
                }
            }
            if (sql.Length > 0)
            {
                cmd.CommandText = sql.ToString();
                cmd.ExecuteNonQuery();
            }

            
            if (dtInsert.Rows.Count > 0)
            {
#if DEBUG
                CPUMonitor monitorBulk = new CPUMonitor("DBItems.Save Bulk Insert");
#endif
                bulkInsert.WriteToServer(dtInsert);
                bulkInsert.Close();
#if DEBUG
                Logger.Log(ELogLevel.Debug,
                           pUpdate.UserHost,
                           monitorBulk.ToString());
#endif
            }

            if (dtUpdate.Rows.Count > 0)
            {                
                cmd.CommandText = "SELECT TOP 0 * INTO #quest FROM quest;";
                cmd.ExecuteNonQuery();
                //cmd.CommandText = "CREATE CLUSTERED INDEX PK_quest ON #quest(id,version_id);";
                //cmd.ExecuteNonQuery();
                bulkUpdate.WriteToServer(dtUpdate);
                bulkUpdate.Close();
                cmd.CommandText = String.Format(sUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #quest;";
                cmd.ExecuteNonQuery();
            }
        }
    }

}
