using Curse;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using System.Xml.Serialization;

namespace WoWDataCenter
{
    /**
     * Handles loading a complex configuration format into exact in-memory format for quick access
     *
     * @author Shane Bryldt
     */
    public sealed class Config
    {
        /**
         * Abstract class for all DBCSchemas to inherit for generic data they all require
         *
         * @author Shane Bryldt
         */
        public abstract class DBCSchema
        {
            public ELocale[] Locales;
            public UInt16 Version;
            public Int32 Id;
        }
        /**
         * The following classes inherit DBCSchema and potentially provide additional data required from a specific DBC
         *
         * @author Shane Bryldt
         */
        public sealed class AreaTableDBCSchema
            : DBCSchema
        {
            public Int32 Name;
        }
        public sealed class ChrClassesDBCSchema
            : DBCSchema
        {
        }
        public sealed class ChrRacesDBCSchema
            : DBCSchema
        {
        }
        public sealed class FactionDBCSchema
            : DBCSchema
        {
            public Int32 Name;
        }
        public sealed class ItemDBCSchema
            : DBCSchema
        {
            public Int32 SlotId;
        }
        public sealed class SkillLineDBCSchema
            : DBCSchema
        {
            public Int32 Name;
        }
        public sealed class SkillLineAbilityDBCSchema
            : DBCSchema
        {
            public Int32 SkillLineId;
            public Int32 SpellId;
        }
        public sealed class SkillRaceClassInfoDBCSchema
            : DBCSchema
        {
            public Int32 SkillLineId;
            public Int32 RequiredRace;
            public Int32 RequiredClass;
        }
        public sealed class SpellDBCSchema
            : DBCSchema
        {
            public Int32 LineId;
            public Int32 TaughtId;
            public Int32 Name;
            public Int32 Rank;
        }

        /**
         * Public static member variables
         */
        public static Int32 LocaleCount = Enum.GetNames(typeof(ELocale)).Length;
        public static Config Instance = null;

        /**
         * Handles loading the config file as defined in the web.config
         * 
         * @return  true if the configuration file was loaded, false otherwise
         */
        public static Boolean Load()
        {
            String path = ConfigurationManager.AppSettings["ConfigFile"];
            if (!File.Exists(path))
            {
                return false;
            }
            using (FileStream stream = new FileStream(path, FileMode.Open, FileAccess.Read))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(Config));
                Instance = (Config)serializer.Deserialize(stream);
            }
            return true;
        }

        /**
         * Public member variables for XML deserialization
         */
        public ELogLevel LogLevel;
        public String LogPath;
        public String DBCPath;
        public String DumpPath;
        public String DLLFile;
        public String AdminKey;
        public String ClientKey;
        public String AuthenticationService;
        public Int32 AuthenticationId;
        public String AuthenticationKey;
        public Int32 SessionExpiration;
        public Int32[] Versions;
        public AreaTableDBCSchema[] AreaTableDBCSchemas;
        public ChrClassesDBCSchema[] ChrClassesDBCSchemas;
        public ChrRacesDBCSchema[] ChrRacesDBCSchemas;
        public FactionDBCSchema[] FactionDBCSchemas;
        public ItemDBCSchema[] ItemDBCSchemas;
        public SkillLineDBCSchema[] SkillLineDBCSchemas;
        public SkillLineAbilityDBCSchema[] SkillLineAbilityDBCSchemas;
        public SkillRaceClassInfoDBCSchema[] SkillRaceClassInfoDBCSchemas;
        public SpellDBCSchema[] SpellDBCSchemas;
        public String WoWDB;
        public Int32 MaxQueuedUpdates;
        public Int32[] Trusted;
        public Int32 RateLimit;
        public String[] Whitelist;
        public Int32 MaxCreatureId;
        public Int32 SkipConfirmationForVersion;
        public Int32 MinConfirmations;
        public Int64 MinProfilerVersion;
        public Int32 CreatureExpiration;
        public Int32 GameObjectExpiration;
        public Int32 ItemExpiration;
        public Int32 PageTextExpiration;
        public Int32 QuestExpiration;
        public Int32 MerchandiseExpiration;
        public Int32 ServerUpdateDelay;
        public Int32 GameServerUpdateInterval;
    }
}
