using System;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using System.Threading.Tasks;
using System.Diagnostics;
using Curse.SocketInterface;
using System.Linq;
using Curse.SocketMessages;

//Implements a sample socket client to connect to the server implmented in the AsyncSocketServer project
//Usage: AsyncSocketClient.exe <destination IP address> <destination port number>

//Destination IP address: The IP Address of the server to connect to
//Destination Port Number: The port number to connect to

namespace AsyncSocketClient
{
    class Program
    { 

        static void Main(string[] args)
        {
            IPAddress destinationAddr = null;          // IP Address of server to connect to
            int destinationPort = 0;                   // Port number of server            

            if (args.Length != 2)
            {
                Console.WriteLine("Usage: AsyncSocketClient.exe <destination IP address> <destination port number>");
            }
            try
            {
                destinationAddr = IPAddress.Parse(args[0]);
                destinationPort = int.Parse(args[1]);
                if (destinationPort <= 0)
                {
                    throw new ArgumentException("Destination port number provided cannot be less than or equal to 0");
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message);
                Console.WriteLine("Usage: AsyncSocketClient.exe <destination IP address> <destination port number>");
            }

            AddContractDispatcher<TestContract>(OnTestContract);            
            int clientCount = 1;

            List<ClientSocketInterface> clients = new List<ClientSocketInterface>();
            for (int i = 0; i < clientCount; i++)
            {
                var asyncSocket = new ClientSocketInterface(destinationAddr, destinationPort);
                asyncSocket.Connected += AsyncSocketOnConnected;
                asyncSocket.Disconnected += AsyncSocketOnDisconnected;
                asyncSocket.MessageReceived += AsyncSocketOnMessageReceived;
                clients.Add(asyncSocket);
            }

            var parallelOptions = new ParallelOptions() { MaxDegreeOfParallelism = 20 };

            Console.Write("Establishing " + clientCount.ToString("###,##0") + " connections...");


            Stopwatch sw = Stopwatch.StartNew();

            Parallel.ForEach(clients, parallelOptions, client =>
                {
                    try
                    {
                        if (!client.Connect())
                        {
                            Console.WriteLine("Failed to connect!");                            
                        }                       
                    }
                    catch (Exception ex)
                    {

                        Console.WriteLine("Serious Fail: " + ex.Message);
                    }     
                    Thread.Sleep(1);
                });

            Console.WriteLine("Done in " + sw.ElapsedMilliseconds.ToString("###,##0.00") + "ms");
            Console.ReadLine();

            while (true)
            {                
                Thread.Sleep(200);
                var connectedClients = clients.Where(p => p.IsConnected).ToArray();
                Console.Write("Sending test message to " + connectedClients.Length.ToString() + " clients...");
                Stopwatch sendStopswatch = Stopwatch.StartNew();
                foreach (var client in connectedClients)
                {
                    client.SendTestMessage();    
                }
                sendStopswatch.Stop();
                Console.WriteLine("Done in " + sendStopswatch.ElapsedMilliseconds.ToString("###,##0.00"));
            }

           
            Console.ReadLine();
        }
        

        private static void AsyncSocketOnDisconnected(object sender, EventArgs eventArgs)
        {
            ClientSocketInterface clientSocket = (ClientSocketInterface)sender;
            clientSocket.Dispose();
        }

        private static void AsyncSocketOnMessageReceived(object sender, MessageEventArgs messageEventArgs)
        {
            ClientSocketInterface clientSocket = (ClientSocketInterface)sender;

            IContractDispatcher dispatcher = null;
            if (_contractDispatchers.TryGetValue(messageEventArgs.Message.Header.MessageType, out dispatcher))
            {
                dispatcher.Dispatch(clientSocket, messageEventArgs.Message);                
            }                     
            
        }

        private static Dictionary<int, IContractDispatcher> _contractDispatchers = new Dictionary<int, IContractDispatcher>();

        private static void AddContractDispatcher<T>(Action<BaseSocketInterface, T> handler) where T : Contract<T>
        {
            int messageType = Contract<T>.MessageType;
            IContractDispatcher dispatcher = new ContractDispatcher<T>(handler);
            _contractDispatchers.Add(messageType, dispatcher);
            
        }

        private static void OnTestContract(BaseSocketInterface socketInterface, TestContract testContract)
        {
            //Console.WriteLine(testContract.Message);
        }

        private static void AsyncSocketOnConnected(object sender, EventArgs eventArgs)
        {
            ClientSocketInterface clientSocket = (ClientSocketInterface)sender;
            ConnectContract connectMessage = new ConnectContract()
                {
                    SessionID = "Test"
                };
            clientSocket.SendMessage(connectMessage.ToMessage());

        }
    }
}
