﻿using Curse.SocketInterface;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace AsyncSocketClient
{
    public class SocketClient : IDisposable
    {
        private SocketAsyncEventArgs _sendEventArgs;
        private SocketAsyncEventArgs _receiveEventArgs;
        private Socket _socket;
        private BufferManager _bufferManager;

        private IPAddress _remoteAddress;
        private int _remotePort;

        Message _outgoingMessage = null;
        Message _incomingMessage = null;

        public SocketClient(IPAddress remoteAddress, int remotePort)
        {
            // Create a socket and connect to the server
            _socket = new Socket(remoteAddress.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
            _remoteAddress = remoteAddress;
            _remotePort = remotePort;
            _bufferManager = new BufferManager(SocketConstants.BufferSize * SocketConstants.OperationsToPreallocate * SocketConstants.NumberOfEventArgsPerClient,
               SocketConstants.BufferSize * SocketConstants.OperationsToPreallocate);
            
        }

        public void Dispose()
        {
            if (_sendEventArgs != null)
            {
                _sendEventArgs.Dispose();
            }

            if (_receiveEventArgs != null)
            {
                _receiveEventArgs.Dispose();
            }
        }

        public bool Connect()
        {
            SocketAsyncEventArgs connectEventArgs = new SocketAsyncEventArgs();
            connectEventArgs.Completed += connectEventArgs_Completed;
            connectEventArgs.RemoteEndPoint = new IPEndPoint(_remoteAddress, _remotePort);
            return _socket.ConnectAsync(connectEventArgs);
        }

        void connectEventArgs_Completed(object sender, SocketAsyncEventArgs e)
        {
            if (e.SocketError != SocketError.Success)
            {
                return;
            }

            // Setup a send socket async event
            _sendEventArgs = new SocketAsyncEventArgs();
            _sendEventArgs.Completed += SocketEventArg_SendCompleted;
            _sendEventArgs.RemoteEndPoint = new IPEndPoint(_remoteAddress, _remotePort);
            _sendEventArgs.UserToken = _socket;
            _bufferManager.SetBuffer(_sendEventArgs);


            // Setup a receive socket async event
            _receiveEventArgs = new SocketAsyncEventArgs();
            _receiveEventArgs.Completed += SocketEventArg_ReceiveCompleted;
            _receiveEventArgs.RemoteEndPoint = new IPEndPoint(_remoteAddress, _remotePort);
            _receiveEventArgs.UserToken = _socket;
            _bufferManager.SetBuffer(_receiveEventArgs);

            StartReceive();
            SendTestMessage();
            
        }

        
        void SocketEventArg_SendCompleted(object sender, SocketAsyncEventArgs e)
        {
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.Send:
                    ProcessSend(e);                    
                    break;
                default:
                    throw new Exception("Invalid operation completed");
            }
        }

        void SocketEventArg_ReceiveCompleted(object sender, SocketAsyncEventArgs e)
        {
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.Receive:
                    ProcessReceive(e);
                    break;
                default:
                    throw new Exception("Invalid operation completed");
            }
        }

        /// <summary>
        /// Called when a ConnectAsync operation completes
        /// </summary>
        private void ProcessConnect(SocketAsyncEventArgs e)
        {
            if (e.SocketError == SocketError.Success)
            {
               
                //Console.WriteLine("Successfully connected to the server");
            }
            else
            {
                throw new SocketException((int)e.SocketError);
            }
        }

        /// <summary>
        /// Called when a ReceiveAsync operation completes
        /// </summary>
        private void ProcessReceive(SocketAsyncEventArgs e)
        {
            if (e.SocketError != SocketError.Success || e.BytesTransferred == 0)
            {
                Disconnect();
                return;
            }

            Console.WriteLine("Received server: {0} bytes", e.BytesTransferred);


            _incomingMessage = _incomingMessage ?? new Message();
            

            StartReceive();
            
        }

        

        public void SendTestMessage()
        {
            byte[] body = System.Text.Encoding.UTF8.GetBytes("Hello World! This is a test of a longer than average message. Will it work?");
            MessageHeader header = new MessageHeader(body.Length, 1);
            _outgoingMessage = new Message(header, body);
            StartSend();
        }

        private void StartSend()
        {
            _outgoingMessage.WriteOutgoingData(_sendEventArgs);
            bool hasCompleted = !_socket.SendAsync(_sendEventArgs);
            if (hasCompleted)
            {
                ProcessSend(_sendEventArgs);
            }
        }

        private void StartReceive()
        {
            _socket.ReceiveAsync(_receiveEventArgs);
        }

        /// <summary>
        /// Called when a SendAsync operation completes
        /// </summary>
        private void ProcessSend(SocketAsyncEventArgs e)
        {
            if (e.SocketError == SocketError.Success)
            {
                Console.WriteLine("Finished send");

                if (_outgoingMessage != null)
                {
                    if (_outgoingMessage.IsSendComplete)
                    {
                        Console.WriteLine("Sending a new message!");
                        //Thread.Sleep(10000);
                        //SendTestMessage();
                    }
                    else
                    {
                        Console.WriteLine("Continuing send...");
                        StartSend();
                    }
                }
            }
            else
            {
                throw new SocketException((int)e.SocketError);
            }
        }

        private void Disconnect()
        {
            _socket.Close(1);
            Dispose();
        }

       
    }
}
