using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Sockets;
using System.Net;
using System.Threading;
using Curse.Logging;
using Curse.SocketInterface;
using System.Collections.Concurrent;
using Curse.SocketMessages;
using System.Threading.Tasks;


namespace Curse.SocketServer
{
    /// <summary>
    /// Implements the connection logic for the socket server.  After accepting a connection, all data read
    /// from the client is sent back to the client.  The read and echo back to the client pattern is continued 
    /// until the client disconnects.
    /// </summary>
    public class SocketServer
    {
        
        protected readonly int _maxConnections;   // the maximum number of connections the sample is designed to handle simultaneously 
        private const int AcceptCount = 100;   // the maximum number of connections the sample is designed to handle simultaneously 
                
        // pool of reusable SocketAsyncEventArgs objects for write, read and accept socket operations
        private readonly SocketClientPool _readWritePool;
        private readonly SocketUdpPool _udpSendPool;
        private readonly bool _udpEnabled;
               
        protected int _numConnectedSockets;      // the total number of clients connected to the server 
        protected Int64 _historicalConnectedSockets;      // the total number of clients connected to the server 

        private int _clientCounter = 0;
        protected readonly  ConcurrentDictionary<int, ServerSocketInterface> _clients = new ConcurrentDictionary<int, ServerSocketInterface>();
        
        protected readonly BufferManager _bufferManager;
        public SocketServerState State { get; protected set; }

        public int NumConnectedSockets
        {
            get
            {
                return _numConnectedSockets;
            }
        }

        public int NextClientID()
        {
            return Interlocked.Increment(ref _clientCounter);
        }

        public SocketServer(int maxConnections, int numberOfPorts, bool udpEnabled = true)
        {
            _maxConnections = maxConnections;            
            _numConnectedSockets = 0;
            _udpEnabled = udpEnabled;

            _bufferManager = new BufferManager(
                SocketConstants.BufferSize 
                * maxConnections 
                * SocketConstants.OperationsToPreallocate
                * numberOfPorts
                * 2, 
                SocketConstants.BufferSize);

            State = SocketServerState.Offline;
            ContractDispatcher.AddContractDispatcher<Handshake>(OnHandshake);

            var random = new Random(DateTime.UtcNow.Millisecond);
            _clientCounter = random.Next(1000, 100000);

            // Initialize our client pool (for reading and writing to connected clients)
            _readWritePool = new SocketClientPool(_bufferManager, _maxConnections, TCP_Completed);

            // Initialize UDP Send Pool
            if (_udpEnabled)
            {
                _udpSendPool = new SocketUdpPool(_bufferManager, _maxConnections, UDP_Completed);
            }
        }

        protected virtual void OnHandshake(ISocketInterface baseSocketInterface, Handshake handshake)
        {            
            var serverSocketInterface = baseSocketInterface as ServerSocketInterface;
            if (serverSocketInterface != null)
            {
                serverSocketInterface.IsHandshaken = true;
                serverSocketInterface.DateLastHandshake = Environment.TickCount;

                if (_udpEnabled)
                {
                    serverSocketInterface.SendPacket(handshake);
                }
            }
        }

        /// <summary>
        /// Starts the server such that it is listening for incoming connection requests.    
        /// </summary>
        /// <param name="localEndPoints">The endpoints which the server will listening for conenction requests on</param>
        public virtual void Start(IPEndPoint[] localEndPoints)
        {
            State = SocketServerState.Starting;
            
            // Initialize the Tcp Connection Manager
            Logger.Info("Initializing TCP Connection Manager");
            TcpConnectionManager.Initialize(localEndPoints, AcceptCount, ProcessTcpAccept);

            // Initialize the Tcp Connection Manager
            if (_udpEnabled)
            {
                Logger.Info("Initializing UDP Connection Manager");
                UdpConnectionManager.Initialize(_bufferManager, localEndPoints, _maxConnections, ProcesUdpMessage);
            }
            
            State = SocketServerState.Started;

            new Thread(IdleSocketCleanup) { IsBackground = true, Name = "IdleSocketCleanup" }.Start();

        }

        public virtual void Stop()
        {
            State = SocketServerState.Stopped;

            Logger.Info("Shutting down TCP Connection Manager");
            TcpConnectionManager.ShutdownAll();

            if (_udpEnabled)
            {
                Logger.Info("Shutting down UDP Connection Manager");
                UdpConnectionManager.ShutdownAll();
            }
        }
        
        void UDP_Completed(object sender, SocketAsyncEventArgs e)
        {
            // determine which type of operation just completed and call the associated handler
            switch (e.LastOperation)
            {               
                case SocketAsyncOperation.SendTo:
                    var channel = (BaseChannelProcessor)e.UserToken;

                    if (channel == null)
                    {
                        return;
                    }

                    try
                    {
                        channel.ProcessSend(e);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unhandled error during Send operating in UDP_Completed.");
                    }
                    break;

                default:
                    Logger.Warn("Unsupported operation in UDP_Completed", new { e.LastOperation });
                    break;
            }
        }

        protected ServerSocketInterface LookupClient(int clientID)
        {
            // Get the mapped client
            ServerSocketInterface client;
            _clients.TryGetValue(clientID, out client);
            return client;
        }
        
        private void ProcesUdpMessage(byte[] incomingData, IPEndPoint remoteEndPoint)
        {
            // Create a new incoming message
            var message = Message.FromIncoming();
            var startingIndex = message.AddHeaderData(incomingData, 0);

            if (!message.Header.IsComplete || message.Header.TotalSize > SocketConstants.BufferSize)
            {
                return;
            }

            // Get the mapped client
            ServerSocketInterface client = null;

            if (!_clients.TryGetValue(message.Header.ClientID, out client))
            {
                Logger.Debug("Incoming packet does not map to a connected client ID", new { message.Header.ClientID });
                return;
            }

            startingIndex += message.AddIncomingData(incomingData, startingIndex);
            client.UpdateRemoteEndpoint(remoteEndPoint);
            client.RaiseMessageReceived(message);     
        }

        private DateTime _lastLoggedConnectionNumbers = DateTime.MinValue;
        
        private void ProcessTcpAccept(int port, Socket socket)
        {             
            // If the server is not fully started, return out
            if (State != SocketServerState.Started)
            {
                Logger.Warn("Connection refused, server is not started!");
                return;
            }

            // If it was not successul, return it to the pool
            if (_numConnectedSockets > _maxConnections)
            {
                Logger.Warn("Connection refused, server is full!", new { CurrentConnection = _numConnectedSockets, MaxConnection = _maxConnections });
                return;
            }   

            // Increment our connection count
            var newCount = Interlocked.Increment(ref _numConnectedSockets);                        

            // Keep track of our historical connection count, for logging purposes
            var newHistoricalCount = Interlocked.Increment(ref _historicalConnectedSockets);

            SocketStats.UpdateConnectionCounts(newCount, newHistoricalCount);

            // Try to create a new server socket interface
            ServerSocketInterface socketInterface = null;

            try
            {
                UdpConnectionManager udpConnectionManager = null;
                if (_udpEnabled)
                {
                    udpConnectionManager = UdpConnectionManager.GetByPort(port);
                    if (udpConnectionManager == null)
                    {
                        Logger.Warn("Unable to find UdpConnectionManager for port.", new { Port = port });
                        return;
                    }
                }

                // Set the user token socket
                socketInterface = new ServerSocketInterface(SocketInterfaceOnDisconnected, Interlocked.Increment(ref _clientCounter), socket, _readWritePool, udpConnectionManager != null ? udpConnectionManager.Socket : null, _udpSendPool);
                socketInterface.MessageReceived += SocketInterfaceOnMessageReceived;
                    
                if (!_clients.TryAdd(socketInterface.ClientID, socketInterface))
                {
                    Logger.Warn("Failed to add new socket interface to client dictionary. Client will be disconnected!");
                    socketInterface.RaiseDisconnect(SocketDisconnectReason.ProcessingError);
                }

                socketInterface.Initialize();
                OnConnected(socketInterface);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process incoming TCP connection! The client will be disconnected.");
                if (socketInterface != null)
                {
                    socketInterface.RaiseDisconnect(SocketDisconnectReason.ProcessingError);
                }
            }
        }

        protected virtual void OnConnected(ServerSocketInterface socket)
        {
            
        }
       
        private void SocketInterfaceOnMessageReceived(object sender, MessageEventArgs messageEventArgs)
        {
            var socketInterface = (ServerSocketInterface)sender;
            ContractDispatcher.TryDispatch(socketInterface, messageEventArgs.Message);            
        }

        protected virtual void SocketInterfaceOnDisconnected(object sender, EventArgs eventArgs)
        {
            // Decrement the counter keeping track of the total number of clients connected to the server
            Interlocked.Decrement(ref _numConnectedSockets);

            var clientID = ((ServerSocketInterface)sender).ClientID;
            
            ServerSocketInterface removed;
            if (!_clients.TryRemove(clientID, out removed))
            {
                Logger.Trace("[Socket Disconnected] Client disconnected without an entry in the client dictionary! Client ID: " + clientID);
                return;
            }

            if (removed.Session != null)
            {
                Logger.Trace("[Socket Disconnected] Disconnecting connection for session '" + removed.Session.SessionID + "' and client id '" + removed.ClientID + "'");
                removed.Session.Disconnect();
            }
            else
            {
                Logger.Debug("[Socket Disconnected] Disconnecting connection for unknown session and client id '" + removed.ClientID + "'");
            }
        }

        private void SendTestData()
        {
            byte[] testData = new byte[255];
            for (int i = 0; i < testData.Length; i++)
            {
                testData[i] = (byte)i;
            }
            while (true)
            {
                Thread.Sleep(500);

                Parallel.ForEach(_clients.Values, client =>
                    {
                        if (client.Session == null)
                        {
                            return;
                        }

                        TestContract testTcpContract = new TestContract()
                        {
                            Message = testData
                        };

                        client.SendContract(testTcpContract);



                        if (client.IsHandshaken)
                        {
                            TestContract testUdpContract = new TestContract()
                                {
                                    Message = testData
                                };
                            client.SendPacket(testUdpContract);
                        }
                    });
            }
        }

        /// <summary>
        /// This method is called whenever a receive or send opreation is completed on a socket 
        /// </summary> 
        /// <param name="e">SocketAsyncEventArg associated with the completed receive operation</param>
        void TCP_Completed(object sender, SocketAsyncEventArgs e)
        {
            var channel = (BaseChannelProcessor)e.UserToken;

            if (channel == null)
            {
                return;
            }

            // determine which type of operation just completed and call the associated handler
            switch (e.LastOperation)
            {

                case SocketAsyncOperation.Receive:
                    try
                    {
                        channel.ProcessReceive(e);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unhandled error during Receive operating in TCP_Completed.");
                    }
                    break;

                case SocketAsyncOperation.Send:
                    try
                    {
                        channel.ProcessSend(e);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unhandled error during Send operating in TCP_Completed.");
                    }
                    break;

                default:
                    Logger.Warn("Unsupported operation in TCP_Completed", new { e.LastOperation });
                    break;

            }

        }       

        private void IdleSocketCleanup()
        {

            while (State == SocketServerState.Started)
            {
                try
                {
                    Thread.Sleep(1000);

                    if (State != SocketServerState.Started)
                    {
                        return;
                    }

                    DisconnectIdleSockets();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unhandled exception in IdleSocketCleanup");
                }

            }

            Logger.Info("Idle socket cleanup thread has completed.", new { State });
        }        

        public static class SocketStats
        {
            private static readonly TimeSpan IdleSocketLoggingInterval = TimeSpan.FromMinutes(1);

            private static Int64 _idleSocketCount = 0;
            private static Int64 _neverConnected = 0;
            private static Int64 _neverAuthenticated = 0;
            private static Int64 _noSession = 0;
            private static Int64 _unhealthy = 0;
            
            private static Int64 _lastLoggedIdleSocketCount = 0;
            private static DateTime _lastLoggedIdleSocketTimestamp = DateTime.MinValue;
            

            public static void IncrementIdleSocketCount(int amount, int neverConnected, int neverAuthenticated, int noSession, int unhealthy)
            {
                _idleSocketCount += amount;
                _neverConnected += neverConnected;
                _neverAuthenticated += neverAuthenticated;
                _noSession += noSession;
                _unhealthy += unhealthy;

                if (_idleSocketCount - _lastLoggedIdleSocketCount < 100 || DateTime.UtcNow.Subtract(_lastLoggedIdleSocketTimestamp) < IdleSocketLoggingInterval)
                {
                    return;
                }

                _lastLoggedIdleSocketTimestamp = DateTime.UtcNow;
                _lastLoggedIdleSocketCount = _idleSocketCount;

                Logger.Trace("[Disconnect Stats] Total Disconnects: " + _idleSocketCount.ToString("###,##0")
                    + ". Never Connected: " + _neverConnected.ToString("###,##0")
                    + ". Never Authenticated: " + _neverAuthenticated.ToString("###,##0")
                    + ". No Session: " + _noSession.ToString("###,##0")
                    + ". Unhealthy: " + _unhealthy.ToString("###,##0")
                    );
            }

            private static int _lastLoggedActiveConnectionCount = 0;
            private static int _activeConnectionCount = 0;
            private static Int64 _lastLoggedHistoricalConnectionCount = 0;
            private static Int64 _historicalConnectionCount = 0;
#if DEBUG
            private const int ConnectionLoggingInterval = 1;
#else
            private const int ConnectionLoggingInterval = 100;
#endif

            private static DateTime _lastLoggedSocketCountTimestamp = DateTime.MinValue;

            public static void UpdateConnectionCounts(int activeConnectionCount, Int64 historicalConnectionCount)
            {
                _activeConnectionCount = activeConnectionCount;
                _historicalConnectionCount = historicalConnectionCount;

                var shouldLog = (Math.Abs(_activeConnectionCount - _lastLoggedActiveConnectionCount) >= ConnectionLoggingInterval) ||
                                (Math.Abs(_historicalConnectionCount - _lastLoggedHistoricalConnectionCount) >= ConnectionLoggingInterval);

                if (!shouldLog || DateTime.UtcNow.Subtract(_lastLoggedSocketCountTimestamp) < IdleSocketLoggingInterval)
                {
                    return;
                }

                _lastLoggedSocketCountTimestamp = DateTime.UtcNow;
                _lastLoggedActiveConnectionCount = _activeConnectionCount;
                _lastLoggedHistoricalConnectionCount = historicalConnectionCount;

                Logger.Trace("[Connection Stats] Active connections: " + _activeConnectionCount.ToString("###,##0") + ". Historical Connections: " + _historicalConnectionCount.ToString("###,##0"));                
            }
        }


        private void DisconnectIdleSockets()
        {
            // Get all socket interfaces that need a health check
            var socketsToCheck = _clients.Values.Where(client => client.NeedsHealthCheck()).ToArray();

            if (socketsToCheck.Length == 0)
            {
                return;
            }

#if SOCKET_LOGGING
            Logger.Trace("Health checking " + socketsToCheck.Length.ToString("###,##0") + " sockets...");
#endif
          
            var socketsToDisconnect = new List<ServerSocketInterface>();
            var socketsNeverConnected = 0;
            var socketsNeverAuthenticated = 0;
            var socketsWithNoSession = 0;
            var socketsUnhealthy = 0;

            foreach (var client in socketsToCheck)
            {
                var needsDisconnect = true;
                if (!client.IsConnected)
                {
                    ++socketsNeverConnected;
                }
                else if (client.Session == null)
                {
                    ++socketsNeverAuthenticated;
                }
                else if (!client.IsAuthenticated)
                {
                    ++socketsWithNoSession;
                }
                else if (!client.IsSocketHealthy())
                {
                    ++socketsUnhealthy;
                }
                else
                {
                    needsDisconnect = false;
                }

                if (needsDisconnect)
                {
                    socketsToDisconnect.Add(client);
                }
            }
                                 
            
            if (socketsToDisconnect.Count == 0)
            {
                return;
            }

            SocketStats.IncrementIdleSocketCount(socketsToDisconnect.Count, socketsNeverConnected, socketsNeverAuthenticated, socketsWithNoSession, socketsUnhealthy);

            Logger.Trace("Disconnecting " + socketsToDisconnect.Count.ToString("###,##0") + " idle clients...");            

            foreach (var client in socketsToDisconnect)
            {
                try
                {                    
                    client.RaiseDisconnect(SocketDisconnectReason.IdleTimeout);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to disconnect client!");
                }

                if (_clients.ContainsKey(client.ClientID))
                {
                    ServerSocketInterface removed;
                    if (!_clients.TryRemove(client.ClientID, out removed))
                    {
                        Logger.Error("Failed to remove client from client dictionary!");
                    }
                }
            }
        }

    }
}
