﻿using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using System.Net.Sockets;

namespace Curse.SocketServer
{
    public class TcpConnectionManager
    {
        public delegate void ProcessAcceptDelegate(int port, Socket socket);
        private static readonly Dictionary<int, TcpConnectionManager> ConnectionManagers = new Dictionary<int, TcpConnectionManager>();

        public static void Initialize(IPEndPoint[] endPoints, int acceptCount, ProcessAcceptDelegate processMessageDelegate)
        {
            foreach (var endpoint in endPoints)
            {
                ConnectionManagers.Add(endpoint.Port, new TcpConnectionManager(endpoint, acceptCount, processMessageDelegate));
            }
        }

        public static void ShutdownAll()
        {
            foreach (var connectionmanager in ConnectionManagers.Values)
            {
                connectionmanager.Shutdown();
            }
        }

        private readonly int _port;
        private readonly SocketAcceptPool _socketAcceptPool;
        private readonly Socket _socket;        
        private readonly ProcessAcceptDelegate _acceptDelegate;
        private bool _isAlive = true;


        public TcpConnectionManager(IPEndPoint endPoint, int poolSize, ProcessAcceptDelegate acceptDelegate)
        {
            Logger.Info("Creating TCP Connection Manager", new { EndPoint = endPoint.ToString(), PoolSize = poolSize });

            _port = endPoint.Port;
            _acceptDelegate = acceptDelegate;
            _socketAcceptPool = new SocketAcceptPool(poolSize, AcceptEventArg_Completed);

            // Create a new TCP socket 
            _socket = new Socket(endPoint.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
            
            // Bind the socket to this specific local endpoint, ex: 0.0.0.0:80
            _socket.Bind(endPoint);
            
            // Start Listening
            _socket.Listen(poolSize);

            // Start Accepting
            for (var i = 0; i < poolSize; i++)
            {
                var eventArgs = _socketAcceptPool.Get();
                StartTcpAccept(eventArgs);
            }
            
        }

        public void Shutdown()
        {
            _isAlive = false;
            var socket = _socket;
            if (socket == null)
            {
                return;
            }

            try
            {                                               
                Logger.Info("[TcpConnectionManager] Shutting down");
                socket.Close(1);
                socket.Dispose();                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[TcpConnectionManager] Failed to close socket!");
            }
            
        }

        /// <summary>
        /// Begins an operation to accept a connection request from the client 
        /// </summary>                
        public void StartTcpAccept(SocketAsyncEventArgs acceptEventArg)
        {
            if (!_isAlive)
            {
                return;
            }

            acceptEventArg.AcceptSocket = null;
            acceptEventArg.RemoteEndPoint = null;

            var completedSyncronously = !_socket.AcceptAsync(acceptEventArg);

            if (completedSyncronously)
            {
                ProcessAccept(acceptEventArg);
            }
        }

        /// <summary>
        /// This method is the callback method associated with Socket.AcceptAsync operations and is invoked
        /// when an accept operation is complete
        /// </summary>
        void AcceptEventArg_Completed(object sender, SocketAsyncEventArgs e)
        {
            if (!_isAlive)
            {
                return;
            }

            ProcessAccept(e);
        }

        private void ProcessAccept(SocketAsyncEventArgs e)
        {
            if (!_isAlive)
            {
                return;
            }

            try
            {                
                var status = e.SocketError;

                // If it was not successul, return it to the pool
                if (status != SocketError.Success)
                {
                    return;
                }
                
                // Process the new connection
                _acceptDelegate(_port, e.AcceptSocket);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error during TcpConnectionManager ProcessAccept");
            }
            finally
            {
                // Resume incoming connections
                try
                {
                    StartTcpAccept(e);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to resume accepting!");
                }                
            }
        }
    }
}
