﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Sockets;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketInterface.Exceptions;

namespace Curse.SocketServer
{
    public class UdpConnectionManager
    {
        private static readonly Dictionary<int, UdpConnectionManager> ConnectionManagers = new Dictionary<int, UdpConnectionManager>();

        public static void Initialize(BufferManager bufferManager, IPEndPoint[] endPoints, int maxConnections, ProcessUdpMessageDelegate processMessageDelegate)
        {
            foreach (var endpoint in endPoints)
            {
                ConnectionManagers.Add(endpoint.Port, new UdpConnectionManager(bufferManager, endpoint, maxConnections, processMessageDelegate));
            }
        }

        public static void ShutdownAll()
        {
            foreach (var connectionmanager in ConnectionManagers.Values)
            {
                connectionmanager.Shutdown();
            }
        }

        public static UdpConnectionManager GetByPort(int port)
        {
            UdpConnectionManager connectionManager;
            return ConnectionManagers.TryGetValue(port, out connectionManager) ? connectionManager : null;
        }

        public delegate void ProcessUdpMessageDelegate(byte[] incomingData, IPEndPoint remoteEndPoint);

        private readonly Socket _socket;
        private readonly SocketUdpPool _socketPool;
        private readonly ProcessUdpMessageDelegate _processUdpMessageDelegate;
        private readonly IPEndPoint _endPoint;
        private bool _isAlive = true;

        public Socket Socket
        {
            get { return _socket; }

        }

        private UdpConnectionManager(BufferManager bufferManager, IPEndPoint endPoint, int poolSize, ProcessUdpMessageDelegate processUdpMessageDelegate)
        {
            Logger.Info("Creating UDP Connection Manager ", new { EndPoint = endPoint.ToString(), PoolSize = poolSize });
            _processUdpMessageDelegate = processUdpMessageDelegate;

            _endPoint = endPoint;

            // Initialize UDP Receive Pool
            _socketPool = new SocketUdpPool(bufferManager, poolSize, Receive_Completed);
            
             // Create a UDP socket which listens for incoming UDP requests
            _socket = new Socket(endPoint.AddressFamily, SocketType.Dgram, ProtocolType.Udp);
            _socket.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.PacketInformation, true);
            _socket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
            _socket.Blocking = false;
            _socket.Bind(endPoint);

            for (var i = 0; i < poolSize; i++)
            {
                var eventArgs = _socketPool.Get();
                StartReceive(eventArgs);
            }
            
        }

        public void Shutdown()
        {
            _isAlive = false;
            var socket = _socket;
            if (socket == null)
            {
                return;
            }

            Logger.Info("[UdpConnectionManager] Shutting down");
            try
            {
                socket.Shutdown(SocketShutdown.Both);
                socket.Close(1);
                socket.Dispose();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[UdpConnectionManager] Failed to close socket!");
            }

        }

        private void StartReceive(SocketAsyncEventArgs e)
        {
            if (!_isAlive)
            {
                return;
            }

            // Reset our socket event arg             
            e.RemoteEndPoint = null;
            e.RemoteEndPoint = new IPEndPoint(_endPoint.Address, _endPoint.Port);

            // Call the API to initiate the async ReceiveFrom call
            var completedSyncronously = !_socket.ReceiveFromAsync(e);

            if (completedSyncronously)
            {
                ProcessReceive(e);
            }

        }

        private void Receive_Completed(object sender, SocketAsyncEventArgs e)
        {
            if (!_isAlive)
            {
                return;
            }

            // determine which type of operation just completed and call the associated handler
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.Receive:
                case SocketAsyncOperation.ReceiveFrom:
                case SocketAsyncOperation.ReceiveMessageFrom:
                    ProcessReceive(e);
                    break;               
                default:
                    Logger.Warn("Unsupported operation in UDP_Completed", new { e.LastOperation });
                    break;
            }
        }
        
        private void ProcessReceive(SocketAsyncEventArgs e)
        {
            if (!_isAlive)
            {
                return;
            }

            var hasResumed = false;
            try
            {
                // If it was not successul, return it to the pool
                if (e.SocketError != SocketError.Success || e.BytesTransferred < MessageHeader.HeaderSize ||
                    e.BytesTransferred > SocketConstants.BufferSize)
                {
                    StartReceive(e);
                    hasResumed = true;
                    return;
                }

                // Get the remote endpoint
                var remoteEndPoint = (IPEndPoint) e.RemoteEndPoint;

                // Copy the buffer
                var incomingData = new byte[e.BytesTransferred];                
                Buffer.BlockCopy(e.Buffer, e.Offset, incomingData, 0, e.BytesTransferred);

                // Resume receiving
                StartReceive(e);
                hasResumed = true;

                // Process the message
                _processUdpMessageDelegate(incomingData, remoteEndPoint);
                incomingData = null;
                remoteEndPoint = null;

            }
            catch (HeaderParseException ex)
            {
                Logger.Debug(ex, "Failed to parse header from incoming UDP packet.");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error during UdpConnectionManager ProcessReceive");
            }
            finally
            {
                if (!hasResumed)
                {
                    StartReceive(e);
                }
            }
        }
        
    }
}
