﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Curse.Diagnostics
{
    public enum OutputMode
    {
        Debug = 1,
        File = 2
    }

    public class AsyncLogger : IDisposable
    {

        public static AsyncLogger GetNewLogger(string filenamePrefix = null, EventLog eventLog = null)
        {
            string baseDirectory = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Logs");
            return GetNewLogger(baseDirectory, filenamePrefix, eventLog);
        }

        public static AsyncLogger GetNewLogger(string baseDirectory, string fileNamePrefix, EventLog eventLog = null, long seedTime = 0)
        {
            if (seedTime == 0)
            {
                seedTime = DateTime.UtcNow.Ticks;
            }

            if (!Directory.Exists(baseDirectory))
            {
                Directory.CreateDirectory(baseDirectory);
            }

            string fileName = seedTime + ".log";

            if (fileNamePrefix != null)
            {
                fileName = fileNamePrefix + "-" + fileName;
            }
            return new AsyncLogger(Path.Combine(baseDirectory, fileName), eventLog);
        }

        private readonly ConcurrentQueue<string> _logQueue;
        private readonly string _filePath = null;

        private delegate void LogDelegate(string message);
        private readonly LogDelegate _writeMethod;
        private readonly EventLog _eventLog;

        private bool _isAlive = true;

        public AsyncLogger() : this(OutputMode.Debug, null, null)
        {
            
        }

        public AsyncLogger(string filePath, EventLog eventLog = null)
            : this(OutputMode.File, filePath, eventLog)
        {

            _filePath = filePath;
            DirectoryInfo di = new DirectoryInfo(Path.GetDirectoryName(filePath));
            di.Create();           
        }

        public AsyncLogger(OutputMode outputMode, string filePath = null, EventLog eventLog = null)
        {            
            switch (outputMode)
            {
                case OutputMode.Debug:
                    _writeMethod = WriteToDebug;
                    break;
                case OutputMode.File:
                    _writeMethod = WriteToFile;
                    break;
            }
            _eventLog = eventLog;
            _logQueue = new ConcurrentQueue<string>();
            new Thread(FlushQueueThread) { IsBackground = true, Name = "Async Logger" }.Start();
        }

        public void Dispose()
        {
            _isAlive = false;           
        }

        private static string FormatMessage(string message)
        {
            return "[" + DateTime.Now.ToString("M/d h:mm:ss tt") + "] " + message;
        }

        public void Log(string message, bool writeToEventLog = false, EventLogEntryType eventLogEntryType = EventLogEntryType.Information)
        {
            _logQueue.Enqueue(FormatMessage(message));

            if (writeToEventLog)
            {
                _eventLog.WriteEntry(message, eventLogEntryType);
            }
        }

        private void FlushQueueThread()
        {
            while (_isAlive)
            {
                string logMessage;
                while (_logQueue.TryDequeue(out logMessage))
                {
                    _writeMethod(logMessage);
                }    

                Thread.Sleep(10);
            }            
        }

        private void WriteToDebug(string message)
        {
            Debug.WriteLine(message);
        }

        private void WriteToFile(string message)
        {
            using (var fileWriter = new StreamWriter(_filePath, true))
            {
                fileWriter.AutoFlush = true;
                fileWriter.WriteLine(message);
            }
        }

    }
}
