﻿using System;
using System.Collections.Concurrent;
using System.Data;
using System.Linq;
using System.Net.Sockets;
using System.Threading;
using Curse.Logging;

namespace Curse.SocketInterface
{
    public enum ChannelType
    {
        Tcp = 1,
        Udp = 2
    }

    public abstract class BaseChannelProcessor : IDisposable
    {
        private static readonly LogCategory Logger = new LogCategory("BaseChannelProcessor")
            { Throttle = TimeSpan.FromSeconds(30) };

        protected static long StartTime = DateTime.UtcNow.Ticks;
        protected Message _currentMessage;
        protected BaseSocketInterface _socketInterface;
        protected Socket _socket;
        protected ISocketEventArgsPool _socketEventArgsPool;
        protected SocketAsyncEventArgs _eventArgs;
        protected ConcurrentQueue<Message> _messageQueue;
        protected readonly object _syncRoot = new object();
        public int ID { get; private set; }
        protected volatile bool _isSending = false;
        protected volatile bool _isAlive = true;
        private readonly ChannelType _type;
        private static int _maxQueueSize = 2;
        private readonly int _bufferOffset;

        private static int _instanceCounter = 0;

        public SocketAsyncEventArgs EventArgs
        {
            get { return _eventArgs; }
        }


#if SOCKET_LOGGING
       
        public void Trace(string message)
        {
            Logger.Trace(message);
        }
#else
        public void Trace(string message)
        {
        }
#endif


        private readonly object _disposeLock = new object();
        public void Dispose()
        {
            var lockAcquired = Monitor.TryEnter(_disposeLock, 500);

            if (!lockAcquired)
            {
                Logger.Warn("Failed to acquire a lock to dispose the channel processor. A dispose must already be in progress");
                return;
            }

            try
            {
                if (!_isAlive)
                {
                    return;
                }

                _isAlive = false;
                if (_currentMessage != null)
                {
                    _currentMessage.Dispose();
                    _currentMessage = null;
                }

                if (_eventArgs != null)
                {
                    _eventArgs.RemoteEndPoint = null;
                    _eventArgs.UserToken = null;
                    if (_socketEventArgsPool != null)
                    {
                        _socketEventArgsPool.Return(_eventArgs);
                    }
                    else
                    {
                        _eventArgs.Dispose();
                    }

                    _eventArgs = null;
                }

                _socketInterface = null;
                _messageQueue = null;
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(_disposeLock);
                }
            }


        }

        protected BaseChannelProcessor(BaseSocketInterface socketInterface, ISocketEventArgsPool socketEventArgsPool, SocketAsyncEventArgs e, ChannelType type, Socket socket)
        {

            ID = Interlocked.Increment(ref _instanceCounter);
            if (socketInterface.ClientID > 0)
            {
                ID = socketInterface.ClientID;
            }

            _type = type;

            _maxQueueSize = type == ChannelType.Tcp ? 20 : 40;

#if DEBUG || LOAD_TESTING
            _maxQueueSize = 0;
#endif
            _messageQueue = new ConcurrentQueue<Message>();
            _socketInterface = socketInterface;
            _socket = socket;
            _socketEventArgsPool = socketEventArgsPool;
            _eventArgs = e;
            _bufferOffset = e.Offset;
            _eventArgs.UserToken = this;
        }

        public void SendMessage(Message message)
        {
            var messageQueue = _messageQueue;
            var eventArgs = _eventArgs;

            if (messageQueue == null || eventArgs == null) // It's been disposed
            {
                return;
            }

            // If this channel is queued up, discard the data
            if (_maxQueueSize > 0 && messageQueue.Count > _maxQueueSize)
            {
                int[] messageTypes;

                try
                {
                    var items = messageQueue.ToArray();
                    messageTypes = items.Select(p => p.Header.MessageType).ToArray();
                }
                catch
                {
                    messageTypes = new int[0];
                }
                
                Logger.Warn("Channel has reached the maximum queue size!", new { Type = _type, messageQueue.Count, messageTypes });
                return;
            }

            // Queue up the message
            messageQueue.Enqueue(message);

            // Try to send it immediately
            StartSend(eventArgs);
        }

        public void StartSend(SocketAsyncEventArgs e)
        {
            try
            {
                // Ensure we're not currently sending, or disposing.
                var hasCompleted = false;

                lock (_syncRoot)
                {
                    if (_isSending || !_isAlive) // Just in case
                    {
                        return;
                    }

                    _isSending = true;

                    Trace("StartSend");

                    var messageQueue = _messageQueue;

                    if (messageQueue == null) // It's been disposed
                    {
                        return;
                    }

                    var message = _currentMessage;
                    var bytesTransferred = e.BytesTransferred;

                    if (message == null && messageQueue.TryDequeue(out message))
                    {
                        bytesTransferred = 0;
                        _currentMessage = message;
                    }

                    if (message == null)
                    {
                        Trace("No current message exists. Exiting events...");
                        _isSending = false;
                        return;
                    }

                    message.WriteOutgoingData(_bufferOffset, e, bytesTransferred);

                    try
                    {
                        hasCompleted = _type == ChannelType.Tcp ? !_socket.SendAsync(e) : !_socket.SendToAsync(e);
                    }
                    catch (ObjectDisposedException)
                    {
                        return;
                    }
                }


                if (hasCompleted)
                {
                    ProcessSend(e);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to start sending on channel processor. Disconnecting client.");                
                RaiseDisconnect(SocketDisconnectReason.ProcessingError);                
            }

        }

        protected void RaiseDisconnect(SocketDisconnectReason reason)
        {
            var socketInterface = _socketInterface;
            if (socketInterface == null)
            {
                return;
            }

            socketInterface.RaiseDisconnect(reason);
        }

        protected void RaiseDisconnect(SocketError socketError)
        {
            var socketInterface = _socketInterface;
            if (socketInterface == null)
            {
                return;
            }

            socketInterface.RaiseDisconnect(socketError);
        }

        public void ProcessSend(SocketAsyncEventArgs e)
        {
            Trace("ProcessSend");
            
            lock (_syncRoot)
            {                
                if (!_isAlive || _socketInterface == null)
                {
                    return;
                }

            }

            if (e.SocketError != SocketError.Success)
            {
                RaiseDisconnect(e.SocketError);
                _isSending = false;
                return;
            }

            var currentMessage = _currentMessage;
            if (currentMessage != null && currentMessage.IsSendComplete(e.BytesTransferred))
            {
                Trace("Disposing completed message");
                currentMessage.Dispose();
                _currentMessage = null;
            }

            Trace("Resuming Send");

            lock (_syncRoot)
            {
                _isSending = false;
                StartSend(e);
            }

        }

        public void Start()
        {
            StartReceive(_eventArgs);
        }

        public void StartReceive(SocketAsyncEventArgs e)
        {
            try
            {
                Trace("StartReceive");
                var willRaiseEvent = false;

                lock (_syncRoot)
                {
                    if (!_isAlive)
                    {
                        return;
                    }

                    e.SetBuffer(_bufferOffset, SocketConstants.BufferSize);

                    try
                    {
                        willRaiseEvent = _type == ChannelType.Tcp ? _socket.ReceiveAsync(e) : _socket.ReceiveFromAsync(e);
                    }
                    catch (ObjectDisposedException)
                    {
                        return;
                    }
                }

                if (!willRaiseEvent)
                {
                    ProcessReceive(e);
                }
            }
            catch (Exception ex)
            {
                var socketInterface = _socketInterface;
                if (socketInterface != null)
                {
                    if (ex is InvalidOperationException)
                    {
                        var session = socketInterface.Session;
                        var age = DateTime.UtcNow - socketInterface.DateCreated;
                        Logger.Warn(ex, "Failed to start receiving on channel processor for socket interface. Client will be disconnected.",
                        new
                        {
                            socketInterface.ClientID,
                            socketInterface.IsConnected,
                            socketInterface.IsAuthenticated,
                            socketInterface.IsDisconnecting,
                            socketInterface.IsHandshaken,
                            HasSession = session != null,
                            SocketAgeSeconds = (int)age.TotalSeconds,
                            socketInterface.DateLastHandshake,
                        });
                    }
                    else
                    {
                        Logger.Error(ex, "Failed to start receiving on channel processor for socket interface. Client will be disconnected.");
                    }

                    RaiseDisconnect(SocketDisconnectReason.ProcessingError);
                }
                else
                {
                    Logger.Error(ex, "Failed to start receiving on channel processor, without a connected client.");
                }
            }
        }

        public abstract void ProcessReceive(SocketAsyncEventArgs e);

        protected void ProcessIncomingData(SocketAsyncEventArgs e)
        {
            if (e.BytesTransferred > Message.MaxMessageSize)
            {
                throw new Exception("More data transferred (" + e.BytesTransferred + ") than permitted (" + Message.MaxMessageSize + ")");
            }

            // Get the transferred bytes from the buffer
            var bytes = new byte[e.BytesTransferred];
            Buffer.BlockCopy(e.Buffer, e.Offset, bytes, 0, e.BytesTransferred);

            Trace("ProcessIncomingData");

            // Loop over the byte array, until every byte ha sbeen read
            var startingIndex = 0;

            var message = _currentMessage;

            while (startingIndex < bytes.Length)
            {

                if (message == null)
                {
                    message = _currentMessage = Message.FromIncoming();
                }

                if (!message.Header.IsComplete) // Create a new message, or add to the existing one.
                {
                    startingIndex += message.AddHeaderData(bytes, startingIndex);
                }
                else
                {
                    startingIndex += message.AddIncomingData(bytes, startingIndex);

                    if (message.IsReceiveComplete)
                    {

                        var socketInterface = _socketInterface;

                        // Dispatch the message, then dispose it.
                        if (socketInterface != null)
                        {
                            socketInterface.RaiseMessageReceived(message);
                        }

                        message.Dispose();
                        message = _currentMessage = null;
                    }
                }
            }
        }

        public void Disconnect()
        {            
            RaiseDisconnect(SocketDisconnectReason.UserInitiated);
        }

    }
}
