using System;
using System.Net.Sockets;
using System.Net;
using System.Security.Cryptography;
using Curse.Logging;

namespace Curse.SocketInterface
{

    /// <summary>
    /// This class is designed for use as the object to be assigned to the SocketAsyncEventArgs.UserToken property. 
    /// </summary>
    public class BaseSocketInterface : IDisposable, ISocketInterface
    {

        public event EventHandler<SocketDisconnectEventArgs> Disconnected;
        public event EventHandler Connected;
        public event EventHandler<MessageEventArgs> MessageReceived;

        public bool IsAuthenticated { get; set; }
        public volatile bool IsDisconnecting = false;
        public volatile bool IsConnected = false;
        public bool IsHandshaken { get; set; }
        private readonly object _syncRoot = new object();
        private readonly DateTime _dateCreated;
        private DateTime _dateLastHealthCheck;
        protected volatile bool _isDisposing = false;
        private bool _udpEnabled = false;

        private int _remoteEndpointPort = -1;

        private static readonly TimeSpan _healthCheckInterval = TimeSpan.FromMinutes(1);
        private static readonly int _idleConnectionTimeout = (int)TimeSpan.FromMinutes(1).TotalMilliseconds;

        public Socket Socket
        {
            get;
            protected set;
        }

        public DateTime DateCreated
        {
            get { return _dateCreated; }
        }

        public int DateLastHandshake { get; set; }

        public int ClientID { get; set; }
        public ISocketSession Session { get; set; }
        public OutgoingChannelProcessor OutgoingChannelProcessor { get; private set; }
        public IncomingChannelProcessor IncomingChannelProcessor { get; private set; }
        public OutgoingChannelProcessor DataChannelProcessor { get; private set; }
        public IncomingUdpChannelProcessor IncomingDataChannelProcessor { get; private set; }

        public IPAddress RemoteAddress
        {
            get { return Socket != null ? ((IPEndPoint)Socket.RemoteEndPoint).Address : null; }
        }

        protected BaseSocketInterface(int clientID)
        {
            ClientID = clientID;
            _dateCreated = DateTime.UtcNow;
        }

        protected void Initialize(Socket tcpSocket,
            ISocketEventArgsPool tcpPool,
            SocketAsyncEventArgs outgoingEventArgs,
            SocketAsyncEventArgs incomingEventArgs,
            Socket udpSocket,
            ISocketEventArgsPool udpPool,
            SocketAsyncEventArgs dataEventArgs,
            bool requireHandshakeBeforeSend,
            SocketAsyncEventArgs dataIncomingEventArgs = null)
        {
            IsDisconnecting = false;
            OutgoingChannelProcessor = new OutgoingChannelProcessor(this, tcpPool, outgoingEventArgs, ChannelType.Tcp, tcpSocket);
            IncomingChannelProcessor = new IncomingChannelProcessor(this, tcpPool, incomingEventArgs, ChannelType.Tcp, tcpSocket);
            IncomingChannelProcessor.Start();

            if (udpSocket != null)
            {
                _udpEnabled = true;
                DataChannelProcessor = new OutgoingChannelProcessor(this, udpPool, dataEventArgs, ChannelType.Udp, udpSocket);
            }

            if (dataIncomingEventArgs != null)
            {
                IncomingDataChannelProcessor = new IncomingUdpChannelProcessor(this, udpPool, dataIncomingEventArgs, udpSocket);
            }

            IsConnected = true;

            if (!requireHandshakeBeforeSend)
            {
                IsHandshaken = true;
            }

            if (Connected != null)
            {
                Connected(this, new EventArgs());
            }
        }

        /// <summary>
        ///  Polls the socket, to see if it is still healthy
        /// </summary>
        /// <returns></returns>
        public bool IsSocketHealthy()
        {
            _dateLastHealthCheck = DateTime.UtcNow;

            try
            {
                if (Environment.TickCount - DateLastHandshake > _idleConnectionTimeout)
                {
                    return false;
                }

                var part1 = Socket.Poll(1000, SelectMode.SelectRead);
                var part2 = (Socket.Available == 0);
                if (part1 & part2)
                {
                    return false;
                }
            }
            catch
            {
                return false;
            }

            return true;
        }

        public bool NeedsHealthCheck()
        {            
            if (DateTime.UtcNow.Subtract(_dateCreated) < TimeSpan.FromMinutes(1))
            {
                return false;
            }

            return DateTime.UtcNow.Subtract(_dateLastHealthCheck) >= _healthCheckInterval;
        }

        public virtual void Dispose()
        {
            if (_isDisposing)
            {
                return;
            }
            _isDisposing = true;
            IsConnected = false;
            IsDisconnecting = true;

            if (OutgoingChannelProcessor != null)
            {
                OutgoingChannelProcessor.Dispose();
            }

            if (IncomingChannelProcessor != null)
            {
                IncomingChannelProcessor.Dispose();
            }

            if (DataChannelProcessor != null)
            {
                DataChannelProcessor.Dispose();
            }

            if (Socket != null)
            {
                try
                {
                    if (Socket.Connected)
                    {
                        Socket.Shutdown(SocketShutdown.Both);
                    }

                    try
                    {
                        Socket.Close(1);
                    }
                    catch { }
                    
                }
                catch { }
                
                Socket.Dispose();
                Socket = null;
            }
                              
        }    
   
        public SymmetricAlgorithm EncryptionAlgorithm
        {
            get
            {
                return Session != null ? Session.EncryptionAlgorithm : null;
            }
        }

        public void SendContract(IContract contract)
        {
            if (!IsConnected)
            {
                return;
            }
            
            var message = contract.ToMessage(ClientID, EncryptionAlgorithm);
            OutgoingChannelProcessor.SendMessage(message);
        }

        public void SendString(string serializedMessage)
        {
            throw new InvalidOperationException("Cannot use BaseSocketInterface to send a raw string");
        }

        public void SendMessage(Message message)
        {
            if (!IsConnected)
            {
                return;
            }

            OutgoingChannelProcessor.SendMessage(message);
        }

        public void SendPacket(IContract contract)
        {
            if (!IsConnected || !IsHandshaken)
            {
                return;
            }

            if (!_udpEnabled)
            {
                throw new Exception("Packets can only be sent when UDP is enabled.");
            }

            DataChannelProcessor.SendMessage(contract.ToMessage(ClientID, EncryptionAlgorithm));
        }

        public void SendPacket(Message message)
        {
            if (!IsConnected || !IsHandshaken)
            {
                return;
            }

            if (!_udpEnabled)
            {
                throw new Exception("Packets can only be sent when UDP is enabled.");
            }

            DataChannelProcessor.SendMessage(message);
        }

        public void RaiseMessageReceived(Message message)
        {
            if (MessageReceived != null)
            {
                MessageReceived(this, new MessageEventArgs(message));
            }
        }

        public void RaiseDisconnect(SocketError socketError)
        {
            RaiseDisconnect(new SocketDisconnectEventArgs(socketError));
        }

        public void RaiseDisconnect(SocketDisconnectReason reason)
        {
            RaiseDisconnect(new SocketDisconnectEventArgs(reason));            
        }

        private void RaiseDisconnect(SocketDisconnectEventArgs eventArgs)
        {
            lock (_syncRoot)
            {

                if (IsDisconnecting)
                {
                    return;
                }
                IsDisconnecting = true;
            }

            Dispose();

            if (Disconnected != null)
            {
                Disconnected(this, eventArgs);
            }
            else
            {
                Logger.Warn("RaiseDisconnect called on SocketInerface, with no event handler!");
            }            
        }

        public void UpdateRemoteEndpoint(IPEndPoint endPoint)
        {
            if (DataChannelProcessor == null)
            {
                return;
            }

            if (_remoteEndpointPort != endPoint.Port)
            {                
                DataChannelProcessor.EventArgs.RemoteEndPoint = endPoint;
                _remoteEndpointPort = endPoint.Port;
                return;
            }

        }
    }
}
