using System;
using System.Collections.Generic;
using System.Text;
using System.Net.Sockets;
using System.Collections.Concurrent;

namespace Curse.SocketInterface
{
    /// <summary>
    /// This class creates a single large buffer which can be divided up and assigned to SocketAsyncEventArgs objects for use
    /// with each socket I/O operation.  This enables bufffers to be easily reused and gaurds against fragmenting heap memory.
    /// 
    /// The operations exposed on the BufferManager class are not thread safe.
    /// </summary>
    public class BufferManager
    {
        readonly int _totalBufferSize;                 // the total number of bytes controlled by the buffer pool
        readonly int _socketBufferSize;
        readonly byte[] _buffer;                // the underlying byte array maintained by the Buffer Manager
        readonly ConcurrentStack<int> _freeIndexPool;
        int _currentIndex;
        
        public BufferManager(int totalBytes, int bufferSize)
        {
            _totalBufferSize = totalBytes;
            _currentIndex = 0;
            _socketBufferSize = bufferSize;
            _freeIndexPool = new ConcurrentStack<int>();
            _buffer = new byte[_totalBufferSize];
        }

        public int BufferSize { get { return _socketBufferSize; } }
        
        /// <summary>
        /// Assigns a buffer from the buffer pool to the specified SocketAsyncEventArgs object
        /// </summary>
        /// <returns>true if the buffer was successfully set, else false</returns>
        public bool SetBuffer(SocketAsyncEventArgs args)
        {

            if (_freeIndexPool.Count > 0)
            {
                int nextIndex = 0;
                if (!_freeIndexPool.TryPop(out nextIndex))
                {
                    throw new Exception("No buffer could be allocated!");
                }
                args.SetBuffer(_buffer, nextIndex, _socketBufferSize);
            }
            else
            {
                if ((_totalBufferSize - _socketBufferSize) < _currentIndex)
                {
                    throw new Exception("No buffer could be allocated! Buffer Size: " + _totalBufferSize + ", Current Index: " + _currentIndex);                    
                }
                args.SetBuffer(_buffer, _currentIndex, _socketBufferSize);
                _currentIndex += _socketBufferSize;
            }
            return true;
        }

        /// <summary>
        /// Removes the buffer from a SocketAsyncEventArg object.  This frees the buffer back to the 
        /// buffer pool
        /// </summary>
        public void FreeBuffer(SocketAsyncEventArgs args)
        {
            _freeIndexPool.Push(args.Offset);
            args.SetBuffer(null, 0, 0);
        }
    }
}
