﻿using Curse.Logging;
using Curse.SocketInterface;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace Curse.SocketInterface
{    
    public enum ClientSocketOptions
    {
        TcpOnly,
        UdpOnly,
        TcpAndUdp
    }

    public class ClientSocketInterface : BaseSocketInterface
    {        
        public event EventHandler Connecting;
        private readonly BufferManager _bufferManager;
        private readonly IPAddress _remoteAddress;
        private readonly ClientSocketOptions _options;

        public int Port { get; protected set; }

        private Socket _udpSocket;
       
        public ClientSocketInterface(string hostName, int port, ClientSocketOptions options = ClientSocketOptions.TcpAndUdp)
            : this(Dns.GetHostEntry(hostName).AddressList.FirstOrDefault(p => p.AddressFamily == AddressFamily.InterNetwork), port)
        {
            _options = options;
        }

        public ClientSocketInterface(IPAddress remoteAddress, int port, ClientSocketOptions options = ClientSocketOptions.TcpAndUdp)
            : base(0)
        {            
            // Create a socket and connect to the server
            _options = options;
            _remoteAddress = remoteAddress;            
            Port = port;
            _bufferManager = new BufferManager(SocketConstants.BufferSize * SocketConstants.OperationsToPreallocate * 4,
               SocketConstants.BufferSize * SocketConstants.OperationsToPreallocate);
            
            Socket = new Socket(_remoteAddress.AddressFamily, SocketType.Stream, ProtocolType.Tcp)
            {
                NoDelay = true
            };
            
            IsAuthenticated = true;
        }
        
        protected bool Connect(int millisecondsTimeout = 2000)
        {
            IAsyncResult result = null;
            var success = false;

            try
            {
                result = Socket.BeginConnect(_remoteAddress, Port, null, null);
                success = result.AsyncWaitHandle.WaitOne(millisecondsTimeout, true);
            }
            catch (SocketException ex)
            {
                Logger.Warn(ex, "[ClientSocketInterface] Async connection failed, trying synchronous method...");
                                
                try
                {
                    Socket.Connect(_remoteAddress, Port);
                    success = true;
                }
                catch (Exception ex2)
                {
                    Logger.Error(ex2, "[ClientSocketInterface] Synchronous connection failed!");
                }                
            }

            // If all went well, make sure EndConnect gets called
            try
            {
                if (success && result != null)
                {
                    Socket.EndConnect(result);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to call EndConnect on connected socket.");
                success = false;
            }

            // If something went wrong or we aren't connected, clean up the socket
            try
            {
                if (!success || !Socket.Connected)
                {
                    Socket.Close(0);
                    return false;
                }
            }
            catch
            {
                return false;
            }

            InitializeConnection(Socket);

            return true;
        }

        void InitializeConnection(Socket socket)
        {            
            // TCP
            var tcpOutgoingArgs = new SocketAsyncEventArgs();
            tcpOutgoingArgs.AcceptSocket = socket;
            tcpOutgoingArgs.Completed += Tcp_SendCompleted;
            _bufferManager.SetBuffer(tcpOutgoingArgs);
            
            var tcpIncomingArgs = new SocketAsyncEventArgs();
            tcpIncomingArgs.AcceptSocket = socket;
            tcpIncomingArgs.Completed += Tcp_ReceiveCompleted;
            _bufferManager.SetBuffer(tcpIncomingArgs);

            // UDP
            SocketAsyncEventArgs dataOutgoingArgs = null;
            SocketAsyncEventArgs dataIncomingArgs = null;
            if (_options == ClientSocketOptions.UdpOnly || _options == ClientSocketOptions.TcpAndUdp)
            {
                _udpSocket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
#if __MonoCS__
                _udpSocket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.Debug, true);
#else
                _udpSocket.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.PacketInformation, true);
#endif


                var endPoint = new IPEndPoint(_remoteAddress, Port);

                dataOutgoingArgs = new SocketAsyncEventArgs();
                dataOutgoingArgs.Completed += Udp_SendCompleted;
                dataOutgoingArgs.RemoteEndPoint = endPoint;
                _bufferManager.SetBuffer(dataOutgoingArgs);

                dataIncomingArgs = new SocketAsyncEventArgs();
                dataIncomingArgs.Completed += Udp_ReceiveCompleted;
                dataIncomingArgs.RemoteEndPoint = endPoint;
                _bufferManager.SetBuffer(dataIncomingArgs);
            }

            Initialize(socket, null, tcpOutgoingArgs, tcpIncomingArgs, _udpSocket, null, dataOutgoingArgs, false, dataIncomingArgs);
        }
        
        public void SendTestMessage()
        {            
            byte[] body = System.Text.Encoding.UTF8.GetBytes("Hello World!");
            var message = Message.FromOutgoing(0, body, 1, false, OutgoingChannelProcessor);
            OutgoingChannelProcessor.SendMessage(message);
        }

        void Tcp_ReceiveCompleted(object sender, SocketAsyncEventArgs e)
        {
            var channel = (IncomingChannelProcessor)e.UserToken;
            if (channel == null)
            {
                return;
            }
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.Receive:
                    channel.ProcessReceive(e);
                    break;                             
                default:
                    throw new Exception("Invalid operation");
            }
        }

        void Tcp_SendCompleted(object sender, SocketAsyncEventArgs e)
        {
            var channel = (OutgoingChannelProcessor)e.UserToken;
            if (channel == null)
            {
                return;
            }
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.Send:
                    channel.ProcessSend(e);
                    break;
                default:
                    throw new Exception("Invalid operation");
            }
        }

        void Udp_SendCompleted(object sender, SocketAsyncEventArgs e)
        {
            var channel = (OutgoingChannelProcessor)e.UserToken;

            if (channel == null)
            {
                return;                
            }            

            switch (e.LastOperation)
            {
                case SocketAsyncOperation.SendTo:
                    channel.ProcessSend(e);
                    break;
                default:
                    throw new Exception("Invalid operation");
            }
        }

        void Udp_ReceiveCompleted(object sender, SocketAsyncEventArgs e)
        {
            var channel = (IncomingUdpChannelProcessor)e.UserToken;

            if (channel == null)
            {
                return;
            }

            switch (e.LastOperation)
            {

                case SocketAsyncOperation.ReceiveFrom:
                case SocketAsyncOperation.ReceiveMessageFrom:
                    channel.ProcessReceive(e);
                    break;
                default:
                    throw new Exception("Invalid operation");
            }
        }

        public override void Dispose()
        {
            if (_isDisposing)
            {
                return;
            }
            
            base.Dispose();

            if (_udpSocket != null)
            {
                _udpSocket.Dispose();
            }            
        }

        public void Disconnect(SocketDisconnectReason reason)
        {            
            RaiseDisconnect(reason);
        }
    }
}
