﻿using System;
using System.Net.Sockets;
using Curse.Logging;

namespace Curse.SocketInterface
{
    public class IncomingUdpChannelProcessor : BaseChannelProcessor
    {
        public IncomingUdpChannelProcessor(BaseSocketInterface socketInterface, ISocketEventArgsPool socketEventArgsPool, SocketAsyncEventArgs e, Socket socket)
            : base(socketInterface, socketEventArgsPool, e, ChannelType.Udp, socket)
        {

        }

        public Func<BaseSocketInterface, byte[], bool> HandlePacket { get; set; }  

        public override void ProcessReceive(SocketAsyncEventArgs e)
        {
            var socketInterface = _socketInterface;

            // The attached socket interface has already been disposed, or is in the process of disconnecting.
            if (socketInterface == null || socketInterface.IsDisconnecting)
            {
                return;
            }

            // If it was not successful, return it to the pool
            if (e.SocketError != SocketError.Success)
            {
                socketInterface.RaiseDisconnect(e.SocketError);
                return;
            }

            var hasResumed = false;

            try
            {
                if (e.BytesTransferred == 0)
                {
                    return;
                }

                // Copy the buffer
                var incomingData = new byte[e.BytesTransferred];
                Buffer.BlockCopy(e.Buffer, e.Offset, incomingData, 0, e.BytesTransferred);
                StartReceive(e);
                hasResumed = true;

                if (HandlePacket == null || !HandlePacket(socketInterface, incomingData))
                {
                    if (incomingData.Length < MessageHeader.HeaderSize)
                    {
                        return;
                    }

                    var message = Message.FromIncoming();
                    var startingIndex = message.AddHeaderData(incomingData, 0);
                    startingIndex += message.AddIncomingData(incomingData, startingIndex);
                    socketInterface.RaiseMessageReceived(message);
                }

                Trace("ProcessReceive");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "IncomingUdpChannelProcessor ProcessReceive error", new { e.BytesTransferred });
            }
            finally
            {
                if (!hasResumed)
                {
                    StartReceive(e);
                }
            }
        }

    }
}
