﻿using System;
using System.Net.Sockets;

namespace Curse.SocketInterface
{
    public class Message : IDisposable
    {       
        public MessageHeader Header { get; set; }
        
        private Message(MessageHeader header)
        {
            Header = header;
        }

        public static Message FromIncoming()
        {
            return new Message(new MessageHeader());
        }

        public int AddHeaderData(byte[] data, int startingIndex)
        {
            if (Header == null)
            {
                throw new ArgumentException("Header is null!");
            }

            var bytesProcessed = Header.AddIncomingData(data, startingIndex);            

            if (Header.IsComplete)
            {
                InitializeBody();
            }

            return bytesProcessed;
        }

        public static Message FromOutgoing(int sessionID, byte[] body, int messageType, bool isSerialized, BaseChannelProcessor parentChannel = null)
        {
            var message = new Message(new MessageHeader(sessionID, body.Length, messageType, isSerialized));
            message.SetOutgoingData(body);
            return message;
        }

        public static Message FromRawOutgoing(byte[] body)
        {
            var message = new Message(new MessageHeader(0, body.Length - MessageHeader.HeaderSize, 0, false));
            message.SetOutgoingData(body, true);
            return message;
        }

        public void Dispose()
        {
            _buffer = null;            
        }

        public byte[] GetBytes()
        {
            if (_buffer == null)
            {
                throw new InvalidOperationException("GetBytes() cannot be called, when _buffer is null!");
            }
            return _buffer;
        }

        public byte[] GetBody()
        {
            if (_buffer == null)
            {
                throw new InvalidOperationException("GetBytes() cannot be called, when _buffer is null!");
            }

            byte[] body = new byte[Header.BodySize];
            Buffer.BlockCopy(_buffer, MessageHeader.HeaderSize, body, 0, Header.BodySize);
            return body;
        }

        private byte[] _buffer;
        private int _incomingIndex = 0;
        
        private bool _bodyIsInitialized = false;
        public const int MaxMessageSize = 1024000;
        public void InitializeBody()
        {
            if (_bodyIsInitialized)
            {
                throw new Exception("Body already initialized!");
            }

            if (Header == null)
            {
                throw new Exception("header is null!");
            }

            if (!Header.IsComplete)
            {
                throw new Exception("header is not complete!");
            }

            if (Header.BodySize < 1)
            {
                throw new Exception("Message BodySize property must be at least one byte. It has been parsed as being " + Header.BodySize + " bytes!");
            }

            if (Header.TotalSize > MaxMessageSize)
            {
                throw new Exception("Message TotalSize property cannot be larger than the maximum defined message size of " + MaxMessageSize + " bytes. It has been parsed as being " + Header.TotalSize + " bytes!");
            }

            _buffer = new byte[Header.TotalSize];
            _bodyIsInitialized = true;
            AddIncomingData(Header.GetBytes());            
        }

        public void SetOutgoingData(byte[] data, bool omitHeader = false)
        {
            if (omitHeader)
            {
                _buffer = data;
            }
            else
            {
                _buffer = new byte[Header.TotalSize];
                Buffer.BlockCopy(Header.GetBytes(), 0, _buffer, 0, MessageHeader.HeaderSize);
                Buffer.BlockCopy(data, 0, _buffer, MessageHeader.HeaderSize, data.Length);
            }
            _bodyIsInitialized = true;
        }

        public int AddIncomingData(byte[] data, int startingIndex = 0)
        {
            if (!_bodyIsInitialized || _buffer == null)
            {
                throw new Exception("Body must be initialized!");
            }

            int count = data.Length - startingIndex;
            if (count > ReceiveBytesRemaining)
            {
                count = ReceiveBytesRemaining;
            }
            Buffer.BlockCopy(data, startingIndex, _buffer, _incomingIndex, count);
            _incomingIndex += count;
            return count;
        }

        private int _totalBytesSent;

        public bool WriteOutgoingData(int initialOffset, SocketAsyncEventArgs e, int bytesTransferred)
        {

            _totalBytesSent += bytesTransferred;

            int bytesRemaining = _buffer.Length - _totalBytesSent;
            int bytesToSend = bytesRemaining;
            if(bytesRemaining > SocketConstants.BufferSize)
            {
                bytesToSend = SocketConstants.BufferSize;                
            }
            Buffer.BlockCopy(_buffer, _totalBytesSent, e.Buffer, initialOffset, bytesToSend);   
            e.SetBuffer(initialOffset, bytesToSend);

            return true;

        }

        public int WriteOutgoingData(ByteBuffer buf)
        {
            if (_buffer.Length > buf.Count)
            {
                throw new ArgumentException("not enough room in output buffer");
            }

            _buffer.BlockCopy(0, buf, 0, _buffer.Length);
            return _buffer.Length;
        }
        
        public int ReceiveBytesRemaining
        {
            get { return Header.TotalSize - _incomingIndex; }
        }

        public bool IsReceiveComplete
        {
            get { return ReceiveBytesRemaining == 0; }
        }

        public bool IsSendComplete(int bytesTransferred)
        {
            return _totalBytesSent + bytesTransferred >= Header.TotalSize;
        }
    }
}
