﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.SocketInterface.Exceptions;

namespace Curse.SocketInterface
{
    public class MessageHeader
    {
        public const Int32 HeaderSize = 13;

        public Int32 ClientID { get; set; }
        public Int32 BodySize { get; set; }        
        public Int32 MessageType { get; set; }
        public bool IsSerialized { get; set; }

        public Int32 TotalSize { get { return HeaderSize + BodySize; } }

        public MessageHeader()
        {
        }

        public MessageHeader(int sessionID, int bodySize, int messageType, bool isSerialized)
        {
            ClientID = sessionID;
            BodySize = bodySize;
            MessageType = messageType;
            IsSerialized = isSerialized;

            _buffer.SetBytesLE(ClientID, 0);
            _buffer.SetBytesLE(BodySize, 4);
            _buffer.SetBytesLE(MessageType, 8);
            _buffer.SetBytesLE(IsSerialized, 12);

            _bytesRead = HeaderSize;
        }
        
        private readonly byte[] _buffer = new byte[HeaderSize];

        private int _bytesRead = 0;

        public bool IsComplete
        {
            get { return _bytesRead == HeaderSize; }
        }
    

        public int AddIncomingData(byte[] bytes, int startingIndex)
        {
            if (IsComplete)
            {
                throw new Exception("Cannot add header data. It is already complete!");
            }

            int bytesRead = 0;
            while (_bytesRead < HeaderSize)
            {
                Buffer.BlockCopy(bytes, startingIndex + bytesRead, _buffer, _bytesRead, 1);
                ++bytesRead;
                ++_bytesRead;
                if (startingIndex + bytesRead >= bytes.Length)
                {
                    break;                    
                }
            }

            if (IsComplete)
            {
                ParseHeader(bytes, startingIndex);
            }

            return bytesRead;
        }

        private void ParseHeader(byte[] deliveredBytes, int startingIndex)
        {
            ClientID = _buffer.ToInt32LE(0);
            BodySize = _buffer.ToInt32LE(4);
            MessageType = _buffer.ToInt32LE(8);
            IsSerialized = _buffer.ToBooleanLE(12);

            if (BodySize < 1 || BodySize > Message.MaxMessageSize)
            {
#if DEBUG
                string exceptionMessage = "Total body size is not within the valid range: " + BodySize + ". It must be between 1 and 102,400 bytes!"
                    + "Starting Index: " + startingIndex + Environment.NewLine
                    + "Delivered bytes: " + string.Join(" ", deliveredBytes);
#else
                string exceptionMessage = "Total body size is not within the valid range: " + BodySize +
                                   ". It must be between 1 and 1,024,000 bytes!";
#endif                
                throw new HeaderParseException(exceptionMessage);
            }
        }

        public byte[] GetBytes()
        {            
            return _buffer;
        }

        
    }
}
