﻿using System;
using System.Collections.Concurrent;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Curse.Logging;

namespace Curse.SocketInterface
{
#if false // make sure this class isn't used (it hasn't been updated with fixes from RTCSocketPair)
    public class UdpSocketServer
    {
        private const int LargeObjectHeapMin = 85000;
        private const int BufferSize = 1500;
        private const int MinBufferCount = LargeObjectHeapMin / BufferSize + 1;

        public delegate void ReceiveCallback(int port, SocketBuffer msg);

        private readonly ConcurrentDictionary<int, Socket> _sockets = new ConcurrentDictionary<int, Socket>();
        private readonly ReceiveCallback _recvCallback;

        private int _availableBuffers;

        private volatile bool _isFinished;

        public int AvailableBuffers
        {
            get { return Interlocked.CompareExchange(ref _availableBuffers, 0, 0); }
        }

        public UdpSocketServer(ReceiveCallback callback, int bufferCount = 0)
        {
            _recvCallback = callback;

            // Make sure the allocated buffer is large enough that it gets places 
            // on the large object heap since it will be permanently pinned anyway
            if (bufferCount < MinBufferCount)
            {
                bufferCount = MinBufferCount;
            }
            var buffer = new byte[bufferCount * BufferSize];

            // Fill the stack (without synchronization since we're still in the constructor)
            var head = default(SocketAsyncEventArgs);
            for (var i = 0; i < bufferCount; ++i)
            {
                var e = new SocketAsyncEventArgs();
                e.Completed += Completed;
                e.RemoteEndPoint = new IPEndPoint(IPAddress.Any, 0);
                e.SetBuffer(buffer, i * BufferSize, BufferSize);
                e.UserToken = head;
                head = e;
            }
            _head = head;
            _availableBuffers = bufferCount;
        }

        public int BindSocket(ref int port, int queueDepth)
        {
            // Lock to make sure we don't try to bind a new socket during shutdown
            lock (_sockets)
            {
                if (_isFinished)
                {
                    throw new InvalidOperationException("Shutdown was called");
                }

                // Create the socket
                var socket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
                socket.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.PacketInformation, true);
                socket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
                socket.Blocking = false;
                socket.Bind(new IPEndPoint(IPAddress.Any, port));

                // Handle the case where port was 0 and was dynamically assigned when bound
                port = ((IPEndPoint)socket.LocalEndPoint).Port;

                // Register the socket
                if (!_sockets.TryAdd(port, socket))
                {
                    Close(socket);
                    throw new InvalidOperationException("port is already bound");
                }

                // Pop receive buffers until we fill the desired queueDepth
                for (var i = 0; i < queueDepth; ++i)
                {
                    var e = Pop();
                    if (e == null)
                    {
                        // Not enough buffers were available
                        Logger.Warn("Not enough AvailableBuffers to fill queueDepth",
                            new { Requested = queueDepth, Available = i });
                        return i;
                    }
                    StartReceive(socket, e);
                }
                return queueDepth;
            }
        }

        public void Shutdown()
        {
            // Lock to make sure we don't try to bind a new socket during shutdown
            lock (_sockets)
            {
                _isFinished = true;

                // The collection can't be modified within this lock, so this is safe
                foreach (var kvp in _sockets)
                {
                    Close(kvp.Value);
                }
                _sockets.Clear();
            }
        }

        private void Close(Socket socket)
        {
            try
            {
                socket.Shutdown(SocketShutdown.Both);
                socket.Close(1);
                socket.Dispose();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to close socket");
            }
        }

        private void StartReceive(Socket socket, SocketAsyncEventArgs e)
        {
            e.SetRemoteEndPoint(IPAddress.Any, 0);
            e.SetBufferLength(BufferSize);

            var willRaiseEvent = socket.ReceiveFromAsync(e);

            if (!willRaiseEvent)
            {
                OnReceiveComplete(socket, e);
            }
        }

        private void Completed(object sender, SocketAsyncEventArgs e)
        {
            if (_isFinished)
                return;

            switch (e.LastOperation)
            {
                case SocketAsyncOperation.ReceiveFrom:
                    OnReceiveComplete(sender, e);
                    break;

                case SocketAsyncOperation.SendTo:
                    OnSendComplete(sender, e);
                    break;

                default:
                    Logger.Warn("Unsupported operation", new { e.LastOperation });
                    break;
            }
        }

        private void OnReceiveComplete(object sender, SocketAsyncEventArgs e)
        {
            var socket = (Socket)sender;
            try
            {
                if (e.SocketError != SocketError.Success)
                    return;

                // Process the packet
                var port = ((IPEndPoint)socket.LocalEndPoint).Port;
                _recvCallback(port, new SocketBuffer(e));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing ");
            }
            finally
            {
                // Queue for another receive
                StartReceive(socket, e);
            }
        }

        public void StartSend<T>(int port, MessageBuilder<T> build, T ctx)
        {
            Socket socket;
            if (!_sockets.TryGetValue(port, out socket))
            {
                throw new ArgumentException("not found", "port");
            }

            var willRaiseEvent = false;

            var e = Pop();
            try
            {
                // Invoke the callback to fill the buffer
                var result = build(ctx, e.GetSendBuffer(BufferSize));
                if (result.EndPoint == null || result.Length <= 0)
                    return;

                // Set the destination address and actual length
                e.SetRemoteEndPoint(result.EndPoint.Address, result.EndPoint.Port);
                e.SetBufferLength(result.Length);

                // Queue the send
                willRaiseEvent = socket.SendToAsync(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error while sending");
            }
            finally
            {
                // Make sure e is returned to the pool if the event won't be raised
                if (!willRaiseEvent)
                {
                    OnSendComplete(socket, e);
                }
            }
        }

        private void OnSendComplete(object sender, SocketAsyncEventArgs e)
        {
            // Return to the send pool
            Push(e);
        }

        #region Concurrent SocketAsyncEventArgs Stack

        // This is a basic ConcurrentStack that avoids allocating linked list Node objects
        // by using the UserToken property of SocketAsyncEventArgs for this purpose.

        private volatile object _head;

#pragma warning disable 420 // ref to volatile field

        private void Push(SocketAsyncEventArgs e)
        {
            var spin = new SpinWait();

            e.UserToken = _head;
            while (Interlocked.CompareExchange(ref _head, e, e.UserToken) != e.UserToken)
            {
                // Spin before re-reading _head to try again
                spin.SpinOnce();
                e.UserToken = _head;
            }
            Interlocked.Increment(ref _availableBuffers);
        }

        private SocketAsyncEventArgs Pop()
        {
            var spin = new SpinWait();

            var e = (SocketAsyncEventArgs)_head;
            while (e != null && Interlocked.CompareExchange(ref _head, e.UserToken, e) != e)
            {
                // Spin before re-reading _head to try again
                spin.SpinOnce();
                e = (SocketAsyncEventArgs)_head;
            }
            if (e != null)
            {
                Interlocked.Decrement(ref _availableBuffers);
            }
            return e;
        }

#pragma warning restore 420

        #endregion Concurrent SocketAsyncEventArgs Stack
    }
#endif
}
