﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using Curse.Logging;
using Curse.MurmurHash;
using ProtoBuf;
using Curse.SocketInterface;
using System.Security.Cryptography;
using Newtonsoft.Json;

namespace Curse.SocketMessages
{
    public abstract class Contract<T> : IContract
    {
        public static int MessageType;
        public static bool IsSerialized;
        public static bool IsEncrypted;

        static Contract()
        {
            var typeName = Encoding.ASCII.GetBytes(typeof(T).FullName);

            var serviceContractAttribute = Attribute.GetCustomAttributes(typeof(T), typeof(ServiceContractAttribute)).FirstOrDefault() as ServiceContractAttribute;
            
            if (serviceContractAttribute != null && serviceContractAttribute.Name != null)
            {
                typeName = Encoding.ASCII.GetBytes(typeof(T).Namespace + "." + serviceContractAttribute.Name);
            }

            var code = MurmurHash2.GetHashCode(ref typeName, typeName.Length, 1);
            MessageType = (int)code;
            IsSerialized = Attribute.GetCustomAttributes(typeof(T), typeof(ProtoContractAttribute)).Any();
            IsEncrypted = Attribute.GetCustomAttributes(typeof(T), typeof(EncryptedContractAttribute)).Any();

            try
            {
                if (IsSerialized)
                {                    
                    ContractSerializer.PrepareSerializer<T>();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to prepare serializers.");
            }
        }

        public virtual string ToJsonMessage(int sessionID = 0, SymmetricAlgorithm encryptionAlgorithm = null)
        {
            var message = new JsonMessage<Contract<T>> { TypeID = MessageType, Body = this };
            return JsonConvert.SerializeObject(message);
        }

        public virtual Message ToMessage(int sessionID = 0, SymmetricAlgorithm encryptionAlgorithm = null)
        {
            byte[] serialized;
            using (var memoryStream = new MemoryStream())
            {
                memoryStream.Position = 0;
                ContractSerializer.Serialize(memoryStream, this);
                serialized = memoryStream.ToArray();
            }

            if (IsEncrypted)
            {

                if (encryptionAlgorithm != null)
                {
                    serialized = encryptionAlgorithm.Encrypt(serialized);
                }
#if DEBUG
                else if (Debugger.IsAttached)
                {
                    Debugger.Break();
                }
#endif
            }

            if (serialized.Length == 0)
            {
                throw new InvalidOperationException("Serialized data for type '" + typeof(T).Name + "' is empty!");
            }

            if (serialized.Length > Message.MaxMessageSize)
            {
                Logger.Warn("Message created that larger than the maximum configured message size. Size: " + serialized.Length + " bytes. Type: " + typeof(T).Name);
            }

            return Message.FromOutgoing(sessionID, serialized, MessageType, IsSerialized);
        }

        public virtual void ParseMessage(Message message)
        {
            throw new NotImplementedException();
        }

        public virtual bool Validate()
        {
            return true;
        }

        public static T FromMessage(Message message, SymmetricAlgorithm encryptionAlgorithm = null)
        {
            T model;

            byte[] messageBody = message.GetBody();

            if (IsEncrypted)
            {
                if (encryptionAlgorithm != null)
                {
                    messageBody = encryptionAlgorithm.Decrypt(messageBody);
                }
#if DEBUG
                else if (Debugger.IsAttached)
                {                                    
                    Debugger.Break();                    
                }
#endif
            }

            using (var memoryStream = new MemoryStream(messageBody))
            {
                memoryStream.Position = 0;
                model = ContractSerializer.Deserialize<T>(memoryStream);
            }

            return model;
        }
    }
}
