﻿using System.Collections.Generic;
using Curse.Logging;
using Curse.SocketInterface;
using System;
using Newtonsoft.Json;

namespace Curse.SocketMessages
{
    public static class ContractDispatcher
    {
        private static readonly Dictionary<int, IContractDispatcher> _contractDispatchers = new Dictionary<int, IContractDispatcher>();

        public static void AddContractDispatcher<T>(Action<ISocketInterface, T> handler) where T : Contract<T>, new()
        {
            var messageType = Contract<T>.MessageType;
            IContractDispatcher dispatcher = new ContractDispatcher<T>(handler);            
            if(_contractDispatchers.ContainsKey(messageType))
            {
                Logger.Info("Replacing previously registered contract dispatcher for " + typeof(T).Name);
            }
            _contractDispatchers[messageType] =  dispatcher;
        }

        public static void TryDispatch(ISocketInterface socketInterface, Message message)
        {
            IContractDispatcher dispatcher = null;
            if (_contractDispatchers.TryGetValue(message.Header.MessageType, out dispatcher))
            {
                dispatcher.Dispatch(socketInterface, message);
            }
        }

        public static void TryDispatch(ISocketInterface socketInterface, string json)
        {
            JsonMessage message;
            try
            {
                message = JsonConvert.DeserializeObject<JsonMessage>(json);
            }
            catch (Exception ex)
            {
#if !CONFIG_RELEASE
                Logger.Warn(ex, "Failed to deserialize message!", new { RawJson = json });
#endif
                return;
            }


            try
            {
                IContractDispatcher dispatcher = null;
                if (_contractDispatchers.TryGetValue(message.TypeID, out dispatcher))
                {
                    dispatcher.Dispatch(socketInterface, json);
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to dispatch message!", new {RawJson = json});
            }
        }
    }

    public class ContractDispatcher<T> : IContractDispatcher where T : Contract<T>, new()
    {
        private readonly Action<ISocketInterface, T> _callback;
        private readonly string _typeName;

        public string TypeName
        {
            get
            {
                return _typeName;
            }
        }

        public ContractDispatcher(Action<ISocketInterface, T> callback)
        {
            _callback = callback;
            _typeName = typeof(T).Name;
        }

        public void Dispatch(ISocketInterface socketInterface, string jsonMessage)
        {
            var contract = JsonConvert.DeserializeObject<JsonMessage<T>>(jsonMessage);
            _callback(socketInterface, contract.Body);
        }

        public void Dispatch(ISocketInterface socketInterface, Message message)
        {
            T contract = null;

            try
            {
                if (Contract<T>.IsSerialized != message.Header.IsSerialized)
                {
                    Logger.Debug("Contract message header serialization value does not match the contract definition.");
                    return;
                }

                if (message.Header.IsSerialized)
                {
                    contract = Contract<T>.FromMessage(message, socketInterface.EncryptionAlgorithm);
                }
                else
                {
                    contract = new T();
                    contract.ParseMessage(message);
                }

                if (!contract.Validate())
                {
                    Logger.Debug("Failed to validate contract of type '" + _typeName + "'");
                    return;
                }
            }
            catch (ContractParserException ex)
            {
                Logger.Debug(ex, "Contract failed to parse from message of type '" + _typeName + "'");
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to create contract from message of type '" + _typeName + "'");
            }            
            if (contract != null)
            {
                _callback(socketInterface, contract);
            }
        }
    }
}
