﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using System.Security.Authentication;
using System.Threading;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;
using System.Security.Cryptography;
using Curse.SocketServer.Testing.Contracts;

namespace Curse.SocketServer.Testing
{
    public class TestSocketClient : ClientSocketInterface
    {
        public TestSocketClient(IPAddress address, int port) : base(address, port, ClientSocketOptions.TcpOnly)
        {
            AddContractDispatcher<SequenceTestContract>(Handler);
            AddContractDispatcher<TestJoinResponse>(OnJoin);
            AddContractDispatcher<TestEncryptedResponse>(OnEncryptedResponse);
            AddContractDispatcher<TestNormalResponse>(OnNormalResponse);
            MessageReceived += OnMessageReceived;
            
        }

        private void OnNormalResponse(BaseSocketInterface baseSocketInterface, TestNormalResponse testNormalResponse)
        {
            TestCoordinator.TrackClientMessage();
        }

        private void OnEncryptedResponse(BaseSocketInterface baseSocketInterface, TestEncryptedResponse testEncryptedResponse)
        {
            TestCoordinator.TrackClientMessage();            
        }

        private void Handler(BaseSocketInterface baseSocketInterface, SequenceTestContract sequenceTestContract)
        {
            TestCoordinator.TrackClientMessage();            
        }

        public void Connect()
        {
            Connect(100);
            var resp = Join();
            if (resp == null)
            {
                throw new Exception("Failed to join!");
            }
        }

        private readonly Dictionary<int, IContractDispatcher> _contractDispatchers = new Dictionary<int, IContractDispatcher>();

        private void AddContractDispatcher<T>(Action<BaseSocketInterface, T> handler) where T : Contract<T>, new()
        {
            var messageType = Contract<T>.MessageType;
            IContractDispatcher dispatcher = new ContractDispatcher<T>(handler);
            _contractDispatchers.Add(messageType, dispatcher);

        }

        private void OnMessageReceived(object sender, MessageEventArgs messageEventArgs)
        {
            if (!IsConnected)
            {
                return;
            }

            IContractDispatcher dispatcher = null;
            if (_contractDispatchers.TryGetValue(messageEventArgs.Message.Header.MessageType, out dispatcher))
            {
                try
                {
                    dispatcher.Dispatch(this, messageEventArgs.Message);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Dispatcher callback failed for message type: " + dispatcher.TypeName);
#if DEBUG
                    Debugger.Break();
#endif
                }
            }
            else
            {
                Debugger.Break();
            }
        }

        private ManualResetEvent _joinResetEvent;
        private TestJoinResponse _joinResponse;

        private void OnJoin(BaseSocketInterface baseSocketInterface, TestJoinResponse response)
        {
            _joinResponse = response;
           
            if (_joinResetEvent != null)
            {
                _joinResetEvent.Set();
            }
            
        }

        private TestJoinResponse Join()
        {
           
            try
            {
                _joinResponse = null;

                using (var rsa = CryptoHelper.NewRsaProvider())
                {
                    var encryptionAlgorithm = new AesCryptoServiceProvider();

                    var joinRequest = new TestJoinRequest
                    {                        
                        PublicKey = rsa.ExportPublicKey(),
                        CipherAlgorithm = CipherAlgorithmType.Aes,
                        CipherStrength = encryptionAlgorithm.BlockSize,
                    };

                    _joinResetEvent = new ManualResetEvent(false);                                        
                    SendContract(joinRequest);
                    _joinResetEvent.WaitOne(10000); // Wait up to 2 seconds for the join call to complete                                
                    _joinResetEvent.Dispose();
                    _joinResetEvent = null;

                    var resp = _joinResponse;
                    if (resp == null)
                    {
                        return null;
                    }
                    
                    CryptoHelper.SetSecret(encryptionAlgorithm, rsa, resp.EncryptedSessionKey);
                    Session = new ClientSession("Test", encryptionAlgorithm);
                    return resp;
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            return _joinResponse;
        }

    }
}
