﻿using System;
using System.Net;
using System.Net.Sockets;
using System.Security.Cryptography;
using Curse.SocketInterface;
using Curse.SocketMessages;
using Microsoft.ServiceModel.WebSockets;

namespace Curse.WebSocketInterface
{
    public class BaseWebSocketServiceInterface : ISocketInterface
    {
        private readonly WebSocketService _socket;
        private readonly int _clientID;
        private readonly DateTime _dateCreated;
        private readonly IPAddress _ipAddress;

        public bool IsConnected { get; set; }

        public bool IsHandshaken { get; set; }

        public int DateLastHandshake { get; set; }

        public int ClientID
        {
            get { return _clientID; }
        }

        public SymmetricAlgorithm EncryptionAlgorithm
        {
            get { return null; }
        }

        public ISocketSession Session { get; set; }

        public bool IsAuthenticated { get; set; }

        public DateTime DateCreated
        {
            get { return _dateCreated; }
        }

        public BaseWebSocketServiceInterface(WebSocketService socket, IPAddress ipAddress, int clientID)
        {
            _socket = socket;
            _clientID = clientID;
            _ipAddress = ipAddress;
            _dateCreated = DateTime.UtcNow;
        }

        public void SendContract(IContract contract)
        {
            var serializedMessage = contract.ToJsonMessage(ClientID);
            _socket.Send(serializedMessage);
        }

        public void SendString(string serializedMessage)
        {
            _socket.Send(serializedMessage);
        }

        public void RaiseDisconnect(SocketDisconnectReason reason)
        {
            _socket.Close();
        }

        public void RaiseDisconnect(SocketError error)
        {
            _socket.Close();
        }

        public void SendPacket(IContract contract)
        {
            throw new NotImplementedException();
        }

        public IPAddress RemoteAddress
        {
            get { return _ipAddress; }
        }

        public void ProcessIncomingMessage(string json)
        {
            // Try to create a message from this
            if (json.Length > Message.MaxMessageSize)
            {
                throw new Exception("More data transferred (" + json.Length + ") than permitted (" + Message.MaxMessageSize + ")");
            }

            ContractDispatcher.TryDispatch(this, json);
        }

        private static readonly TimeSpan _healthCheckInterval = TimeSpan.FromMinutes(1);
        private static readonly int _idleConnectionTimeout = (int)TimeSpan.FromMinutes(1).TotalMilliseconds;

        private DateTime _dateLastHealthCheck;

        public bool IsSocketHealthy()
        {
            _dateLastHealthCheck = DateTime.UtcNow;

            try
            {
                return Environment.TickCount - DateLastHandshake <= _idleConnectionTimeout;
            }
            catch
            {
                return false;
            }
        }

        public bool NeedsHealthCheck()
        {
            if (DateTime.UtcNow.Subtract(_dateCreated) < TimeSpan.FromMinutes(1))
            {
                return false;
            }

            return DateTime.UtcNow.Subtract(_dateLastHealthCheck) >= _healthCheckInterval;
        }
    }
}
