﻿using System;
using System.Configuration;
using System.Data.SqlClient;
using System.Linq;
using System.Net;
using System.ServiceModel;
using System.ServiceModel.Activation;
using Curse.AchievementService.Helpers;
using Curse.AchievementService.Models;
using Curse.Caching;
using Curse.Extensions;
using Curse.Logging;
using Curse.Logging.Uploader;

namespace Curse.AchievementService
{
    [ServiceBehavior(Name = "AchievementService",
        Namespace = "http://achievement-service.curse.us/",
        AddressFilterMode = AddressFilterMode.Any,
        ConcurrencyMode = ConcurrencyMode.Multiple,
        InstanceContextMode = InstanceContextMode.PerSession)]
    [AspNetCompatibilityRequirements(RequirementsMode = AspNetCompatibilityRequirementsMode.Allowed)]
    public class AchievementService : IAchievementService
    {
        private static readonly object StartLock = new object();

        static AchievementService()
        {
            lock (StartLock)
            {

                #region Init Logger

                Logger.Init(ConfigurationManager.AppSettings["LogPath"], "AchievementService");

                LogUploader.Initialize(14, ConfigurationManager.AppSettings["LogServiceUrl"], ConfigurationManager.AppSettings["LogServiceApiKey"]);

                Logger.Info("Achievement Service Starting...");

                #endregion

                #region Init Cache Cluster

                using (var conn = DatabaseUtility.GetConnection(false))
                {
                    try
                    {
                        Logger.Info("Cluster Manager Initializing.");
                        ClusterManager.Instance.Initialize("AchievementService", conn, true);

                        Logger.Info("Cache Cluster Initializing.");
                        CacheCluster.Instance.Start();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error("Unable to start cache cluster: " + ex.Message);
                        throw;
                    }
                }

                Logger.Info("Achievement Service Started.");

                #endregion
            }
        }

        #region Achievements

        public Achievement[] GetAllAchievements()
        {
            try
            {
                return Achievement.GetAll();
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of achievements. " + ex.Message);
                throw ApiError.Create(1001, "There was an error retrieving the list of achievements. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public Achievement GetAchievementByID(string achievementID)
        {
            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(1005, "You must supply a value for achievementID.");
            }

            int newAchievementID;
            int.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(1005, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return Achievement.GetAchievementByID(newAchievementID);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the achievement. " + ex.Message);
                throw ApiError.Create(1002, "There was an error retrieving the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public Achievement[] GetAchievementsBySiteID(string siteID, string subSiteKey = null)
        {
            if (siteID.IsNullOrEmpty())
            {
                throw ApiError.Create(1006, "You must supply a value for siteID.");
            }

            int newSiteID;
            int.TryParse(siteID, out newSiteID);

            if (newSiteID <= 0)
            {
                throw ApiError.Create(1006, "You must supply a value for siteID that is greater than 0.");
            }

            try
            {
                return Achievement.GetAchievementsBySiteID(newSiteID, subSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of achievements. " + ex.Message);
                throw ApiError.Create(1003, "There was an error retrieving the list of achievements. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public Achievement AddAchievement(string name, string description, string imagePath, string rules = null, string requires = null)
        {
            if (name.IsNullOrEmpty())
            {
                throw ApiError.Create(1007, "You must supply a value for name.");
            }

            if (description.IsNullOrEmpty())
            {
                throw ApiError.Create(1008, "You must supply a value for description.");
            }

            if (imagePath.IsNullOrEmpty())
            {
                throw ApiError.Create(1009, "You must supply a value for imagePath.");
            }

            try
            {
                return Achievement.AddAchievement(name, description, imagePath, rules, requires);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error adding the achievement. " + ex.Message);
                throw ApiError.Create(1004, "There was an error adding the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public ApiReturn DeleteAchievement(string achievementID)
        {
            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(1010, "You must supply a value for achievementID.");
            }

            long newAchievementID;
            long.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(1010, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return Achievement.DeleteAchievement(newAchievementID)
                    ? new ApiReturn(true, "Sucessfully deleted achievement")
                    : new ApiReturn(false, "Could not delete achievement");
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error removing the achievement. " + ex.Message);
                throw ApiError.Create(1005, "There was an error removing the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public ApiReturn UpdateAchievement(string achievementID, string status, string name = null, string description = null, string imagePath = null, string rules = null, string requires = null)
        {
            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(1011, "You must supply a value for achievementID.");
            }

            long newAchievementID;
            long.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(1011, "You must supply a value for achievementID that is greater than 0.");
            }

            if (status.IsNullOrEmpty())
            {
                throw ApiError.Create(1012, "You must supply a value for status that is greater than 0.");
            }

            int newStatus;
            int.TryParse(status, out newStatus);

            if (newStatus <= 0)
            {
                throw ApiError.Create(1012, "You must supply a value for status that is greater than 0.");
            }

            try
            {
                return Achievement.UpdateAchievement(newAchievementID, newStatus, name, description, imagePath, rules, requires)
                    ? new ApiReturn(true, "Sucessfully updated achievement")
                    : new ApiReturn(false, "Could not update achievement");
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error updating the achievement. " + ex.Message);
                throw ApiError.Create(1006, "There was an error updating the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        #endregion

        #region Auth Sites

        public AuthSite[] GetAllSites()
        {
            try
            {
                return AuthSite.AuthSites;
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of sites. " + ex.Message);
                throw ApiError.Create(2001, "There was an error retrieving the list of sites. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public AuthSite[] GetSitesBySiteID(string externalID, string subSiteKey = null)
        {
            if (externalID.IsNullOrEmpty())
            {
                throw ApiError.Create(2006, "You must supply a value for externalID.");
            }

            int newExternalID;
            int.TryParse(externalID, out newExternalID);

            if (newExternalID <= 0)
            {
                throw ApiError.Create(2006, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return AuthSite.GetAuthSitesByID(newExternalID, subSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the site. " + ex.Message);
                throw ApiError.Create(2002, "There was an error retrieving the site. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public AuthSite[] GetSitesByAchievementID(string achievementID)
        {

            int newAchievementID;
            int.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(2008, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return AuthSite.GetAuthSitesByAchievementID(newAchievementID);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of sites. " + ex.Message);
                throw ApiError.Create(2003, "There was an error retrieving the list of sites. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public AuthSite AddSite(string externalID, string hostName, string subSiteKey = null)
        {
            if (hostName.IsNullOrEmpty())
            {
                throw ApiError.Create(2005, "You must supply a value for hostName.");
            }

            if (externalID.IsNullOrEmpty())
            {
                throw ApiError.Create(2006, "You must supply a value for externalID.");
            }

            int newExternalID;
            int.TryParse(externalID, out newExternalID);

            if (newExternalID <= 0)
            {
                throw ApiError.Create(2006, "You must supply a value for achievementID that is greater than 0.");
            }

            var existingSite = GetSitesBySiteID(externalID, subSiteKey);
            if (existingSite.Any())
            {
                throw ApiError.Create(2007, "A record already exists for externalID: " + externalID + " and subSiteKey: " + subSiteKey);
            }

            try
            {
                return AuthSite.AddAuthSite(newExternalID, hostName, subSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error adding the site. " + ex.Message);
                throw ApiError.Create(2004, "There was an error adding the site. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        #endregion

        #region Earned Achievements

        public EarnedAchievement[] GetAllAchievementsByUserID(string userID)
        {
            if (userID.IsNullOrEmpty())
            {
                throw ApiError.Create(3006, "You must supply a value for userID.");
            }

            int newUserID;
            int.TryParse(userID, out newUserID);

            if (newUserID <= 0)
            {
                throw ApiError.Create(3006, "You must supply a value for userID that is greater than 0.");
            }

            try
            {
                return EarnedAchievement.GetAllEarnedAchievementsByUserID(newUserID);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of earned achievements. " + ex.Message);
                throw ApiError.Create(3001, "There was an error retrieving the list of earned achievements. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public EarnedAchievement[] GetSiteAchievementsByUserID(string userID, string siteID, string subSiteKey = null)
        {
            if (userID.IsNullOrEmpty())
            {
                throw ApiError.Create(3007, "You must supply a value for userID.");
            }

            int newUserID;
            int.TryParse(userID, out newUserID);

            if (newUserID <= 0)
            {
                throw ApiError.Create(3007, "You must supply a value for userID that is greater than 0.");
            }

            if (siteID.IsNullOrEmpty())
            {
                throw ApiError.Create(3008, "You must supply a value for userID.");
            }

            int newSiteID;
            int.TryParse(siteID, out newSiteID);

            if (newSiteID <= 0)
            {
                throw ApiError.Create(3008, "You must supply a value for siteID that is greater than 0.");
            }

            try
            {
                return EarnedAchievement.GetSiteEarnedAchievementsByUserID(newUserID, newSiteID, subSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error retrieving the list of earned achievements. " + ex.Message);
                throw ApiError.Create(3002, "There was an error retrieving the list of earned achievements. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public EarnedAchievement AwardAchievement(string userID, string achievementID, string siteID, string subSiteKey = null)
        {
            if (userID.IsNullOrEmpty())
            {
                throw ApiError.Create(3007, "You must supply a value for userID.");
            }

            int newUserID;
            int.TryParse(userID, out newUserID);

            if (newUserID <= 0)
            {
                throw ApiError.Create(3007, "You must supply a value for userID that is greater than 0.");
            }

            if (siteID.IsNullOrEmpty())
            {
                throw ApiError.Create(3008, "You must supply a value for siteID.");
            }

            int newSiteID;
            int.TryParse(siteID, out newSiteID);

            if (newSiteID <= 0)
            {
                throw ApiError.Create(3008, "You must supply a value for siteID that is greater than 0.");
            }

            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(3009, "You must supply a value for achievementID.");
            }

            int newAchievementID;
            int.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(3009, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return EarnedAchievement.AwardUserAchievement(newUserID, newAchievementID, newSiteID, subSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error awarding the achievement. " + ex.Message);
                throw ApiError.Create(3004, "There was an error awarding the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public ApiReturn UnawardAchievement(string earnedAchievementID)
        {
            if (earnedAchievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(3012, "You must supply a value for earnedAchievementID.");
            }

            int newEarnedAchievementID;
            int.TryParse(earnedAchievementID, out newEarnedAchievementID);

            if (newEarnedAchievementID <= 0)
            {
                throw ApiError.Create(3012, "You must supply a value for earnedAchievementID that is greater than 0.");
            }

            try
            {
                return EarnedAchievement.UnawardUserAchievement(newEarnedAchievementID)
                    ? new ApiReturn(true, "Sucessfully unawarded achievement")
                    : new ApiReturn(false, "Could not unaward achievement");
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error unawarding the achievement. " + ex.Message);
                throw ApiError.Create(3005, "There was an error unawarding the achievement. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        #endregion

        #region Site Achievement Maps

        public ApiReturn AddSiteAchievementMap(string siteID, string achievementID)
        {
            if (siteID.IsNullOrEmpty())
            {
                throw ApiError.Create(4003, "You must supply a value for siteID.");
            }

            int newSiteID;
            int.TryParse(siteID, out newSiteID);

            if (newSiteID <= 0)
            {
                throw ApiError.Create(4003, "You must supply a value for siteID that is greater than 0.");
            }

            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(4004, "You must supply a value for achievementID.");
            }

            int newAchievementID;
            int.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(4004, "You must supply a value for achievementID that is greater than 0.");
            }

            var authSite = AuthSite.GetByID(newSiteID);
            if (authSite == null)
            {
                throw ApiError.Create(4005, "There is no site record with a siteID of " + siteID);
            }

            var achievement = Achievement.GetAchievementByID(newAchievementID);
            if (achievement == null)
            {
                throw ApiError.Create(4006, "There is no achievement with an achievementID of " + achievementID);
            }

            try
            {
                return Achievement.AddSiteAchievementMap(newSiteID, newAchievementID)
                    ? new ApiReturn(true, "Sucessfully added site achievement map")
                    : new ApiReturn(false, "Could not add site achievement map");
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error adding the site achievement map. " + ex.Message);
                throw ApiError.Create(4001, "There was an error adding the site achievement map. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        public ApiReturn RemoveSiteAchievementMap(string siteID, string achievementID)
        {
            if (siteID.IsNullOrEmpty())
            {
                throw ApiError.Create(4003, "You must supply a value for userID.");
            }

            int newSiteID;
            int.TryParse(siteID, out newSiteID);

            if (newSiteID <= 0)
            {
                throw ApiError.Create(4003, "You must supply a value for siteID that is greater than 0.");
            }

            if (achievementID.IsNullOrEmpty())
            {
                throw ApiError.Create(4004, "You must supply a value for achievementID.");
            }

            int newAchievementID;
            int.TryParse(achievementID, out newAchievementID);

            if (newAchievementID <= 0)
            {
                throw ApiError.Create(4004, "You must supply a value for achievementID that is greater than 0.");
            }

            try
            {
                return Achievement.RemoveSiteAchievementMap(newSiteID, newAchievementID)
                    ? new ApiReturn(true, "Sucessfully removed site achievement map")
                    : new ApiReturn(false, "Could not remove site achievement map");
            }
            catch (Exception ex)
            {
                Logger.Error("There was an error removing the site achievement map. " + ex.Message);
                throw ApiError.Create(4002, "There was an error removing the site achievement map. " + ex.Message, HttpStatusCode.InternalServerError);
            }
        }

        #endregion

        public string HealthCheck()
        {
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["RoamingDBAchievement"].ConnectionString))
            {
                conn.Open();
            }

            return "Success";
        }

        public string Index()
        {
            return "Achivement Service";
        }
    }
}