﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Runtime.Serialization;
using System.Web.Caching;
using Curse.AchievementService.Helpers;
using Curse.Caching;

namespace Curse.AchievementService.Models
{
    public enum AchievementStatus
    {
        Active = 1,
        Inactive,
        Expired,
    }

    [DataContract]
    public class Achievement
    {
        private const string AchievementCacheKey = "AchievementByID:{0}";
        private const string AchievementsBySiteCacheKey = "AchievementsBySiteID:{0}";

        #region DataMembers

        [DataMember(Name = "id")]
        public double ID { get; private set; }

        [DataMember(Name = "name")]
        public string Name { get; set; }

        [DataMember(Name = "description")]
        public string Description { get; set; }

        [DataMember(Name = "rules")]
        public string Rules { get; set; }

        [DataMember(Name = "requires")]
        public string Requires { get; set; }

        [DataMember(Name = "imagePath")]
        public string ImagePath { get; set; }

        [DataMember(Name = "status")]
        public AchievementStatus Status { get; set; }

        [DataMember(Name = "statusMessage")]
        public string StatusMessage
        {
            get { return Status.ToString(); }
            set { }
        }

        [DataMember(Name = "siteID")]
        public int? SiteID { get; set; }

        [DataMember(Name = "subSiteKey")]
        public string SubSiteKey { get; set; }

        #endregion

        #region Model Materialzation

        private Achievement(SqlDataReader reader)
        {
            ID = reader.GetInt64(reader.GetOrdinal("ID"));
            Name = reader.GetString(reader.GetOrdinal("Name"));
            Description = reader.GetString(reader.GetOrdinal("Description"));
            Rules = reader.IsDBNull(reader.GetOrdinal("Rules")) ? String.Empty : reader.GetString(reader.GetOrdinal("Rules"));
            Requires = reader.IsDBNull(reader.GetOrdinal("Requires")) ? String.Empty : reader.GetString(reader.GetOrdinal("Requires"));
            ImagePath = reader.GetString(reader.GetOrdinal("Imagepath"));
            Status = (AchievementStatus)reader.GetByte(reader.GetOrdinal("Status"));
            
            if(!reader.IsDBNull(reader.GetOrdinal("SiteID")))
            {
                SiteID = reader.GetInt32(reader.GetOrdinal("SiteID"));
            }

            if (!reader.IsDBNull(reader.GetOrdinal("SubSiteKey")))
            {
                SubSiteKey = reader.GetString(reader.GetOrdinal("SubSiteKey"));    
            }
        }

        #endregion

        private static Achievement[] Achievements
        {
            get { return CacheManager.GetOrAdd("Achievements", GetAllAchievements, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable); }
        }

        private static Achievement[] GetAllAchievements()
        {
            var models = new List<Achievement>();

            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spGetAchievements";
                    command.CommandType = CommandType.StoredProcedure;

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            models.Add(new Achievement(reader));
                        }
                    }
                }
            }

            return models.ToArray();
        }

        public static Achievement[] GetAll()
        {
            return Achievements;
        }

        public static Achievement GetAchievementByID(long id)
        {
            var cacheKey = String.Format(AchievementCacheKey, id);

            return CacheManager.GetOrAdd(cacheKey, () =>
            {
                using (var conn = DatabaseUtility.GetConnection())
                {
                    return GetByID(id, conn);
                }
            }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);
        }

        private static Achievement GetByID(long id, SqlConnection conn)
        {
            Achievement achievement = null;

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "spGetAchievementByID";
                command.CommandType = CommandType.StoredProcedure;

                command.Parameters.AddWithValue("@ID", id);
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        achievement = new Achievement(reader);
                    }
                }
            }

            return achievement;
        }

        public static Achievement[] GetAchievementsBySiteID(int siteID, string subSiteKey = null)
        {
            var cacheKey = String.Format(AchievementsBySiteCacheKey, siteID);

            var achievements = CacheManager.GetOrAdd(cacheKey, () =>
            {
                using (var conn = DatabaseUtility.GetConnection())
                {
                    return GetBySiteID(siteID, conn, subSiteKey);
                }
            }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);

            if (subSiteKey != null)
            {
                achievements = achievements.Where(p => p.SubSiteKey == subSiteKey).ToArray();
            }

            return achievements;
        }

        private static Achievement[] GetBySiteID(int siteID, SqlConnection conn, string subSiteKey = null)
        {
            var achievements = new List<Achievement>();

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "spGetAchievementsBySiteID";
                command.CommandType = CommandType.StoredProcedure;

                command.Parameters.AddWithValue("@SiteID", siteID);
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        achievements.Add(new Achievement(reader));
                    }
                }
            }

            return achievements.ToArray();
        }

        public static Achievement AddAchievement(string name, string description, string imagePath, string rules = null, string requires = null)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                long newAchievementID = 0;

                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spInsertAchievement";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@Name", name);
                    command.Parameters.AddWithValue("@Description", description);
                    command.Parameters.AddWithValue("@ImagePath", imagePath);
                    command.Parameters.AddWithValue("@Rules", (rules ?? ""));
                    command.Parameters.AddWithValue("@Requires", (requires ?? ""));
                    command.Parameters.AddWithValue("@Status", AchievementStatus.Active);
                    command.Parameters.AddWithValue("@DateCreated", DateTime.UtcNow);
                    command.Parameters.AddWithValue("@DateModified", DateTime.UtcNow);

                    newAchievementID = (long)command.ExecuteScalar();
                }

                ExpireAllAchievementCache();

                return Achievement.GetByID(newAchievementID, conn);
            }
        }

        public static bool DeleteAchievement(long achievementID)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spDeleteAchievement";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@ID", achievementID);
                    command.Parameters.AddWithValue("@Status", AchievementStatus.Inactive);
                    command.Parameters.AddWithValue("@DateModified", DateTime.UtcNow);

                    var rows = command.ExecuteNonQuery();

                    if (rows.Equals(1))
                    {
                        var a = GetByID(achievementID, conn);

                        ExpireAllAchievementCache();
                        a.ExpireAchievementByID();
                        a.ExpireAchievementsBySiteID();
                        
                        return true;
                    }
                    return false;
                }
            }
        }

        public static bool UpdateAchievement(long achievementID, int status, string name = null, string description = null, string imagePath = null, string rules = null, string requires = null)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spUpdateAchievement";
                    command.CommandType = CommandType.StoredProcedure;

                    //required
                    command.Parameters.AddWithValue("@ID", achievementID);
                    command.Parameters.AddWithValue("@Status", (AchievementStatus)status);
                    command.Parameters.AddWithValue("@Datemodified", DateTime.UtcNow);

                    //optional
                    command.Parameters.AddWithValue("@Name", name);
                    command.Parameters.AddWithValue("@Description", description);
                    command.Parameters.AddWithValue("@ImagePath", imagePath);
                    command.Parameters.AddWithValue("@Rules", rules);
                    command.Parameters.AddWithValue("@Requires", requires);

                    var rows = command.ExecuteNonQuery();

                    if (rows.Equals(1))
                    {
                        var a = GetByID(achievementID, conn);

                        ExpireAllAchievementCache();
                        a.ExpireAchievementByID();
                        a.ExpireAchievementsBySiteID();

                        return true;
                    }

                    return false;
                }
            }
        }

        public static bool AddSiteAchievementMap(int siteID, int achievementID)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                int rows = 0;

                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spInsertSiteAchievementMap";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@SiteID", siteID);
                    command.Parameters.AddWithValue("@AchievementID", achievementID);

                    rows = command.ExecuteNonQuery();
                }

                if (rows.Equals(1))
                {
                    ExpireAllAchievementCache();
                    ExpireAchievementsBySiteID(siteID);
                    AuthSite.ExpireSitesByAchievementID(achievementID);
                    var a = GetByID(achievementID, conn);
                    a.ExpireAchievementByID();

                    return true;
                }

                return false;
            }
        }

        public static bool RemoveSiteAchievementMap(int siteID, int achievementID)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                int rows = 0;

                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spRemoveSiteAchievementMap";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@SiteID", siteID);
                    command.Parameters.AddWithValue("@AchievementID", achievementID);

                    rows = command.ExecuteNonQuery();
                }

                if (rows.Equals(1))
                {
                    ExpireAllAchievementCache();
                    ExpireAchievementsBySiteID(siteID);
                    AuthSite.ExpireSitesByAchievementID(achievementID);
                    var a = GetByID(achievementID, conn);
                    a.ExpireAchievementByID();

                    return true;
                }

                return false;
            }
        }

        #region Cache Expirations

        private static void ExpireAllAchievementCache()
        {
            CacheManager.Expire("Achievements");
        }

        private void ExpireAchievementByID()
        {
            CacheManager.Expire(String.Format(AchievementCacheKey, ID));
        }

        private void ExpireAchievementsBySiteID()
        {
            CacheManager.Expire(String.Format(AchievementsBySiteCacheKey, SiteID));
        }

        private static void ExpireAchievementsBySiteID(int siteID)
        {
            CacheManager.Expire(String.Format(AchievementsBySiteCacheKey, siteID));
        }

        #endregion
    }
}