﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Runtime.Serialization;
using System.Web.Caching;
using Curse.AchievementService.Helpers;
using Curse.Caching;

namespace Curse.AchievementService.Models
{
    [DataContract]
    public class AuthSite
    {
        private const string SitesByAchievementCacheKey = "SitesByAchievementsID:{0}";

        #region Data Members

        [DataMember(Name = "externalID")]
        public int ExternalID { get; set; }

        [DataMember(Name = "hostName")]
        public string HostName { get; set; }

        [DataMember(Name = "id")]
        public int ID { get; set; }

        [DataMember(Name = "subSiteKey")]
        public string SubSiteKey { get; set; }

        #endregion

        #region Model Materialization

        private AuthSite(SqlDataReader reader)
        {
            ID = reader.GetInt32(reader.GetOrdinal("ID"));
            ExternalID = reader.GetInt32(reader.GetOrdinal("ExternalID"));
            SubSiteKey = reader.IsDBNull(reader.GetOrdinal("SubSiteKey")) ? String.Empty : reader.GetString(reader.GetOrdinal("SubSiteKey"));
            HostName = reader.GetString(reader.GetOrdinal("HostName"));
        }

        #endregion

        public static AuthSite AddAuthSite(int externalID, string hostName, string subSiteKey = null)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = @"spInsertAuthSite";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@ExternalID", externalID);
                    command.Parameters.AddWithValue("@SubSiteKey", subSiteKey ?? "");
                    command.Parameters.AddWithValue("@HostName", hostName);
                    command.Parameters.AddWithValue("@DateCreated", DateTime.UtcNow);
                    command.Parameters.AddWithValue("@DateModified", DateTime.UtcNow);

                    int authSiteID;
                    Int32.TryParse(command.ExecuteScalar().ToString(), out authSiteID);

                    ExpireAll();

                    return GetByID(authSiteID);
                }
            }
        }

        public static AuthSite[] AuthSites
        {
            get { return CacheManager.GetOrAdd("AuthSites", GetAuthSites, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable); }
        }

        private static AuthSite[] GetAuthSites()
        {
            var sites = new List<AuthSite>();

            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spGetAuthSites";
                    command.CommandType = CommandType.StoredProcedure;

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            sites.Add(new AuthSite(reader));
                        }
                    }
                }
            }

            return sites.ToArray();
        }

        public static AuthSite[] GetAuthSitesByID(int externalID, string subSiteKey = null)
        {
            var sites = AuthSites.Where(p => p.ExternalID == externalID);

            if (subSiteKey != null)
            {
                sites = sites.Where(p => p.SubSiteKey == subSiteKey);
            }

            return sites.ToArray();
        }

        public static AuthSite[] GetAuthSitesByAchievementID(int achievementID)
        {
            var cacheKey = String.Format(SitesByAchievementCacheKey, achievementID);

            var authSites = CacheManager.GetOrAdd(cacheKey, () =>
            {
                using (var conn = DatabaseUtility.GetConnection())
                {
                    return GetByAchievementID(achievementID, conn);
                }
            }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);

            return authSites;
        }

        private static AuthSite[] GetByAchievementID(int achievementID, SqlConnection conn)
        {
            var authSites = new List<AuthSite>();

            using (var command = conn.CreateCommand())
            {
                command.CommandText = @"spGetAuthSitesByAchievementID";
                command.CommandType = CommandType.StoredProcedure;

                command.Parameters.AddWithValue("@AchievementID", achievementID);
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        authSites.Add(new AuthSite(reader));
                    }
                }
            }

            return authSites.ToArray();
        }

        public static AuthSite GetByID(int id)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = @"spGetAuthSiteByID";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@ID", id);

                    using (var reader = command.ExecuteReader())
                    {
                        return reader.Read() ? new AuthSite(reader) : null;
                    }
                }
            }
        }

        #region Cache Expiration Methods

        private static void ExpireAll()
        {
            CacheManager.Expire("AuthSites");
        }

        public static void ExpireSitesByAchievementID(int achievementID)
        {
            CacheManager.Expire(String.Format(SitesByAchievementCacheKey, achievementID));
        }

        #endregion
    }
}