﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Runtime.Serialization;
using System.Web.Caching;
using Curse.AchievementService.Helpers;
using Curse.Caching;
using Curse.Extensions;

namespace Curse.AchievementService.Models
{
    public enum EarnedAchievementStatus
    {
        Awarded = 1,
        Unawarded,
        Expired
    }

    [DataContract]
    public class EarnedAchievement
    {
        private const string EarnedAchievementsAllByUserCacheKey = "EarnedAchievementsByUserID:{0}";

        #region DataMembers

        [DataMember(Name="id")]
        public long ID { get; private set; }

        [DataMember(Name="curseID")]
        public int CurseID { get; set; }

        [DataMember(Name="achievementID")]
        public int AchievementID { get; set; }

        [DataMember(Name="authSiteID")]
        public int AuthSiteID { get; set; }

        [DataMember(Name="dateCreated")]
        public DateTime DateCreated { get; set; }

        [DataMember(Name="dateModified")]
        public DateTime DateModified { get; set; }

        [DataMember(Name="status")]
        public EarnedAchievementStatus Status { get; set; }

        [DataMember(Name="statusMessage")]
        public string StatusMessage
        {
            get { return Status.ToString(); }
            set { }
        }

        [DataMember(Name="name")]
        public string Name { get; set; }

        [DataMember(Name="description")]
        public string Description { get; set; }

        [DataMember(Name="rules")]
        public string Rules { get; set; }

        [DataMember(Name="imagePath")]
        public string ImagePath { get; set; }

        [DataMember(Name="hostName")]
        public string HostName { get; set; }

        [DataMember(Name="subSiteKey")]
        public string SubSiteKey { get; set; }

        #endregion

        #region Model Materialzation

        private EarnedAchievement(SqlDataReader reader)
        {
            ID = reader.GetInt64(reader.GetOrdinal("ID"));
            CurseID = reader.GetInt32(reader.GetOrdinal("CurseID"));
            AchievementID = reader.GetInt32(reader.GetOrdinal("AchievementID"));
            AuthSiteID = reader.GetInt32(reader.GetOrdinal("AuthSiteID"));
            DateCreated = reader.GetDateTime(reader.GetOrdinal("DateCreated"));
            DateModified = reader.GetDateTime(reader.GetOrdinal("DateModified"));
            Status = (EarnedAchievementStatus)reader.GetByte(reader.GetOrdinal("Status"));
            Name = reader.GetString(reader.GetOrdinal("Name"));
            Description = reader.GetString(reader.GetOrdinal("Description"));
            Rules = reader.GetString(reader.GetOrdinal("Rules"));
            ImagePath = reader.GetString(reader.GetOrdinal("Imagepath"));
            HostName = reader.GetString(reader.GetOrdinal("HostName"));
            SubSiteKey = reader.GetString(reader.GetOrdinal("SubSiteKey"));
        }

        #endregion

        public static EarnedAchievement[] GetAllEarnedAchievementsByUserID(int userID)
        {
            var cacheKey = String.Format(EarnedAchievementsAllByUserCacheKey, userID);

            var achievements = CacheManager.GetOrAdd(cacheKey, () =>
            {
                using (var conn = DatabaseUtility.GetConnection())
                {
                    return GetByUserID(userID, conn);
                }
            }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);

            return achievements;
        }

        private static EarnedAchievement[] GetByUserID(int userID, SqlConnection conn)
        {
            var achievements = new List<EarnedAchievement>();

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "spGetAchievementsByUserID";
                command.CommandType = CommandType.StoredProcedure;

                command.Parameters.AddWithValue("@UserID", userID);

                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        achievements.Add(new EarnedAchievement(reader));
                    }
                }
            }

            return achievements.ToArray();
        }

        public static EarnedAchievement[] GetSiteEarnedAchievementsByUserID(int userID, int siteID, string subSiteKey = null)
        {
            var achievements = GetAllEarnedAchievementsByUserID(userID);
            achievements = achievements.Where(p => p.AuthSiteID == siteID).ToArray();
            if (!subSiteKey.IsNullOrEmpty())
            {
                achievements = achievements.Where(p => p.SubSiteKey == subSiteKey).ToArray();
            }

            return achievements;
        }

        public static EarnedAchievement AwardUserAchievement(int curseID, long achievementID, int siteID, string subSiteKey = null)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spAwardAchievement";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@UserID", curseID);
                    command.Parameters.AddWithValue("@AchievementID", achievementID);
                    command.Parameters.AddWithValue("@ExternalID", siteID);
                    command.Parameters.AddWithValue("@DateCreated", DateTime.Now);
                    command.Parameters.AddWithValue("@DateModified", DateTime.Now);
                    command.Parameters.AddWithValue("@SubSiteKey", (subSiteKey ?? ""));

                    var earnedID = Convert.ToInt32(command.ExecuteScalar());

                    var earnedAchievement = GetEarnedAchievementByID(earnedID);

                    ExpireEarnedAchievementsByUserID(curseID);

                    return earnedAchievement;
                }
            }
        }

        public static bool UnawardUserAchievement(int earnedAchievementID)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spUnawardAchievement";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@ID", earnedAchievementID);
                    command.Parameters.AddWithValue("@Status", EarnedAchievementStatus.Unawarded);
                    command.Parameters.AddWithValue("@DateModified", DateTime.Now);

                    command.ExecuteNonQuery();
                }
            }

            var e = GetEarnedAchievementByID(earnedAchievementID);
            ExpireEarnedAchievementsByUserID(e.CurseID);

            return true;
        }

        public static EarnedAchievement GetEarnedAchievementByID(int earnedID)
        {
            using (var conn = DatabaseUtility.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "spGetEarnedAchievementByID";
                    command.CommandType = CommandType.StoredProcedure;

                    command.Parameters.AddWithValue("@EarnedID", earnedID);

                    using (var reader = command.ExecuteReader())
                    {
                        return reader.Read() ? new EarnedAchievement(reader) : null;
                    }
                }
            }
        }

        private static void ExpireEarnedAchievementsByUserID(int userID)
        {
            CacheManager.Expire(String.Format(EarnedAchievementsAllByUserCacheKey, userID));
        }
    }
}