﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Configuration;
using System.Linq;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Data.SqlClient;
using System.Text.RegularExpressions;
using System.Xml;
using System.IO;
using Curse;
using AddOnService;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;

namespace AddOnService
{
    public class AddOn
    {        
        // Statics
        private static string sFeedBaseUrl = ConfigurationManager.AppSettings["FeedBaseUrl"];        
        private static string sDownloadBaseUrl = ConfigurationManager.AppSettings["DownloadBaseUrl"];
        private static string sCategoryBaseUrl = ConfigurationManager.AppSettings["CategoryBaseUrl"];
        private static string sAuthorBaseUrl = ConfigurationManager.AppSettings["AuthorBaseUrl"];
        private static Regex sHtmlStripper = new Regex("<!--.*?-->|<[^>]*>", RegexOptions.Compiled);


        // Privates
        private int id;
        private string name;
        private string url;
        private string feedUrl;
        private string gameName;
        private string portalName;
        private string sectionName;
        private string downloadName;
        private int gameId;
        private string description;
        private string summary;
        private int defaultFileId;        
        private int numComments;
        private double numDownloads;
        private int rating;
        private DateTime dateModified;
        private DateTime dateCreated;
        private DateTime dateReleased;
        private List<KeyValuePair<string,string>> categories;
        private List<AddOnAuthor> authors;
        private Dictionary<int,AddOnFile> files;
        private List<AddOnFileKey> fileKeys;        
        private List<AddOnLocalization> localizations;
        private List<AddOnFile> mLatestFiles = new List<AddOnFile>();
        private string projectElementXml;
        private Byte[] projectElementBytes;
        private Byte[] projectListBytes;
        private bool isAvailable = false;
        public EProjectStatus status = EProjectStatus.Normal;
        private string donationLink = null;


        public Dictionary<int, AddOnFile> Files
        {
            get
            {
                return files;
            }
        }

        public List<AddOnFile> LatestFiles
        {
            get
            {
                return mLatestFiles;
            }
        }

        // Properties
        public int Id
        {
            get
            {
                return id;
            }
        }

        public int DefaultFileId
        {
            get
            {
                return defaultFileId;
            }
        }

        public int GameId
        {
            get
            {
                return gameId;
            }
        }

        public string Downloads
        {
            get
            {
                return numDownloads.ToString();
            }
        }


        public DateTime LastUpdated
        {
            get
            {
                return dateModified;
            }
        }

        public long EpochLastRelease
        {
            get
            {
                return Utility.GetEpochDate(dateReleased);
            }
        }

        public string DefaultFileName
        {
            get
            {
                return files[defaultFileId].Name;
            }
        }

        public DateTime LastRelease
        {
            get
            {
                return dateReleased;
            }
        }

        public string Name
        {
            get
            {
                return name;
            }
        }

        public string Url
        {
            get
            {
                return url;
            }
        }

        public string FeedUrl
        {
            get
            {
                return feedUrl;
            }
        }

        public string PrimaryAuthor
        {
            get
            {
                if (authors.Count > 0)
                {
                    return authors[0].Name;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        public bool Available
        {
            get
            {
                return isAvailable;
            }
            set
            {
                isAvailable = value;
            }
        }

        public bool IncludeInIndex
        {
            get
            {
                return status != EProjectStatus.Deleted;
            }
        }

        public string DonationLink
        {
            get
            {
                return donationLink;
            }
        }

        public void WriteSummary(XmlWriter writer)
        {

            writer.WriteStartElement("localizations");
            foreach (AddOnLocalization i in localizations)
                i.serialize(writer);
            writer.WriteEndElement();
            
        }

        public void WriteKeys(XmlWriter writer)
        {

            writer.WriteStartElement("keys");
            foreach (AddOnFileKey i in fileKeys)
                i.serialize(writer);
            writer.WriteEndElement();
        }       

        public int FileKeyCount
        {
            get
            {
                return fileKeys.Count;
            }
        }
        public string ProjectElementXml
        {
            get
            {
                return projectElementXml;
            }
        }

        public void WriteElement(Stream stream)
        {
            stream.Write(projectElementBytes, 0, projectElementBytes.Length);
        }

        public void WriteListElement(MemoryStream stream)
        {
            stream.Write(projectListBytes, 0, projectListBytes.Length);
        }

        public static string GetFeedUrl(SqlDataReader pReader)
        {
            return String.Format(sFeedBaseUrl, pReader.GetInt32(pReader.GetOrdinal("addon_id")));
        }
      
        public AddOn(SqlConnection pConn,
            SqlDataReader pReader,
            Dictionary<int, List<IndividualFileFingerprint>> pIndividualFingerprints,
            Dictionary<int, List<AddOnFile>> pFileArchiveCache,
            Dictionary<int, List<AddOnFileDependency>> pFileDependencyCache)
        {
            id = pReader.GetInt32(pReader.GetOrdinal("addon_id"));
            name = pReader.GetString(pReader.GetOrdinal("addon_name"));
            feedUrl = GetFeedUrl(pReader);
            donationLink = pReader.GetString(pReader.GetOrdinal("addon_donation_link"));            
            portalName = pReader.GetString(pReader.GetOrdinal("addon_portal"));
            sectionName = pReader.GetString(pReader.GetOrdinal("addon_section"));
            downloadName = pReader.GetString(pReader.GetOrdinal("addon_download_name"));
            url = String.Format(sDownloadBaseUrl, portalName, sectionName, downloadName);
            gameName = pReader.GetString(pReader.GetOrdinal("addon_game"));
            gameId = pReader.GetInt32(pReader.GetOrdinal("addon_game_id"));            
            dateModified = pReader.GetDateTime(pReader.GetOrdinal("addon_last_updated"));
            dateCreated = pReader.GetDateTime(pReader.GetOrdinal("addon_date_created"));
            dateReleased = pReader.GetDateTime(pReader.GetOrdinal("addon_date_released"));
            numComments = pReader.GetInt32(pReader.GetOrdinal("addon_num_comments"));
            numDownloads = pReader.GetDouble (pReader.GetOrdinal("addon_num_downloads"));
            rating = pReader.GetInt32(pReader.GetOrdinal("addon_average_rating"));
            description = pReader.GetString(pReader.GetOrdinal("addon_description"));
            isAvailable = pReader.GetBoolean(pReader.GetOrdinal("addon_is_available"));
            status = (EProjectStatus)pReader.GetByte(pReader.GetOrdinal("addon_status"));
            summary = sHtmlStripper.Replace(description, String.Empty);
            summary = summary.Replace(Environment.NewLine, " ");
            summary = summary.Replace("\n", " ");
            if (summary.Length > 512)
            {
                summary = summary.Substring(0, 512);
            }
            populateAuthors(pConn, pReader.GetString(pReader.GetOrdinal("addon_primary_author")));
            populateFiles(pConn, pFileArchiveCache, pFileDependencyCache);
            populateFileKeys(pConn);
            populateFileFingerprints(pConn);
            populateIndividualFileFingerprints(pIndividualFingerprints);
            populateCategories(pConn);
            populateLocalizations(pReader.GetString(pReader.GetOrdinal("addon_languages")));

            projectElementXml = serialize();
            projectElementBytes = System.Text.Encoding.UTF8.GetBytes(projectElementXml);

            projectListBytes = serializeBinary();            
        }
      
        private byte[] serializeBinary()
        {
                        
            using (MemoryStream ms = new MemoryStream(1000))
            {
            
                using (BinaryWriter bw = new BinaryWriter(ms))
                {

                    // AddOn Name
                    Utility.WriteNetworkString(bw, name);
                    // AddOn Lower Case Name
                    Utility.WriteNetworkString(bw, name.ToLower());
                    // First Category Name
                    if (categories.Count > 0)
                    {
                        Utility.WriteNetworkString(bw, categories[0].Value);
                    }
                    else
                    {
                        Utility.WriteNetworkString(bw, String.Empty);
                    }
                    
                    // First Category URL
                    Utility.WriteNetworkString(bw, string.Format(sCategoryBaseUrl, portalName, sectionName, categories[0].Key.ToString()));
                    // Summary
                    Utility.WriteNetworkString(bw, summary);      
                    // Description - intentionally an empty string
                    Utility.WriteNetworkString(bw, "");                
                    // URL
                    Utility.WriteNetworkString(bw, url);                                
                    // Download URL                
                    Utility.WriteNetworkString(bw, url);
                    // Feed URL
                    Utility.WriteNetworkString(bw, feedUrl);

                    // Relese Name / Default Filename
                    Utility.WriteNetworkString(bw, DefaultFileName);

                    // Curse ID
                    Utility.WriteNetworkString(bw, id.ToString());                                   
                    
                    // Dates in epoch format, accurate to the second:
                    bw.Write((UInt64)Utility.GetEpochDate(dateModified));
                    bw.Write((UInt64)Utility.GetEpochDate(dateCreated));
                    bw.Write((UInt64)Utility.GetEpochDate(dateReleased));

                    // Author count
                    bw.Write((UInt32)authors.Count);
                    foreach (AddOnAuthor author in authors)
                    {
                        // Author Name, E-Mail, and URL
                        Utility.WriteNetworkString(bw, author.Name);
                        Utility.WriteNetworkString(bw, "");
                        Utility.WriteNetworkString(bw, author.Url);                                                       
                    }
                }            
                return ms.ToArray();
            }                        
            
        }

        private string serialize()
        {

            StringWriter sw = new StringWriter();
            XmlTextWriter writer = new XmlTextWriter(sw);

            writer.WriteStartElement("project");
            writer.WriteAttributeString("curseID", id.ToString());

            //Name
            writer.WriteStartElement("name");            
            writer.WriteValue(name);
            writer.WriteEndElement();


            //Game
            writer.WriteStartElement("game");
            writer.WriteAttributeString("id", gameId.ToString());
            writer.WriteValue(gameName);
            writer.WriteEndElement();

            //URL
            writer.WriteStartElement("url");
            writer.WriteValue(url);
            writer.WriteEndElement();

            //Curse-Specific Attributes
            writer.WriteStartElement("curse");
            writer.WriteAttributeString("rating", rating.ToString());
            writer.WriteAttributeString("comments", numComments.ToString());
            //Categories
            writer.WriteStartElement("categories");
            foreach (KeyValuePair<string, string> i in categories)
            {
                writer.WriteStartElement("category");
                writer.WriteAttributeString("url", string.Format(sCategoryBaseUrl, portalName, sectionName, i.Key.ToString()));
                writer.WriteValue(i.Value);
                writer.WriteEndElement();
            }
            writer.WriteEndElement();
            writer.WriteEndElement();

            //Feed URL
            writer.WriteStartElement("feed");
            writer.WriteValue(feedUrl);
            writer.WriteEndElement();

            //Feed URL
            writer.WriteStartElement("donationLink");
            writer.WriteValue(donationLink);
            writer.WriteEndElement();

            //Authors
            writer.WriteStartElement("authors");

            foreach (AddOnAuthor author in authors)
            {
                author.serialize(writer);
            }

            writer.WriteEndElement();

            // Files
            writer.WriteStartElement("files");
            writer.WriteAttributeString("default", defaultFileId.ToString());
            foreach (KeyValuePair<int, AddOnFile> file in files)
            {
                if (file.Value.IsAvailable)
                {
                    file.Value.WriteElement(writer);
                }
            }

            writer.WriteEndElement();

            // Localizations
            writer.WriteStartElement("localizations");
            foreach (AddOnLocalization i in localizations)
            {
                i.serialize(writer);
            }

            writer.WriteEndElement();

            writer.WriteEndElement();
            return sw.ToString();
        }

        private void populateAuthors(SqlConnection pConn, string primaryAuthor)
        {
            authors = new List<AddOnAuthor>();
            authors.Add(new AddOnAuthor(primaryAuthor, String.Format(sAuthorBaseUrl, primaryAuthor)));

            SqlCommand command = new SqlCommand("curseService_GetAddOnAuthors", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = id;
            
            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    if(reader.GetString(0) != primaryAuthor)
                    {
                        authors.Add(new AddOnAuthor(reader.GetString(0), String.Format(sAuthorBaseUrl,reader.GetString(0))));
                    }

                }
            }
        }

        public AddOn Clone()
        {
            return (AddOn)MemberwiseClone();
        }

        private void populateCategories(SqlConnection pConn)
        {

            categories = new List<KeyValuePair<string, string>>();
            SqlCommand command = new SqlCommand("curseService_GetAddOnCategories", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    KeyValuePair<string, string> category = new KeyValuePair<string, string>(reader.GetString(2), reader.GetString(0));
                    categories.Add(category);
                }
            }

            if (categories.Count == 0)
            {
                KeyValuePair<string, string> category = new KeyValuePair<string, string>("17", "Miscellaneous");
                categories.Add(category);
            }

        }

        private int getDefaultFileId()
        {            
            // Release
            if (mLatestFiles.Exists(p => p.Type == EFileType.Release && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.Type == EFileType.Release && p.IsAlternate == false).Id;
            }

            // Beta
            if (mLatestFiles.Exists(p => p.Type == EFileType.Beta && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.Type == EFileType.Beta && p.IsAlternate == false).Id;
            }

            // Alpha
            if (mLatestFiles.Exists(p => p.Type == EFileType.Alpha && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.Type == EFileType.Alpha && p.IsAlternate == false).Id;
            }

            return 0;
        }

        private void populateFiles(SqlConnection pConn,
                                   Dictionary<int, List<AddOnFile>> pFileArchiveCache,
                                   Dictionary<int, List<AddOnFileDependency>> pFileDependencyCache)
        {
            files = new Dictionary<int,AddOnFile>();
            SqlCommand command = new SqlCommand("curseService_GetAddOnFiles", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    
                    AddOnFile file = new AddOnFile(reader, false);                                      
                    files.Add(file.Id, file);

                    // Set the dependencies:
                    if (pFileDependencyCache.ContainsKey(file.Id))
                    {
                        file.Dependencies = pFileDependencyCache[file.Id];
                    }                    
                }
            }

            setLatestFiles();
            defaultFileId = getDefaultFileId();
            if (pFileArchiveCache.ContainsKey(id))
            {
                foreach (AddOnFile file in pFileArchiveCache[id])
                {
                    if (!files.ContainsKey(file.Id))
                    {
                        files.Add(file.Id, file);
                    }
                }
                
            }
        }

        private void setLatestFiles()
        {

            // Set latest files:
            foreach (KeyValuePair<int, AddOnFile> kvp in files)
            {
                AddOnFile file = kvp.Value;
                if (!file.IsAlternate)
                {

                    if (file.AlternateFileId > 0 && !files.ContainsKey(file.AlternateFileId))
                    {
                        file.AlternateFileId = 0;
                    }

                    AddOnFile existingFile = mLatestFiles.FirstOrDefault(p => p.Type == file.Type && p.IsAlternate == false);

                    if (existingFile == null || file.Date > existingFile.Date)
                    {
                        if (existingFile != null)
                        {
                            mLatestFiles.Remove(existingFile);
                            mLatestFiles.Find(p => p.Id == existingFile.AlternateFileId);
                        }

                        mLatestFiles.Add(file);
                    }

                }
            }

            // Now the alternates:
            foreach (EFileType releaseType in System.Enum.GetValues(typeof(EFileType)))
            {

                AddOnFile primaryFile = mLatestFiles.FirstOrDefault(p => p.Type == releaseType);

                if (primaryFile == null)
                {
                    continue;
                }

                if (primaryFile.AlternateFileId > 0)
                {
                    if (files.ContainsKey(primaryFile.AlternateFileId))
                    {
                        AddOnFile secondaryFile = files[primaryFile.AlternateFileId];
                        if (!mLatestFiles.Exists(p => p.Id == secondaryFile.Id))
                        {
                            mLatestFiles.Add(secondaryFile);
                        }
                    }
                }
            }
        }

        private void populateFileKeys(SqlConnection pConn)
        {
            fileKeys = new List<AddOnFileKey>();
            return;

            SqlCommand command = new SqlCommand("curseService_GetAddOnFileKeys", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {

                    fileKeys.Add(new AddOnFileKey(reader.GetString(0), reader.GetBoolean(1)));

                }
            }
        }

        private void populateFileFingerprints(SqlConnection pConn)
        {
            SqlCommand command = new SqlCommand("curseService_GetAddOnFileFingerprints", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                if (!reader.HasRows)
                {
                    return;
                }

                while (reader.Read())
                {
                    int fileId = reader.GetInt32(0);
                    long fingerprint = reader.GetInt64(1);
                    if (!files.ContainsKey(fileId))
                    {
                        continue;
                    }
                    files[fileId].Fingerprints.Add(fingerprint);                                        
                }
            }
        }

        private void populateIndividualFileFingerprints(Dictionary<int, List<IndividualFileFingerprint>> pIndividualFingerprints)
        {

            if (!pIndividualFingerprints.ContainsKey(id))
            {
                return;
            }

            foreach (IndividualFileFingerprint fingerprint in pIndividualFingerprints[id])
            {

                if (!files.ContainsKey(fingerprint.FileId))
                {
                    continue;
                }
                if (!files[fingerprint.FileId].FolderFingerprints.ContainsKey(fingerprint.Folder))
                {
                    files[fingerprint.FileId].FolderFingerprints.Add(fingerprint.Folder, new List<long>());
                    files[fingerprint.FileId].FolderNames.Add(fingerprint.Folder);
                }
                files[fingerprint.FileId].FolderFingerprints[fingerprint.Folder].Add(fingerprint.Fingerprint);                
            }
            
            // Make sure the individual fingerprints are sorted:
            foreach (KeyValuePair<int, AddOnFile> fileEntry in files)
            {
                foreach (KeyValuePair<string, List<long>> folderFingerprints in fileEntry.Value.FolderFingerprints)
                {
                    folderFingerprints.Value.Sort();
                }
            }
        }

        private void populateLocalizations(string pLanguages)
        {
            localizations = new List<AddOnLocalization>();
            localizations.Add(new AddOnLocalization("en", summary));
        }
                               
    }
}
