﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Reflection;
using System.ServiceModel;
using Curse.DownloadStatisticService.Models;
using Curse.Extensions;
using Curse.Logging;
using Curse.ServiceModels.Configuration;
using System.ServiceModel.Activation;

namespace Curse.DownloadStatisticService
{
    [ServiceBehavior(Name = "DownloadStatisticService", Namespace = "http://downloadstats.curseforge.com/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.Single)]
    [AspNetCompatibilityRequirements(RequirementsMode = AspNetCompatibilityRequirementsMode.Allowed)]
    public class DownloadStatisticService : IDownloadStatisticService
    {
        /// <summary>
        /// This method returns a DataContract of Files and Projects from their in memory collections. Filterd by items greater than the date supplied parameter.
        /// If not date is supplied The method default to using 01/01/0001 (which will pull all records).
        /// </summary>
        /// <param name="since">date</param>
        /// <returns>FileProjectCount DataContract</returns>
        public FileProjectCounts GetDownloadCounts(DateTime since)
        {
            if (since.Year == 1)
            {
                since = new DateTime(2001, 1, 1);
            }
            return new FileProjectCounts { FC = GetFileCountsByDate(since), PC = GetProjectCountsByDate(since) };
        }

        /// <summary>
        /// This method returns a DataContract of Projects from their in memory collections. Filterd by items greater than the date supplied parameter.
        /// If not date is supplied The method default to using 01/01/0001 (which will pull all records).
        /// </summary>
        /// <param name="since">date</param>
        /// <returns>ProjectPopularities DataContract</returns>
        public ProjectPopularities GetProjectPopularity(DateTime since)
        {
            if (since.Year == 1)
            {
                since = new DateTime(2001, 1, 1);
            }
            return GetProjectPopularitiesByDate(since);
        }

        public string HealthCheck()
        {
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                conn.Open();
            }

            return "Success";
        }

        private static FileCounts GetFileCountsByDate(DateTime since)
        {
            FileCounts fileList = new FileCounts();
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                var cmd = new SqlCommand("GetFileCountsByDateModified", conn) { CommandType = CommandType.StoredProcedure };
                cmd.Parameters.AddWithValue("@QueryTime", since);
                cmd.CommandTimeout = 120;
                try
                {
                    conn.Open();
                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var file = new FileCount();
                            file.SetFieldsFromDataReader(reader);
                            fileList.Add(file);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to retrieve the list of objects. Details: {0}");
                }
            }
            return fileList;
        }

        private static ProjectCounts GetProjectCountsByDate(DateTime since)
        {
            ProjectCounts projectList = new ProjectCounts();

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                var cmd = new SqlCommand("GetProjectCountsByDateModified", conn) { CommandType = CommandType.StoredProcedure };
                cmd.Parameters.AddWithValue("@QueryTime", since);
                cmd.CommandTimeout = 120;
                try
                {
                    conn.Open();
                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var project = new ProjectCount();
                            project.SetFromDataReader(reader);
                            projectList.Add(project);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to update to retrieve the list of objects: {0}");
                }
            }
            return projectList;
        }

        private static ProjectPopularities GetProjectPopularitiesByDate(DateTime since)
        {
            ProjectPopularities projectList = new ProjectPopularities();

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                var cmd = new SqlCommand("GetProjectPopularitiesByDateModified", conn) { CommandType = CommandType.StoredProcedure };
                cmd.Parameters.AddWithValue("@QueryTime", since);
                cmd.CommandTimeout = 120;
                try
                {
                    conn.Open();
                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var project = new ProjectPopularity();
                            project.SetFromDataReader(reader);
                            projectList.Add(project);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to update to retrieve the list of objects: {0}");
                }
            }
            return projectList;
        }

        /// <summary>
        /// call the sproc that get the total unique download for the month you pass the first day of into it
        /// </summary>
        /// <param name="date">the first day of the month that you want the calulation ran on</param>
        /// <returns></returns>
        public int GetUniqueDownloadsForMonth(DateTime date)
        {
            int uniqueDownloadsForMonth = 0;

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                var cmd = new SqlCommand("RP_GetUniqueDownloadsForMonth", conn) { CommandType = CommandType.StoredProcedure };
                cmd.Parameters.AddWithValue("@firstDayOfMonthToCalculate", date);
                cmd.CommandTimeout = 120;
                try
                {
                    var downloads = new SqlParameter("@uniqueDownloadsForMonth", SqlDbType.Int) { Direction = ParameterDirection.Output };
                    cmd.Parameters.Add(downloads);
                    conn.Open();
                    cmd.ExecuteNonQuery();
                    uniqueDownloadsForMonth = Int32.Parse(cmd.Parameters["@uniqueDownloadsForMonth"].Value.ToString());
                     
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to update to retrieve the list of objects: {0}");
                }
            }
            return uniqueDownloadsForMonth;
        }

    }  // end of class
} // end of namespace
