﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ServiceModels.Caching;
using Curse.DownloadStatisticService.Models;
using System.Data.SqlClient;
using Curse.ServiceModels.Configuration;
using Curse.Extensions;

namespace Curse.DownloadStatisticService.Caching
{
	public class FileCache : IncrementalCache<File>
	{
		private ConcurrentDictionary<int, File> _currentResults = new ConcurrentDictionary<int, File>();

		public static FileCache _instance = new FileCache();
		public static FileCache Instance { get { return _instance; } }

		public FileCache() : base("DownloadStatistics", "spGetFileData") { }

		public override bool UpdateCache()
		{
			var results = new ConcurrentDictionary<int, File>(_currentResults);

			using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
			{
				var cmd = new SqlCommand("spGetFileData", conn) {CommandType = System.Data.CommandType.StoredProcedure};
			    cmd.Parameters.Add(LastQueryParameter);

				conn.Open();
				using (var reader = cmd.ExecuteReader())
				{
					while (reader.Read())
					{
						var file = new File();
						file.SetFromDataReader(reader);
                        _currentResults.AddOrUpdate(file.ID, file, (key, value) => file);    
					}
				}
			}

			return true;
		}

		public File GetByID(int id)
		{
			if (_currentResults.ContainsKey(id))
			{
				return _currentResults[id];
			}
			else return null;
		}

		public bool CheckFileData(AddDownloadLogEntryMessage message, out string returnMessage)
		{
			if (!_currentResults.ContainsKey(message.FileID))
			{
				if (!CreateFile(message))
				{
					returnMessage = "Failed to Create File record.";
					return false;
				}
			}

			if (message.ProjectID != _currentResults[message.FileID].ProjectID)
			{
				returnMessage = string.Format("ProjectIDs do not match. Input ProjectID {0} should be {1} for FileID {2}.", message.ProjectID, _currentResults[message.FileID].ProjectID, message.FileID);
				return false;
			}

			returnMessage = "Success";
			return true;
		}

		private bool CreateFile(AddDownloadLogEntryMessage message)
		{
			using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
			{
				var cmd = new SqlCommand("INSERT INTO [File] (ID, ProjectID, TotalDownloads, HistoricCurseDownloads, HistoricCurseForgeDownloads, DateModified) VALUES(@ID, @ProjectID, 0, 0, 0, GETUTCDATE())", conn);
				cmd.Parameters.Add("ID", System.Data.SqlDbType.Int).Value = message.FileID;
				cmd.Parameters.Add("ProjectID", System.Data.SqlDbType.Int).Value = message.ProjectID;

				try
				{
					conn.Open();
					cmd.ExecuteNonQuery();
				}
				catch (SqlException sqlex)
				{
					if (sqlex.Message.Contains("Violation of PRIMARY KEY constraint"))
					{
						//swallow the exception as the file got inserted but the service cache just doesn't know it yet
					}
					else
					{
						throw new Exception(sqlex.Message, sqlex.InnerException);
					}
				}
				catch (Exception exc)
				{
					Logger.Log("Failed to create File Record. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
					return false;
				}

				var file = new File
				{
					ID = message.FileID,
					ProjectID = message.ProjectID,
					TotalDownloads = 0,
					HistoricCurseDownloads = 0,
					HistoricCurseForgeDownloads = 0
				};

                _currentResults.AddOrUpdate(file.ID, file, (key, value) => file);

				return true;

			}
		}

	}
}