using System;
using System.Collections.Concurrent;
using System.Data;
using System.Linq.Expressions;
using System.Web;
using System.Collections.Generic;
using System.Web.Script.Serialization;
using System.Web.Services;
using System.ComponentModel;
using System.Data.SqlClient;
using System.Text;
using System.Net;
using System.IO;
using System.Configuration;
using System.Security.Cryptography;
using System.Xml;
using System.Threading;
using System.Linq;

using Curse.Extensions;
using Curse.Logging.Uploader;
using Curse.Mailer;
using AuthUser = Curse.Auth.User;
using Curse.Caching;
using Curse.Logging;
using System.Diagnostics;
using System.Web.Script.Services;

namespace Curse.Auth
{

    [WebService(Namespace = "http://auth.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public class NetworkService : WebService
    {
        // Constants        
        private const int PasswordRequestPruneInterval = 60 * 1000; // 1 minute

        // Dictionaries
        static readonly ConcurrentDictionary<string, byte> IPBans = new ConcurrentDictionary<string, byte>();

        static readonly bool DebugMode;
        static readonly byte[] AdminKey;
        static readonly string ServiceUrl;
        static readonly bool ReadOnly;
        static readonly bool ValidateReferrer;
        static readonly bool LogInvalidDomains;

        // Static Constructor
        static NetworkService()
        {
            var sw = Stopwatch.StartNew();

            DebugMode = Convert.ToBoolean(ConfigurationManager.AppSettings["Debug"]);
            ReadOnly = Convert.ToBoolean(ConfigurationManager.AppSettings["ReadOnly"]);
            AdminKey = Encoding.UTF8.GetBytes(ConfigurationManager.AppSettings["SecretKey"]);
            ServiceUrl = ConfigurationManager.AppSettings["ServiceUrl"];
            ValidateReferrer = Convert.ToBoolean(ConfigurationManager.AppSettings["ValidateReferrer"]);
            LogInvalidDomains = Convert.ToBoolean(ConfigurationManager.AppSettings["LogInvalidDomains"]);

            Logger.Init(ConfigurationManager.AppSettings["LogPath"], "AuthenticationService");
            Logger.Info("Authentication Service Starting...");
            LogUploader.Initialize(8, ConfigurationManager.AppSettings["LogServiceUrl"], ConfigurationManager.AppSettings["LogServiceApiKey"]);

            using (var conn = DatabaseUtility.GetAuthConnection(false))
            {
                try
                {
                    Logger.Info("Cluster Manager Initializing...");
                    ClusterManager.Instance.Initialize("AuthenticationService", conn, true);

                    Logger.Info("Cache Cluster Initializing...");
                    CacheCluster.Instance.Start();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to start cache cluster");
                    throw;
                }

                Logger.Info("Network Sites Loading...");
                NetworkSite.Initialize(conn);
            }

            MailProvider.Instance.Initialize();
            Logger.Info("Authentication Service Started in " + sw.ElapsedMilliseconds + " milliseconds");
        }

        // Per-Request Constructor
        public NetworkService()
        {
            if (IPBans.ContainsKey(Context.Request.GetClientIPAddress().ToString()))
            {
                Context.ApplicationInstance.CompleteRequest();
            }
        }

        #region Worker and Utility Methods

        private ESetUserPasswordResult DoSetUserPassword(int siteid, int userid, string oldpass, string newpass)
        {

            if (ReadOnly)
            {
                return ESetUserPasswordResult.Unsuccessful;
            }

            try
            {
                var user = AuthUser.GetByID(userid);
                var site = NetworkSite.GetByID(siteid);
                var oldPlainTextPassword = site.DecryptString(oldpass);
                var newPlainTextPassword = site.DecryptString(newpass);

                if (!InputValidation.IsValidPassword(newPlainTextPassword))
                {
                    return ESetUserPasswordResult.InvalidPassword;
                }

                using (var sql = DatabaseUtility.GetAuthConnection(false))
                {
                    if (!user.ValidatePassword(oldPlainTextPassword, siteid, sql))
                    {
                        return ESetUserPasswordResult.InvalidPassword;
                    }

                    user.ChangePassword(newPlainTextPassword, sql);
                }
            }
            catch (KeyNotFoundException)
            {
                return ESetUserPasswordResult.UserNotFound;
            }

            return ESetUserPasswordResult.Success;

        }

        private ESetUserProfileResult DoSetUserProfileInfo(int userid, string firstname, string lastname, string gender, string country, string region, string city, int primaryemail, bool newsletter)
        {

            UserEmail email = null;
            try
            {
                email = UserEmail.GetByID(primaryemail);
            }
            catch (KeyNotFoundException)
            {
                return ESetUserProfileResult.EmailNotFound;
            }

            string message = null;

            if (!AuthUser.TryValidateProfile(firstname, lastname, gender, country, region, city, email.Address, out message))
            {
                return ESetUserProfileResult.InvalidProfileData;
            }

            try
            {
                var user = AuthUser.GetByID(userid);

                using (var sql = DatabaseUtility.GetAuthConnection(false))
                {
                    user.Profile.Update(sql,
                                        firstname,
                                        lastname,
                                        gender,
                                        country,
                                        region,
                                        city,
                                        primaryemail,
                                        newsletter);

                    user.Profile = new UserProfile(user.ID, sql);
                }

            }
            catch (KeyNotFoundException)
            {
                return ESetUserProfileResult.UserNotFound;
            }

            return ESetUserProfileResult.Success;

        }

        private ESetUserProfileAndEmailResult v2DoSetUserProfileInfo(int userid, string firstname, string lastname, string gender, string country, string region, string city, string email, bool newsletter)
        {

            if (DoIsEmailAddressInUse(email))
            {
                return ESetUserProfileAndEmailResult.EmailInUse;
            }

            if (!InputValidation.IsValidEmail(email))
            {
                return ESetUserProfileAndEmailResult.InvalidEmail;
            }

            try
            {
                AuthUser.GetByID(userid).UpdateEmailAddress(email);
            }
            catch (KeyNotFoundException)
            {
                return ESetUserProfileAndEmailResult.Unsuccessful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to update primary e-mail address");
                return ESetUserProfileAndEmailResult.UnknownError;
            }

            string message = null;

            if (!AuthUser.TryValidateProfile(firstname, lastname, gender, country, region, city, email, out message))
            {
                return ESetUserProfileAndEmailResult.InvalidProfileData;
            }

            try
            {
                var user = AuthUser.GetByID(userid);

                var primaryEmailID = user.PrimaryEmail.ID;

                using (var sql = DatabaseUtility.GetAuthConnection(false))
                {
                    user.Profile.Update(sql,
                                        firstname,
                                        lastname,
                                        gender,
                                        country,
                                        region,
                                        city,
                                        primaryEmailID,
                                        newsletter);

                    user.Profile = new UserProfile(user.ID, sql);
                }

            }
            catch (KeyNotFoundException)
            {
                return ESetUserProfileAndEmailResult.UserNotFound;
            }

            return ESetUserProfileAndEmailResult.Success;
        }

        private bool DoSignOutUser(int siteID, int userID)
        {
            try
            {
                using (var sql = DatabaseUtility.GetAuthConnection(false))
                {
                    AuthUser.GetByID(userID).ClearUserSessions(sql);
                }

                return true;
            }
            catch (KeyNotFoundException)
            {
                return false;
            }
        }

        public EAdminSetUserPasswordStatus DoAdminSetUserPassword(int siteid, int userid, string newpass)
        {
            var site = NetworkSite.GetByID(siteid);
            var decpwd = site.DecryptString(newpass);

            if (!InputValidation.IsValidPassword(decpwd))
            {
                return EAdminSetUserPasswordStatus.InvalidPassword;
            }

            User user = null;
            try
            {
                user = AuthUser.GetByID(userid);
            }
            catch (KeyNotFoundException)
            {
                return EAdminSetUserPasswordStatus.UserNotFound;
            }

            using (var sql = DatabaseUtility.GetAuthConnection(false))
            {
                user.ChangePassword(decpwd, sql);
            }

            return EAdminSetUserPasswordStatus.Success;
        }

        public EAdminSetUserPasswordStatus DoAdminSetBinaryUserLegacyPassword(int siteid, int legacyid, int userid, string passwordHash)
        {
            User user = null;
            try
            {
                user = AuthUser.GetByID(userid);
            }
            catch (KeyNotFoundException)
            {
                return EAdminSetUserPasswordStatus.UserNotFound;
            }

            using (var sql = DatabaseUtility.GetAuthConnection(false))
            {
                if (user.ChangeLegacyPassword(siteid, legacyid, passwordHash, null, ELegacyEncryptionType.None, sql) == ESetUserPasswordResult.Success)
                {
                    return EAdminSetUserPasswordStatus.Success;
                }

                return EAdminSetUserPasswordStatus.Unsuccessful;
            }
        }

        private LoginResult DoLogin(SessionType sessionType, int siteID, string username, string encryptedPassword, string sessionID)
        {
            // If this is not a client login, ensure a valid session:
            if (sessionType != SessionType.Client && !InputValidation.IsValidSession(sessionID))
            {
                return new LoginResult(ELoginStatus.InvalidSession);
            }

            // Validate inputs for sanity check
            if (string.IsNullOrEmpty(username) || string.IsNullOrEmpty(encryptedPassword) || siteID <= 0)
            {
                return new LoginResult(ELoginStatus.Unsuccessful);
            }

            NetworkSite site = null;

            try
            {
                site = NetworkSite.GetByID(siteID);
            }
            catch (KeyNotFoundException)
            {
                return new LoginResult(ELoginStatus.Unsuccessful);
            }

            var plainTextPassword = site.DecryptString(encryptedPassword);
            string returnSession = null;

            User user;

            // Check if the login has an @. If so, assume it is an e-mail address.
            if (username.Contains("@"))
            {
                try
                {
                    user = AuthUser.GetByEmailAddress(username);
                }
                catch (KeyNotFoundException)
                {
                    return new LoginResult(ELoginStatus.UnknownEmail);
                }
            }
            else
            {
                try
                {
                    user = AuthUser.GetByName(username);
                }
                catch (KeyNotFoundException)
                {
                    return new LoginResult(ELoginStatus.UnknownUsername);
                }
            }

            using (var conn = DatabaseUtility.GetAuthConnection(false))
            {
                // Validate the password:
                if (!user.ValidatePassword(plainTextPassword, siteID, conn))
                {
                    return new LoginResult(ELoginStatus.InvalidPassword);
                }

                // If this is a client login, try to lookup ther existing session
                if (sessionType == SessionType.Client && string.IsNullOrEmpty(sessionID))
                {
                    // Look for an existing session, for this site ID
                    sessionID = AuthUser.GetSessionFromUserID(user.ID, siteID) ?? AuthUser.GenerateSessionKey(HttpContext.Current.Request.GetClientIPAddress());
                }


                // Establish the new session
                returnSession = user.CreateSession(conn, sessionID, siteID);
            }

            return new LoginResult(ELoginStatus.Success, user.ID, user.GetSubscriptionStatus(ESubscriptionType.Premium) ? (Byte)1 : (Byte)0, returnSession, user.Name);
        }

        private void LegacyDoHandleBadLogin(ELoginStatus status)
        {
            switch (status)
            {
                case ELoginStatus.InvalidPassword:
                    EndWithClientError("Invalid username or password");
                    break;
                case ELoginStatus.InvalidSession:
                    EndWithClientError("Invalid session");
                    break;
                case ELoginStatus.UnknownEmail:
                    EndWithClientError("No such e-mail address.");
                    break;
                case ELoginStatus.UnknownUsername:
                    EndWithClientError("No such username.");
                    break;
                case ELoginStatus.UnauthorizedLogin:
                    // Do nothing for this status, as it is already handled.                    
                    break;
            }
        }

        private bool DoCanRenameUser(int siteid, int uid)
        {
            try
            {
                using (var sql = DatabaseUtility.GetAuthConnection(true))
                {
                    using (var cmd = sql.CreateCommand())
                    {
                        cmd.CommandText = "SELECT COUNT(0) FROM userrenames WHERE _uid=@uid";
                        cmd.Parameters.AddWithValue("uid", uid);
                        return (int)cmd.ExecuteScalar() > 0;
                    }
                }
            }
            catch (KeyNotFoundException)
            {
                return false;
            }
        }

        public EUpdatePrimaryEmailStatus DoUpdatePrimaryEmailAddress(int siteid, int userid, string emailaddress)
        {
            if (DoIsEmailAddressInUse(emailaddress))
            {
                return EUpdatePrimaryEmailStatus.EmailInUse;
            }

            if (!InputValidation.IsValidEmail(emailaddress))
            {
                return EUpdatePrimaryEmailStatus.InvalidEmail;
            }

            try
            {
                AuthUser.GetByID(userid).UpdateEmailAddress(emailaddress);
                return EUpdatePrimaryEmailStatus.Successful;
            }
            catch (KeyNotFoundException)
            {
                return EUpdatePrimaryEmailStatus.UnknownUser;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to update primary e-mail address");
                return EUpdatePrimaryEmailStatus.UnknownError;
            }
        }

        public EAdminRenameUserStatus DoAdminRenameUser(int siteid, int uid, string username)
        {
            User user = null;

            try
            {
                user = AuthUser.GetByID(uid);
            }
            catch (KeyNotFoundException)
            {
                return EAdminRenameUserStatus.UserNotFound;
            }

            if (!InputValidation.IsValidUserNameWithNoLeadingOrTrailingUnderscores(username))
            {
                return EAdminRenameUserStatus.InvalidUsername;
            }

            if (!username.Equals(user.Name, StringComparison.InvariantCultureIgnoreCase) && DoIsUserNameInUse(username))
            {
                return EAdminRenameUserStatus.UsernameInUse;
            }

            using (var sql = DatabaseUtility.GetAuthConnection(false))
            {
                if (!user.Rename(username, sql, true))
                {
                    return EAdminRenameUserStatus.Unsuccessful;
                }
            }

            return EAdminRenameUserStatus.Success;
        }

        private static void PrunePasswordResetsThread()
        {
            while (true)
            {
                Thread.Sleep(PasswordRequestPruneInterval); // Every minute     
                try
                {
                    PasswordResetRequest.Prune();
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to prune password reset requests");
                }
            }
        }

        private void Log(string message)
        {
            Logger.Info(message, new { IPAddress = Context.Request.GetClientIPAddress(true).ToString() });
        }

        private void BanCurrentHost()
        {
            BanIPAddress(Context.Request.UserHostAddress);
        }

        private void BanIPAddress(string ip)
        {
            if (DebugMode)
                return;


            if (!IPBans.ContainsKey(ip))
            {
                IPBans.TryAdd(ip, 1);
            }

        }

        private void UnbanIPAddress(string ip)
        {
            if (!IPBans.ContainsKey(ip))
            {
                return;
            }

            byte val;
            IPBans.TryRemove(ip, out val);
        }

        private void LogAndBan(string logstr)
        {
            BanCurrentHost();
            Log(logstr);
        }

        private void EndWithClientError(string error)
        {
            Context.Response.StatusCode = 400;
            Context.Response.Write(error);
            Context.Response.Flush();
        }

        private bool TryRequireAdminKey(string key, out string message)
        {
            if (!DebugMode && !Context.Request.IsSecureConnection && !Context.Request.IsLocal && !Context.Request.GetClientIPAddress(false).ToString().StartsWith("10."))
            {
                message = "Must use a local or secure connection for this method.";
                return false;
            }

            if (key != ConfigurationManager.AppSettings["AdminKey"])
            {
                LogAndBan("Tried to pose as administrative site when not authorized. Banned.");
                message = "Invalid shared key. Banned.";
                return false;
            }
            message = "Successful";
            return true;
        }

        private bool TryIsHostNetworkSite(int siteid, out string message)
        {

            var site = GetCurrentNetworkSite(siteid);

            if (site != null)
            {
                message = string.Empty;
                return true;
            }

            LogAndBan("Site " + siteid + " tried to impersonate other site.");
            message = "Not in network site access list.";
            return false;

        }

        private bool IsHostNetworkSite(int siteid)
        {

            var site = GetCurrentNetworkSite(siteid);

            if (site != null)
            {                
                return true;
            }

            LogAndBan("Host tried to impersonate a site: " + siteid);
            return false;

        }

        private bool LegacyIsHostNetworkSite(int siteid)
        {
            var site = GetCurrentNetworkSite(siteid);

            if (site != null)
            {
                return true;
            }

            LogAndBan("Site tried to impersonate other site.");
            EndWithClientError("Not in network site access list.");
            return false;

        }

        private bool TryCanHostAlterUser(int siteID, out string message)
        {

            var site = GetCurrentNetworkSite(siteID);

            if (site == null)
            {
                LogAndBan("Site " + siteID + " tried to impersonate other site.");
                message = "Not in network site access list.";
                return false;
            }
            else if (!site.CanAlterUser)
            {
                LogAndBan("Site not permitted to alter users.");
                message = "Site not permitted to alter users.";
                return false;
            }
            else
            {
                message = string.Empty;
                return true;
            }
        }

        private bool LegacyCanHostAlterUser(int siteid)
        {

            var site = GetCurrentNetworkSite(siteid);

            if (site == null || !site.CanAlterUser)
            {
                LogAndBan("Site tried to alter user.");
                EndWithClientError("Not authorized to alter users.");
                return false;
            }
            return true;
        }

        private NetworkSite GetCurrentNetworkSite(int siteid)
        {
            NetworkSite site = null;

            try
            {
                site = NetworkSite.GetByID(siteid);
            }
            catch (KeyNotFoundException)
            {
                Logger.Warn("Unknown site: " + siteid);
                return null;
            }

            if (Context.Request.IsLocal || Context.Request.GetClientIPAddress(false).ToString().StartsWith("10.") || site.ValidateIPAddress(Context.Request.GetClientIPAddress().ToString()))
            {
                return site;
            }

            return null;
        }

        private XmlTextWriter BeginComplexResponse()
        {
            var stream = new MemoryStream(1024);
            var writer = new XmlTextWriter(stream, Encoding.UTF8);

            writer.WriteStartDocument(true);
            return writer;
        }

        private void EndComplexResponse(XmlTextWriter writer)
        {
            writer.WriteEndDocument();
            writer.Flush();

            var stream = (MemoryStream)writer.BaseStream;
            var outstream = Context.Response.OutputStream;
            Context.Response.ContentType = "text/xml";
            stream.WriteTo(outstream);
            outstream.Flush();
            writer.Close();
            Context.ApplicationInstance.CompleteRequest();
        }

        private void SimpleResponse(params object[] response)
        {
            var split = false;
            foreach (object o in response)
            {
                if (o != null)
                {
                    if (!split)
                    {
                        split = true;
                    }
                    else
                    {
                        Context.Response.Write(':');
                    }
                    Context.Response.Write(o);
                }
            }
            Context.Response.Flush();
            Context.ApplicationInstance.CompleteRequest();
        }

        private string DoHashEmail(string emailaddress)
        {
            var mailbytes = Encoding.UTF8.GetBytes(emailaddress);
            var hasher = SHA1.Create();

            var data = new byte[mailbytes.Length + AdminKey.Length];

            Buffer.BlockCopy(AdminKey, 0, data, 0, AdminKey.Length);

            Buffer.BlockCopy(mailbytes, 0, data, AdminKey.Length, mailbytes.Length);

            return Convert.ToBase64String(hasher.ComputeHash(data));
        }

        private void DoResendActivationEmail(string username, int emailid, string from, string redirect, bool suppressresponse)
        {
            UserEmail email;

            try
            {
                email = UserEmail.GetByID(emailid);
            }
            catch (KeyNotFoundException)
            {
                if (!suppressresponse)
                {
                    SimpleResponse((int)ESendActivationResult.EmailNotFound);
                }
                return;
            }

            if (email.Verified)
            {
                if (!suppressresponse)
                {
                    SimpleResponse((int)ESendActivationResult.AlreadyActivated);
                }
                return;
            }

            var activationCode = DoHashEmail(email.Address);
            var activationLink = string.Format("{0}/activateEmail?emailid={1}&code={2}&redirect={3}", ServiceUrl, emailid, HttpUtility.UrlEncodeUnicode(activationCode), HttpUtility.UrlEncodeUnicode(redirect));

            UserMailer.Instance.SendAccountActivation(email.Address, username, activationLink, activationCode);

            if (!suppressresponse)
            {
                SimpleResponse((int)ESendActivationResult.Sent);
            }
        }

        private ERecoverUsernameStatus DoRecoverUsername(int siteID, string emailAddress)
        {

            User user = null;

            // Find the user by e-mail or username:

            if (!InputValidation.IsValidEmail(emailAddress))
            {
                return ERecoverUsernameStatus.InvalidEmail;
            }

            try
            {
                user = AuthUser.GetByEmailAddress(emailAddress);
            }
            catch (KeyNotFoundException)
            {
                return ERecoverUsernameStatus.UnknownEmail;
            }


            // Email the user
            UserMailer.Instance.SendUsernameRecovery(user.PrimaryEmail.Address, user.Name);

            // Return the info needed for the UI
            return ERecoverUsernameStatus.Successful;

        }

        private AccountRecoveryResult DoResetPassword(int siteID, Int32 token, string encryptedPassword, string encryptedConfirmPassword)
        {

            var site = NetworkSite.GetByID(siteID);

            String decryptedNewPassword = null;
            String decryptedConfirmNewPassword = null;

            try
            {
                decryptedNewPassword = site.DecryptString(encryptedPassword);
                decryptedConfirmNewPassword = site.DecryptString(encryptedConfirmPassword);
            }
            catch (FormatException)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InvalidPassword);
            }

            // Make sure the passwords match
            if (decryptedNewPassword != decryptedConfirmNewPassword)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.NonMatchingPasswords);
            }

            // Make sure the password is valid
            if (!InputValidation.IsValidPassword(decryptedNewPassword))
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InvalidPassword);
            }

            var request = PasswordResetRequest.GetByToken(token); ;

            // Get the request by the token
            if (request == null)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.TokenNotFound);
            }

            // Get the user, given the request ID
            User user = null;
            try
            {
                user = AuthUser.GetByID(request.UserID);
            }
            catch (KeyNotFoundException)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownUsername);
            }

            // Change the password
            using (SqlConnection sql = DatabaseUtility.GetAuthConnection(false))
            {
                user.ChangePassword(decryptedNewPassword, sql);
            }

            // Remove the reset request
            try
            {
                request.Delete();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to delete password reset request");
            }

            return new AccountRecoveryResult(EAccountRecoveryStatus.Successful, user.PrimaryEmail.Address, user.Name);
        }

        private AccountRecoveryResult DoGetAccountRecoveryRequestByToken(int token)
        {

            // Get the request, by token            
            var request = PasswordResetRequest.GetByToken(token);

            if (request == null)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.TokenNotFound);
            }

            // Get the associated user, by ID
            User user = null;
            try
            {
                user = AuthUser.GetByID(request.UserID);
            }
            catch (KeyNotFoundException)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownUsername);
            }

            return new AccountRecoveryResult(EAccountRecoveryStatus.Successful, user.PrimaryEmail.Address, user.Name, request.Token, PasswordResetRequest.Cooldown, request.DateCreated);

        }

        private AccountRecoveryResult DoGetAccountRecoveryRequestByUserID(int userID)
        {

            User user = null;

            // Get the associated user, by ID
            try
            {
                user = AuthUser.GetByID(userID);
            }
            catch (KeyNotFoundException)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownUsername);
            }


            // Get the request, by token
            var request = PasswordResetRequest.GetByUserID(userID);

            if (request == null)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.TokenNotFound);
            }

            return new AccountRecoveryResult(EAccountRecoveryStatus.Successful, user.PrimaryEmail.Address, user.Name, request.Token, PasswordResetRequest.Cooldown, request.DateCreated);

        }

        private AccountRecoveryResult DoRecoverAccount(string usernameOrEmail, string redirectUrl)
        {

            User user = null;

            // Find the user by e-mail or username:
            if (usernameOrEmail.Contains("@"))
            {
                if (!InputValidation.IsValidEmail(usernameOrEmail))
                {
                    return new AccountRecoveryResult(EAccountRecoveryStatus.InvalidEmail);
                }

                try
                {
                    user = AuthUser.GetByEmailAddress(usernameOrEmail);
                }
                catch (KeyNotFoundException)
                {
                    return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownEmail);
                }

            }
            else
            {
                if (!InputValidation.IsValidUserName(usernameOrEmail))
                {
                    return new AccountRecoveryResult(EAccountRecoveryStatus.InvalidUsername);
                }

                try
                {
                    user = AuthUser.GetByName(usernameOrEmail);
                }
                catch (KeyNotFoundException)
                {
                    return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownUsername);
                }
            }

            var existingRequest = PasswordResetRequest.GetByUserID(user.ID);

            // Check the state of the existing request
            if (existingRequest != null)
            {
                if (!existingRequest.IsRenewable) // Still in cooldown
                {
                    return new AccountRecoveryResult(EAccountRecoveryStatus.ResetAlreadyRequested, user.PrimaryEmail.Address, user.Name, PasswordResetRequest.Cooldown);
                }

                // Cooled down, so we can delete it                
                existingRequest.Delete();

            }

            // Create a token and e-mail the user
            var token = PasswordResetRequest.Create(user.ID);

            if (token == null)
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownError, "The server failed to generate a unique token. Please try again.");
            }

            // Generate the reset link
            var resetLink = redirectUrl + (redirectUrl.Contains("?") ? "&" : "?") + "token=" + HttpUtility.UrlEncodeUnicode(token.Token.ToString());

            // Email the user
            UserMailer.Instance.SendAccountRecovery(user.PrimaryEmail.Address,
                                                      user.Name,
                                                      resetLink,
                                                      token.DateCreated.ToLongDateString(),
                                                      token.DateCreated.ToShortTimeString() + " UTC",
                                                      token.DateExpires.ToLongDateString(),
                                                      token.DateExpires.ToShortTimeString() + " UTC");

            // Return the info needed for the UI
            return new AccountRecoveryResult(EAccountRecoveryStatus.Successful, user.PrimaryEmail.Address, user.Name, token.Token, PasswordResetRequest.Cooldown, token.DateCreated);

        }

        private bool DoIsEmailAddressInUse(string address)
        {
            using (var sql = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = new SqlCommand("spUserIDFromEmail", sql))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strEmail", SqlDbType.VarChar, 64);
                    cmd.Parameters["@strEmail"].Value = address;
                    return (cmd.ExecuteScalar() != null);
                }
            }

        }

        private bool DoIsUserNameInUse(string username)
        {
            return AuthUser.GetIDByName(username) > 0;
        }

        private bool DoActivateEmailAddress(int id, string code)
        {
            try
            {
                var email = UserEmail.GetByID(id);

                if (DoHashEmail(email.Address) != code)
                {
                    EndWithClientError("Email activation code is wrong.");
                    return false;
                }
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection(false))
                {
                    email.verify(sql);
                    var user = AuthUser.GetByEmailAddress(email.Address);
                    user.ClearCache();
                }

            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such email address.");
                return false;
            }
            return true;
        }

        private EValidateEmailStatus DoActivateEmailAddressV2(int id, string code)
        {

            try
            {
                var email = UserEmail.GetByID(id);

                if (DoHashEmail(email.Address) != code)
                {
                    return EValidateEmailStatus.InvalidActivationCode;
                }

                using (var conn = DatabaseUtility.GetAuthConnection(false))
                {
                    email.verify(conn);
                    try
                    {
                        var user = AuthUser.GetByEmailAddress(email.Address);
                        user.ClearCache();
                    }
                    catch (KeyNotFoundException)
                    {
                        return EValidateEmailStatus.UserNotFound;
                    }
                }

                return EValidateEmailStatus.Success;
            }
            catch (KeyNotFoundException)
            {
                return EValidateEmailStatus.EmailNotFound;
            }
        }

        private AddUserResult DoAddUser(int siteid,
                            string username,
                            string password,
                            string firstName,
                            string lastname,
                            string gender,
                            string country,
                            string region,
                            string city,
                            string email,
                            bool newsletter,
                            string redirect,
                            string sessionID)
        {

            if (ReadOnly)
            {
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            if (DoIsEmailAddressInUse(email))
            {
                return new AddUserResult(EAddUserStatus.EmailInUse);
            }

            if (DoIsUserNameInUse(username))
            {
                return new AddUserResult(EAddUserStatus.UsernameInUse);
            }

            if (gender == "0")
            {
                gender = null;
            }

            if (firstName == null)
            {
                firstName = string.Empty;
            }

            string message;

            if (!AuthUser.TryValidateProfile(firstName, lastname, gender, country, region, city, email, out message))
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile, message);
            }

            if (!InputValidation.IsValidUserNameWithNoLeadingOrTrailingUnderscores(username))
            {
                return new AddUserResult(EAddUserStatus.InvalidUsername);
            }

            var site = NetworkSite.GetByID(siteid);
            if (site == null)
            {
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            string plainTextPassword = null;

            try
            {
                plainTextPassword = site.DecryptString(password);

                if (!InputValidation.IsValidPassword(plainTextPassword))
                {
                    return new AddUserResult(EAddUserStatus.InvalidPassword);
                }
            }
            catch (FormatException)
            {
                return new AddUserResult(EAddUserStatus.InvalidPassword);
            }


            User user = null;

            using (var conn = DatabaseUtility.GetAuthConnection(false))
            {
                user = AuthUser.Create(conn, siteid, username, plainTextPassword, email, firstName, lastname, gender, country, city, region, newsletter);
                if (user != null)
                {
                    if (string.IsNullOrEmpty(sessionID))
                    {
                        sessionID = AuthUser.GenerateSessionKey(HttpContext.Current.Request.GetClientIPAddress(true));
                    }

                    user.CreateSession(conn, sessionID, siteid);
                }
                else
                {
                    Logger.Warn("Failed to create new user");
                }
            }

            if (user == null)
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile);
            }

            redirect = string.IsNullOrEmpty(redirect) ? site.Url : redirect;
            DoResendActivationEmail(username, user.PrimaryEmail.ID, site.Email, redirect, true);
            return new AddUserResult(EAddUserStatus.Successful, user.ID, sessionID);
        }

        private AddUserResult DoAddLegacyUser(int siteid,
                            int legacyid,
                            string username,
                            string password,
                            string firstName,
                            string lastname,
                            string gender,
                            string country,
                            string region,
                            string city,
                            string email,
                            bool newsletter,
                            string redirect,
                            DateTime registeredDate,
                            string sessionID)
        {

            if (ReadOnly)
            {
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            if (DoIsEmailAddressInUse(email))
            { //Check the email address
                return new AddUserResult(EAddUserStatus.EmailInUse);
            }

            if (DoIsUserNameInUse(username))
            { //Check the username
                return new AddUserResult(EAddUserStatus.UsernameInUse);
            }

            if (gender == "0")
            { // I dont know why they are doing this
                gender = "";
            }

            string message;
            if (!AuthUser.TryValidateProfile(firstName, lastname, gender, country, region, city, email, out message))
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile, message);
            }

            if (!InputValidation.IsValidUserNameWithNoLeadingOrTrailingUnderscores(username))
            { //Validate the username
                return new AddUserResult(EAddUserStatus.InvalidUsername);
            }

            NetworkSite site = NetworkSite.GetByID(siteid);
            if (site == null)
            { // Check that the site is valid
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            User user = null;
            using (SqlConnection conn = DatabaseUtility.GetAuthConnection())
            {
                user = AuthUser.CreateLegacy(conn, siteid, legacyid, null, ELegacyEncryptionType.None, username, password, email, firstName, lastname, gender, country, city, region, newsletter, registeredDate);
                user.CreateSession(conn, sessionID, siteid);
            }

            if (user == null)
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile);
            }
            else
            {
                return new AddUserResult(EAddUserStatus.Successful, user.ID, sessionID);
            }
        }

        private AddUserResult v2DoAddLegacyUser(int siteid, int legacyid, string salt, ELegacyEncryptionType encryptionType,
                            string username,
                            string password,
                            string firstName,
                            string lastname,
                            string gender,
                            string country,
                            string region,
                            string city,
                            string email,
                            bool newsletter,
                            string redirect,
                            DateTime dateRegistered,
                            string sessionID)
        {

            if (ReadOnly)
            {
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            if (DoIsEmailAddressInUse(email))
            { //Check the email address
                return new AddUserResult(EAddUserStatus.EmailInUse);
            }

            if (DoIsUserNameInUse(username))
            { //Check the username
                return new AddUserResult(EAddUserStatus.UsernameInUse);
            }

            if (gender == "0")
            { // I dont know why they are doing this
                gender = "";
            }

            string message;
            if (!AuthUser.TryValidateProfile(firstName, lastname, gender, country, region, city, email, out message))
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile, message);
            }

            if (!InputValidation.IsValidUserNameWithNoLeadingOrTrailingUnderscores(username))
            { //Validate the username
                return new AddUserResult(EAddUserStatus.InvalidUsername);
            }

            var site = NetworkSite.GetByID(siteid);
            if (site == null)
            { // Check that the site is valid
                return new AddUserResult(EAddUserStatus.InvalidNetworkSite);
            }

            User user = null;
            using (var conn = DatabaseUtility.GetAuthConnection())
            {
                user = AuthUser.CreateLegacy(conn, siteid, legacyid, salt, encryptionType, username, password, email, firstName, lastname, gender, country, city, region, newsletter, dateRegistered);
                if (!string.IsNullOrEmpty(sessionID))
                {
                    user.CreateSession(conn, sessionID, siteid);
                }
            }

            if (user == null)
            {
                return new AddUserResult(EAddUserStatus.InvalidProfile);
            }
            else
            {
                return new AddUserResult(EAddUserStatus.Successful, user.ID, sessionID);
            }
        }

        private DeleteUserResult DoDeleteUser(int siteid, int userid)
        {
            var site = NetworkSite.GetByID(siteid);
            if (site == null)
            {
                return new DeleteUserResult(EDeleteUserStatus.InvalidNetworkSite);
            }

            try
            {
                var user = AuthUser.GetByID(userid);
                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    user.Delete(sql);
                }
                return new DeleteUserResult(EDeleteUserStatus.Success);
            }
            catch (KeyNotFoundException)
            {
                return new DeleteUserResult(EDeleteUserStatus.UserNotFound);
            }
        }

        private EoAuthDataStatus DoAddUseroAuthData(int userId, int oAuthPlatform, string oAuthId)
        {
            try
            {
                var user = AuthUser.GetByID(userId);
                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    user.AddUseroAuthData(userId, oAuthPlatform, oAuthId, sql);
                }
                return EoAuthDataStatus.Success;
            }
            catch (KeyNotFoundException)
            {
                return EoAuthDataStatus.UserNotFound;
            }
            catch (Exception)
            {
                return EoAuthDataStatus.Unsuccessful;
            }
        }

        private EAddSubscriptionCreditStatus DoAddSubscriptionCredit(int userid, byte months, byte level, byte type)
        {
            try
            {
                var user = AuthUser.GetByID(userid);
                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    user.AddSubscriptionCredit(sql, months, level, type);
                }

                return EAddSubscriptionCreditStatus.Success;
            }
            catch (Exception exc)
            {
                if (exc is KeyNotFoundException)
                {
                    return EAddSubscriptionCreditStatus.UserNotFound;
                }
                return EAddSubscriptionCreditStatus.Unsuccessful;
            }
        }

        private ESetPremiumExpirationStatus DoSetPremiumExpiration(int userid, DateTime expiration, byte level)
        {
            try
            {
                var user = AuthUser.GetByID(userid);
                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    user.SetPremiumExpiration(sql, expiration, level);
                }
                return ESetPremiumExpirationStatus.Success;
            }
            catch (Exception exc)
            {
                if (exc is KeyNotFoundException)
                {
                    return ESetPremiumExpirationStatus.UserNotFound;
                }
                return ESetPremiumExpirationStatus.Unsuccessful;
            }
        }
        #endregion

        #region Legacy Web Methods

        [WebMethod]
        public bool userIdExists(int siteid, int userid)
        {
            string message = null;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                AuthUser.GetByID(userid);
            }
            catch (KeyNotFoundException)
            {
                return false;
            }

            return true;
        }


#if DEBUG

        [WebMethod]
        public void testSendActivationEmail(string username, string emailAddress, string redirect)
        {
            var activationCode = DoHashEmail(emailAddress);
            var emailId = UserEmail.GetIDByEmailAddress(emailAddress);
            var activationLink = string.Format("{0}/activateEmail?emailid={1}&code={2}&redirect={3}", ServiceUrl, emailId, HttpUtility.UrlEncodeUnicode(activationCode), HttpUtility.UrlEncodeUnicode(redirect));

            UserMailer.Instance.SendAccountActivation(emailAddress, username, activationLink, activationCode);
        }

        [WebMethod]
        public void testSendPasswordResetRequestEmail(string username, string emailAddress, string redirect)
        {

            var emailId = UserEmail.GetIDByEmailAddress(emailAddress);
            var code = DoHashEmail(emailAddress.ToLower());
            var passwordResetLink = string.Format("{0}/resetPassword?siteid={1}&emailid={2}&code={3}&redirect={4}", ServiceUrl, 1, emailId, HttpUtility.UrlEncodeUnicode(code), HttpUtility.UrlEncodeUnicode(redirect));

            UserMailer.Instance.SendLegacyPasswordResetRequest(emailAddress, username, passwordResetLink);
        }
#endif

        [WebMethod]
        public string getEmailActivationCode(int siteid, string address)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return String.Empty;
            }

            return DoHashEmail(address);
        }

        [WebMethod]
        public void activateEmailByCode(string code)
        {
            string[] split = code.Split(new char[] { '_' }, 2);

            if (split.Length != 2)
            {
                EndWithClientError("Invalid activation code.");
                return;
            }

            int emailid;
            try
            {
                emailid = Convert.ToInt32(split[0]);
            }
            catch (FormatException)
            {
                EndWithClientError("Invalid email id");
                return;
            }

            DoActivateEmailAddress(emailid, split[1]);
        }

        [WebMethod]
        public void activateEmail(int emailid, string code, string redirect)
        {
            if (!DoActivateEmailAddress(emailid, code))
            {
                return;
            }

            if (!IsTrustedDomain(redirect, true, new { redirect, HttpContext.Current.Request.UrlReferrer, method = "activateEmail" }))
            {
                redirect = "https://www.curse.com";
            }

            Context.Response.Redirect(redirect, true);
        }

        [WebMethod]
        public void sendActivationEmailByAddress(int siteid, string emailaddress, string from, string redirect)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                SimpleResponse((int)ESendActivationResult.UnknownFailure);
                return;
            }

            try
            {
                var user = AuthUser.GetByEmailAddress(emailaddress);
                DoResendActivationEmail(user.Name, user.Profile.PrimaryEmail, from, redirect, false);
            }
            catch (KeyNotFoundException)
            {
                SimpleResponse((int)ESendActivationResult.EmailNotFound);
            }

        }

        [WebMethod]
        public void sendActivationEmail(int siteid, int emailid, string from, string redirect)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                SimpleResponse((int)ESendActivationResult.UnknownFailure);
                return;
            }

            try
            {
                var email = UserEmail.GetByID(emailid);
                var user = AuthUser.GetByEmailAddress(email.Address);
                DoResendActivationEmail(user.Name, emailid, from, redirect, false);
            }
            catch (KeyNotFoundException) { }
        }

        [WebMethod]
        public void addUser(int siteid,
                            string username,
                            string password,
                            string firstname,
                            string lastname,
                            string gender,
                            string country,
                            string region,
                            string city,
                            string email,
                            bool newsletter,
                            string from,
                            string redirect,
                            string session)
        {

            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var result = DoAddUser(siteid, username, password, firstname, lastname, gender, country, region, city, email, newsletter, redirect, session);

            switch (result.Status)
            {
                case EAddUserStatus.Successful:
                    SimpleResponse(result.UserID);
                    break;
                case EAddUserStatus.EmailInUse:
                    EndWithClientError("Email address is already in use.");
                    return;
                case EAddUserStatus.UsernameInUse:
                    EndWithClientError("Username is already in use.");
                    return;
                case EAddUserStatus.InvalidNetworkSite:
                    EndWithClientError("No such network site.");
                    return;
                case EAddUserStatus.InvalidPassword:
                    EndWithClientError("Invalid password.");
                    return;
                case EAddUserStatus.InvalidUsername:
                    EndWithClientError("Invalid user name.");
                    return;
                case EAddUserStatus.InvalidProfile:
                    EndWithClientError("Invalid profile.");
                    return;
            }

        }

        [WebMethod]
        public void isLegacyUser(int siteid, int uid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            try
            {
                var isLegacy = false;
                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    using (var cmd = sql.CreateCommand())
                    {
                        cmd.CommandText = "SELECT COUNT(0) FROM userlegacy WHERE _uid=@uid";
                        cmd.Parameters.AddWithValue("uid", uid);
                        isLegacy = (int)cmd.ExecuteScalar() > 0;
                    }
                }
                SimpleResponse(isLegacy);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("Invalid user.");
            }
        }

        [WebMethod]
        public void canRenameUser(int siteid, int uid)
        {

            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var canRename = DoCanRenameUser(siteid, uid);
            SimpleResponse(canRename);
        }

        [WebMethod]
        public void renameUser(int siteid, int uid, string username)
        {

            var status = DoRenameUser(siteid, uid, username);

            switch (status)
            {
                case ERenameUserStatus.InsufficientPermissions:
                    return;

                case ERenameUserStatus.UserNotFound:
                    EndWithClientError("Invalid user.");
                    return;

                case ERenameUserStatus.InvalidUsername:
                    EndWithClientError("Invalid user name.");
                    return;

                case ERenameUserStatus.UsernameInUse:
                    EndWithClientError("User name already used.");
                    return;

                case ERenameUserStatus.NoRenameCredit:
                    EndWithClientError("User is not credited for a rename.");
                    return;
            }

        }

        [WebMethod]
        public ERenameUserStatus v2RenameUser(int siteID, int userID, string newUsername)
        {
            return DoRenameUser(siteID, userID, newUsername);
        }

        [WebMethod]
        public ERenameUserDeclinedStatus v2RenameUserDeclined(int siteID, int userID)
        {
            string message;
            if (!TryCanHostAlterUser(siteID, out message))
            {
                return ERenameUserDeclinedStatus.InsufficientPermissions;
            }

            User user = null;
            try
            {
                user = AuthUser.GetByID(userID);
            }
            catch (KeyNotFoundException)
            {
                return ERenameUserDeclinedStatus.UserNotFound;
            }

            using (var conn = DatabaseUtility.GetAuthConnection())
            {
                user.DeclineRename(conn);
            }

            return ERenameUserDeclinedStatus.Success;
        }

        private ERenameUserStatus DoRenameUser(int siteID, int userID, string newUsername)
        {
            string message;
            if (!TryCanHostAlterUser(siteID, out message))
            {
                return ERenameUserStatus.InsufficientPermissions;
            }

            User user = null;

            try
            {
                user = AuthUser.GetByID(userID);
            }
            catch (KeyNotFoundException)
            {
                return ERenameUserStatus.UserNotFound;
            }

            if (!InputValidation.IsValidUserNameWithNoLeadingOrTrailingUnderscores(newUsername))
            {
                return ERenameUserStatus.InvalidUsername;
            }

            if (newUsername.ToLower() != user.Name.ToLower() && DoIsUserNameInUse(newUsername))
            {
                return ERenameUserStatus.UsernameInUse;
            }

            using (var conn = DatabaseUtility.GetAuthConnection())
            {
                if (!user.Rename(newUsername, conn, false))
                {
                    return ERenameUserStatus.NoRenameCredit;
                }
            }

            return ERenameUserStatus.Success;
        }

        [WebMethod]
        public void validateUserName(int siteid, string username)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            SimpleResponse(DoIsUserNameInUse(username));
        }

        [WebMethod]
        public void getUserSubscriptions(int siteid, int uid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            try
            {
                var writer = BeginComplexResponse();
                var user = AuthUser.GetByID(uid);
                user.SerializeSubscriptions(writer);
                EndComplexResponse(writer);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such user.");
            }

        }

        [WebMethod]
        public void getSubscriptionStatus(int siteid, int uid, ESubscriptionType subtype)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            try
            {
                var user = AuthUser.GetByID(uid);
                var subscriptionStatus = user.GetSubscriptionStatus(subtype);
                if (!subscriptionStatus)
                {
                    subscriptionStatus = user.Entitlements.Any(p => p.Type == subtype && p.Expires > DateTime.UtcNow);
                }

                SimpleResponse(subscriptionStatus ? 1 : 0);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("Invalid uid");
            }
        }

        [WebMethod]
        public void getPremiumLevel(int siteid, int uid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            try
            {
                var user = AuthUser.GetByID(uid);
                SimpleResponse(user.GetPremiumLevel());
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("Invalid uid");
            }
        }

        [WebMethod]
        public string getUserNameFromUserId(int siteid, int userid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return null;
            }
            try
            {
                return AuthUser.GetByID(userid).Name;
            }
            catch (KeyNotFoundException)
            {
                return null;
            }
        }

        [WebMethod]
        public string getUserEmailFromUserId(int siteid, int userid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return null;
            }
            try
            {
                return AuthUser.GetByID(userid).PrimaryEmail.Address;
            }
            catch (KeyNotFoundException)
            {
                return null;
            }

        }

        [WebMethod]
        public void getUserIdFromUsername(int siteid, string username)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            SimpleResponse(AuthUser.GetIDByName(username));
        }

        [WebMethod]
        public void getUserIdFromEmail(int siteid, string email)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            SimpleResponse(AuthUser.GetIDByEmailAddress(email));
        }

        [WebMethod]
        public void validateClientUserSession(int siteid, string session)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            try
            {
                var user = AuthUser.GetBySessionID(session);
                if (!user.SessionIsSecure)
                {
                    EndWithClientError("Session is not secure.");
                    return;
                }
                SimpleResponse(user.ID);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such session.");
            }
        }

        [WebMethod]
        public void validateClientUser(int siteid, string username, string password)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            var loginResult = DoLogin(SessionType.Client, siteid, username, password, "");
            if (loginResult.Status == ELoginStatus.Success)
            {
                SimpleResponse(loginResult.UserId, loginResult.SessionId, loginResult.SubscriptionLevel);
                return;
            }

            LegacyDoHandleBadLogin(loginResult.Status);

        }

        [WebMethod]
        public void getUserProfile(int siteid, int userid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            try
            {
                var writer = BeginComplexResponse();
                var user = AuthUser.GetByID(userid);
                user.Serialize(writer);
                EndComplexResponse(writer);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such user.");
            }
        }

        [WebMethod]
        public void validateInsecureClientUser(int siteid, string username, string password)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;

            }

            var loginResult = DoLogin(SessionType.Client, siteid, username, password, "");

            if (loginResult.Status == ELoginStatus.Success)
            {
                SimpleResponse(loginResult.UserId, loginResult.SessionId, loginResult.SubscriptionLevel);
            }
            else
            {
                LegacyDoHandleBadLogin(loginResult.Status);
            }
        }

        [WebMethod]
        public void validateUser(int siteid, string username, string password, string session)
        {
            if (!LegacyIsHostNetworkSite(siteid))
            {
                return;
            }

            var loginResult = DoLogin(SessionType.Web, siteid, username, password, session);

            if (loginResult.Status == ELoginStatus.Success)
            {
                SimpleResponse(loginResult.UserId);
            }
            else
            {
                LegacyDoHandleBadLogin(loginResult.Status);
            }
        }

        [WebMethod]
        public void validateUserSession(int siteid, string session)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            try
            {
                SimpleResponse(AuthUser.GetBySessionID(session).ID);
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such session.");
            }

        }

        [WebMethod]
        public void adminSetUserPassword(int siteid, int userid, string newpass)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var status = DoAdminSetUserPassword(siteid, userid, newpass);

            switch (status)
            {
                case EAdminSetUserPasswordStatus.Success:
                    break;
                case EAdminSetUserPasswordStatus.InsufficientPermissions:
                    break;
                case EAdminSetUserPasswordStatus.UserNotFound:
                    EndWithClientError("No such user.");
                    break;
                case EAdminSetUserPasswordStatus.InvalidPassword:
                    EndWithClientError("Invalid password.");
                    break;
                case EAdminSetUserPasswordStatus.Unsuccessful:
                    EndWithClientError("Unknown Error");
                    break;

            }
        }

        [WebMethod]
        public void adminDeleteUser(int siteid, int userid)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            try
            {
                var user = AuthUser.GetByID(userid);

                if (user.Subscriptions.Count > 0)
                {
                    EndWithClientError("User cannot be deleted. They have subscriptions.");
                    return;
                }

                using (var sql = DatabaseUtility.GetAuthConnection())
                {
                    user.Delete(sql);
                }
            }
            catch (KeyNotFoundException)
            {
                EndWithClientError("No such user.");
            }
        }

        [WebMethod]
        public void setUserPassword(int siteid, int userid, string oldpass, string newpass)
        {

            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var result = DoSetUserPassword(siteid, userid, oldpass, newpass);

            switch (result)
            {
                case ESetUserPasswordResult.UserNotFound:
                case ESetUserPasswordResult.InvalidPassword:
                    EndWithClientError("No such user and password.");
                    return;
            }
        }

        [WebMethod]
        public void setUserProfileInfo(int siteid, int userid, string firstname, string lastname, string gender, string country, string region, string city, int primaryemail, bool newsletter)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var result = this.DoSetUserProfileInfo(userid, firstname, lastname, gender, country, region, city, primaryemail, newsletter);

            if (result == ESetUserProfileResult.Unsuccessful)
            {
                return;
            }
            else if (result == ESetUserProfileResult.EmailNotFound)
            {
                EndWithClientError("Invalid email ID.");
                return;
            }
        }

        [WebMethod]
        public void signOutUser(int siteid, int userid)
        {
            if (!LegacyIsHostNetworkSite(siteid))
                return;

            if (!DoSignOutUser(siteid, userid))
            {
                EndWithClientError("User not logged in.");
            }

        }

        [WebMethod]
        public void updatePrimaryEmailAddress(int siteid, int userid, string emailaddress)
        {
            if (!LegacyCanHostAlterUser(siteid))
            {
                return;
            }

            var result = DoUpdatePrimaryEmailAddress(siteid, userid, emailaddress);

            switch (result)
            {
                case EUpdatePrimaryEmailStatus.InsufficientPermissions:
                    return;
                case EUpdatePrimaryEmailStatus.Successful:
                    return;
                case EUpdatePrimaryEmailStatus.EmailInUse:
                    EndWithClientError("Email address is already in use.");
                    break;
                case EUpdatePrimaryEmailStatus.InvalidEmail:
                    EndWithClientError("Invalid e-mail address.");
                    break;
                case EUpdatePrimaryEmailStatus.UnknownUser:
                    EndWithClientError("No such user.");
                    break;
                case EUpdatePrimaryEmailStatus.UnknownError:
                default:
                    EndWithClientError("Unknown error.");
                    break;
            }
        }

        #endregion

        #region Soap Web Methods

        [WebMethod]
        public ESetUserProfileResult v2SetUserProfileInfo(int siteid, int userid, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, int primaryemail, bool newsletter)
        {
            string message = null;

            if (!TryCanHostAlterUser(siteid, out message))
            {
                return ESetUserProfileResult.Unsuccessful;
            }

            return DoSetUserProfileInfo(userid, firstname, lastname, gender, country, region, city, primaryemail, newsletter);
        }

        [WebMethod]
        public ESetUserProfileAndEmailResult v3SetUserProfileInfo(int siteid, int userid, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter)
        {
            string message = null;

            if (!TryCanHostAlterUser(siteid, out message))
            {
                return ESetUserProfileAndEmailResult.Unsuccessful;
            }

            return v2DoSetUserProfileInfo(userid, firstname, lastname, gender, country, region, city, email, newsletter);
        }

        [WebMethod]
        public ESetUserPasswordResult v2SetUserPassword(int siteid, int userid, string oldpass, string newpass)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return ESetUserPasswordResult.Unsuccessful;
            }

            return DoSetUserPassword(siteid, userid, oldpass, newpass);
        }

        [WebMethod]
        public EAdminSetUserPasswordStatus v2AdminSetUserPassword(int siteid, int userid, string newpass)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return EAdminSetUserPasswordStatus.InsufficientPermissions;
            }

            return DoAdminSetUserPassword(siteid, userid, newpass);

        }

        [WebMethod]
        public EAdminSetUserPasswordStatus v2AdminSetBinaryUserLegacyPassword(int siteid, int legacyid, int userid, string passwordHash)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return EAdminSetUserPasswordStatus.InsufficientPermissions;
            }

            return DoAdminSetBinaryUserLegacyPassword(siteid, legacyid, userid, passwordHash);
        }

        [WebMethod]
        public EAdminCreditUserWithRenameStatus v2AdminCreditUserWithRename(int siteID, int userID)
        {
            string message;
            if (!TryCanHostAlterUser(siteID, out message))
            {
                return EAdminCreditUserWithRenameStatus.InsufficientPermissions;
            }


            User user = null;
            try
            {
                user = AuthUser.GetByID(userID);
            }
            catch (KeyNotFoundException)
            {
                return EAdminCreditUserWithRenameStatus.UserNotFound;
            }

            var successful = false;
            using (var conn = DatabaseUtility.GetAuthConnection())
            {
                successful = user.CreditRename(conn);
            }

            return successful ? EAdminCreditUserWithRenameStatus.Success : EAdminCreditUserWithRenameStatus.AlreadyCredited;
        }

        [WebMethod]
        public EAdminClearUserCacheStatus v2AdminClearUserCache(int siteid, int userid)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return EAdminClearUserCacheStatus.InsufficientPermissions;
            }

            User user = null;
            try
            {
                user = AuthUser.GetByID(userid);
            }
            catch (KeyNotFoundException)
            {
                return EAdminClearUserCacheStatus.UserNotFound;
            }


            user.ClearCache();


            return EAdminClearUserCacheStatus.Success;
        }

        [WebMethod]
        public User v2GetUserProfile(int siteid, int userid)
        {
            try
            {
                string message;
                if (!TryIsHostNetworkSite(siteid, out message))
                {
                    return null;
                }

                if (userid < 1)
                {
                    Logger.Trace("Site " + siteid + " supplied an invalid user ID while calling v2GetUserProfile");
                    return null;
                }

                return AuthUser.GetByID(userid);
            }
            catch (KeyNotFoundException)
            {
                return null;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2GetUserProfile");
                throw;
            }
        }

        [WebMethod]
        public LoginResult v2ValidateClientUser(int siteid, string username, string password)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return new LoginResult(ELoginStatus.UnauthorizedLogin);
            }

            try
            {
                return DoLogin(SessionType.Client, siteid, username, password, "");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2ValidateClientUser");
                throw;
            }

        }

        [WebMethod]
        public LoginResult v2ValidateWebUser(int siteid, string username, string password, string session)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return new LoginResult(ELoginStatus.UnauthorizedLogin);
            }

            try
            {
                return DoLogin(SessionType.Web, siteid, username, password, session);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2ValidateWebUser");
                throw;
            }

        }

        [WebMethod]
        public Int32 v2ValidateUserSession(int siteid, string session)
        {

            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return -1;
            }

            try
            {
                return AuthUser.GetIDBySessionID(session, true);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2ValidateUserSession");
                throw;
            }
        }

        [WebMethod]
        public EAdminRenameUserStatus v2AdminRenameUser(int siteid, int uid, string username)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return EAdminRenameUserStatus.InsufficientPermissions;
            }

            return DoAdminRenameUser(siteid, uid, username);
        }

        [WebMethod]
        public bool v2SignOutUser(int siteid, int userid)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                return DoSignOutUser(siteid, userid);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2SignOutUser");
                throw;
            }
        }

        [WebMethod]
        public EUpdatePrimaryEmailStatus v2UpdatePrimaryEmailAddress(int siteid, int userid, string emailaddress)
        {
            string message = null;

            if (!TryCanHostAlterUser(siteid, out message))
            {
                return EUpdatePrimaryEmailStatus.InsufficientPermissions;

            }
            return DoUpdatePrimaryEmailAddress(siteid, userid, emailaddress);
        }

        [WebMethod]
        public int v2GetUserIDFromEmail(int siteid, string email)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return 0;
            }

            try
            {
                return AuthUser.GetIDByEmailAddress(email);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2GetUserIDFromEmail");
                throw;
            }
        }

        [WebMethod]
        public bool v2GetSubscriptionStatus(int siteid, int uid, ESubscriptionType subtype)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                var user = AuthUser.GetByID(uid);
                var legacySubscruptionStatus = user.GetSubscriptionStatus(subtype);

                if (legacySubscruptionStatus)
                {
                    return legacySubscruptionStatus;
                }

                return user.Entitlements.Any(p => p.Type == subtype && p.Active);
            }
            catch (KeyNotFoundException)
            {
                return false;
            }

        }

        [WebMethod]
        public AccountRecoveryResult v2RecoverAccount(int siteID, string usernameOrEmail, string redirectUrl)
        {
            string message;

            if (!TryCanHostAlterUser(siteID, out message))
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InsufficientPermissions, message);
            }

            try
            {
                return DoRecoverAccount(usernameOrEmail, redirectUrl);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2RecoverAccount");
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownError, ex.Message);
            }
        }

        [WebMethod]
        public AccountRecoveryResult v2GetAccountRecoveryRequestByToken(int siteID, Int32 token)
        {
            string message;

            if (!TryCanHostAlterUser(siteID, out message))
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InsufficientPermissions, message);
            }

            try
            {
                return DoGetAccountRecoveryRequestByToken(token);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2GetAccountRecoveryRequestByToken");
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownError, ex.Message);
            }

        }

        [WebMethod]
        public AccountRecoveryResult v2GetAccountRecoveryRequestByUserID(int siteID, Int32 userID)
        {
            string message;

            if (!TryCanHostAlterUser(siteID, out message))
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InsufficientPermissions, message);
            }

            try
            {
                return DoGetAccountRecoveryRequestByUserID(userID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2GetAccountRecoveryRequestByUserID");
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownError, ex.Message);
            }

        }

        [WebMethod]
        public AccountRecoveryResult v2ResetPassword(int siteID, Int32 token, string encryptedPassword, string encryptedConfirmPassword)
        {
            string message;

            if (!TryCanHostAlterUser(siteID, out message))
            {
                return new AccountRecoveryResult(EAccountRecoveryStatus.InsufficientPermissions, message);
            }

            try
            {
                return DoResetPassword(siteID, token, encryptedPassword, encryptedConfirmPassword);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in v2ResetPassword");
                return new AccountRecoveryResult(EAccountRecoveryStatus.UnknownError);
            }
        }

        [WebMethod]
        public ERecoverUsernameStatus v2RecoverUsername(int siteID, string emailAddress)
        {
            string message;
            if (!TryCanHostAlterUser(siteID, out message))
            {
                return ERecoverUsernameStatus.InsufficientPermissions;
            }

            try
            {
                return DoRecoverUsername(siteID, emailAddress);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled Exception in v2RecoverUsername");
                return ERecoverUsernameStatus.UnknownError;
            }
        }

        [WebMethod]
        public AddUserResult v2AddUser(int siteid, string username, string password, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter, string from, string redirect, string session)
        {
            string message;

            if (!TryCanHostAlterUser(siteid, out message))
            {
                return new AddUserResult(EAddUserStatus.InsufficientPermissions, message);
            }

            try
            {
                return DoAddUser(siteid, username, password, firstname, lastname, gender, country, region, city, email, newsletter, redirect, session);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled Exception in v2AddUser");
                throw;
            }

        }

        [WebMethod]
        public AddUserResult v2AdminAddLegacyUser(int siteid, int legacyid, string username, string passwordHash, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter, string from, string redirect, string session)
        {
            string message;

            if (!TryCanHostAlterUser(siteid, out message))
            {
                return new AddUserResult(EAddUserStatus.InsufficientPermissions, message);
            }

            return DoAddLegacyUser(siteid, legacyid, username, passwordHash, firstname, lastname, gender, country, region, city, email, newsletter, redirect, DateTime.UtcNow, session);
        }

        [WebMethod]
        public AddUserResult v3AdminAddLegacyUser(int siteId, int legacyId, string salt, ELegacyEncryptionType encryptionType, string username, string passwordHash, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter, string from, string redirect, DateTime dateRegistered, string session)
        {
            string message;

            if (!TryCanHostAlterUser(siteId, out message))
            {
                return new AddUserResult(EAddUserStatus.InsufficientPermissions, message);
            }

            return v2DoAddLegacyUser(siteId, legacyId, salt, encryptionType, username, passwordHash, firstname, lastname, gender, country, region, city, email, newsletter, redirect, dateRegistered, session);
        }

        [WebMethod]
        public bool v2ValidateUsername(int siteid, string username)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            return DoIsUserNameInUse(username);
        }

        [WebMethod]
        public bool v2IsUsernameAvailable(int siteid, string username)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            return !DoIsUserNameInUse(username);
        }

        [WebMethod]
        public int v2GetUserIDFromUsername(int siteid, string username)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return 0;
            }

            return AuthUser.GetIDByName(username);
        }

        [WebMethod]
        public List<int> v2GetDeletedUsersFromTime(DateTime deletedAfter)
        {
            if (deletedAfter < new DateTime(2000, 1, 1))
            {
                deletedAfter = DateTime.Parse("01/01/2000");
            }

            var deletedUsersIDs = new List<int>();
            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = new SqlCommand("select _uid from userdeletion where _date >= @deletedAfter", conn))
                {
                    cmd.Parameters.Add("deletedAfter", SqlDbType.DateTime).Value = deletedAfter;

                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            deletedUsersIDs.Add(reader.GetInt32(0));
                        }
                    }
                }
            }

            return deletedUsersIDs;
        }

        [WebMethod]
        public EAddSubscriptionCreditStatus v2AddSubscriptionCredit(int siteid, int userid, byte months, byte level, byte type)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return EAddSubscriptionCreditStatus.InvalidNetworkSite;
            }

            return DoAddSubscriptionCredit(userid, months, level, type);
        }

        [WebMethod]
        public ESetPremiumExpirationStatus v2SetPremiumExpiration(int siteid, int userid, DateTime expiration, byte level)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return ESetPremiumExpirationStatus.InvalidNetworkSite;
            }

            return DoSetPremiumExpiration(userid, expiration, level);
        }

        [WebMethod]
        public EValidateEmailStatus v2ActivateEmailAddress(int siteId, int emailId, string code)
        {
            string message;
            if (!TryIsHostNetworkSite(siteId, out message))
            {
                return EValidateEmailStatus.InvalidNetworkSite;
            }

            return DoActivateEmailAddressV2(emailId, code);
        }

        [WebMethod]
        public EoAuthDataStatus v2AddUserOAuthAccount(int siteId, int userId, int oAuthPlatform, string oAuthId)
        {
            string message;
            if (!TryIsHostNetworkSite(siteId, out message))
            {
                return EoAuthDataStatus.InvalidNetworkSite;
            }

            return DoAddUseroAuthData(userId, oAuthPlatform, oAuthId);
        }

        [WebMethod]
        public User v2GetUserOAuthAccount(int siteId, int oAuthPlatform, string oAuthId)
        {
            string message;
            if (!TryIsHostNetworkSite(siteId, out message))
            {
                return null;
            }

            return AuthUser.GetByOAuthAccount(oAuthPlatform, oAuthId);
        }

        [WebMethod]
        [ScriptMethod(UseHttpGet = false)]
        public EUpdatePolicyAcceptedStatus v2UpdatePolicyAccepted(int siteid, int userid)
        {
            string message;
            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return EUpdatePolicyAcceptedStatus.InvalidNetworkSite;
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(userid).AcceptPolicy(sql);
                }

                return EUpdatePolicyAcceptedStatus.Success;
            }
            catch (KeyNotFoundException)
            {
                return EUpdatePolicyAcceptedStatus.Unsuccessful;
            }
        }


        #region Admin Methods

        [WebMethod]
        public int v2GetMailQueueCount(string adminkey)
        {
            string message = null;
            if (!TryRequireAdminKey(adminkey, out message))
            {
                return -1;
            }

            return MailProvider.Instance.Statistics.MailQueueCount;
        }

        [WebMethod]
        public GenericAuthenticationResult v2RemoveNetworkSite(string adminkey, int siteid)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            try
            {
                var site = NetworkSite.GetByID(siteid);

                using (var conn = DatabaseUtility.GetAuthConnection())
                {
                    site.Delete(conn);
                }

            }
            catch (KeyNotFoundException)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.Unsuccessful, "No site exists with the id '{0}'".FormatWith(siteid.ToString()));
            }
            catch (Exception ex)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.UnknownError, ex.Message);
            }


            return new GenericAuthenticationResult(EGenericAuthenticationStatus.Successful, "Site '{0}' has been deleted.".FormatWith(siteid.ToString()));
        }

        [WebMethod]
        public GenericAuthenticationResult v2UnbanIPAddress(string adminkey, string ipAddress)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            if (!InputValidation.IsValidIp(ipAddress))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.FailedValidation, "The IP address supplied, '{0}', is invalid.".FormatWith(ipAddress));
            }

            try
            {
                UnbanIPAddress(ipAddress);
            }
            catch (Exception ex)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.UnknownError, ex.Message);
            }
            return new GenericAuthenticationResult(EGenericAuthenticationStatus.Successful, "The IP address '{0}' has been banned.".FormatWith(ipAddress));

        }

        [WebMethod]
        public GenericAuthenticationResult v2AddNetworkSiteIP(string adminkey, int siteID, string ipAddress)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            if (!InputValidation.IsValidIp(ipAddress))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.FailedValidation, "The IP address supplied, '{0}', is invalid.".FormatWith(ipAddress));
            }

            try
            {
                NetworkSite site = NetworkSite.GetByID(siteID);

                if (site.ValidateIPAddress(ipAddress))
                {
                    return new GenericAuthenticationResult(EGenericAuthenticationStatus.Unsuccessful, "The IP address supplied, '{0}', has already been added to this site.".FormatWith(ipAddress));
                }

                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    site.AddIPAddressToDatabase(sql, ipAddress, null);
                }
            }
            catch (KeyNotFoundException)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InvalidNetworkSite, "The site ID requested, '{0}', does not exist.".FormatWith(siteID.ToString()));
            }
            catch (Exception ex)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.UnknownError, ex.Message);
            }

            return new GenericAuthenticationResult(EGenericAuthenticationStatus.Successful);
        }

        [WebMethod]
        public GenericAuthenticationResult v2RemoveNetworkSiteIP(string adminkey, int siteID, string ipAddress)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            if (!InputValidation.IsValidIp(ipAddress))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.FailedValidation, "The IP address supplied, '{0}', is invalid.".FormatWith(ipAddress));
            }

            try
            {
                NetworkSite site = NetworkSite.GetByID(siteID);

                if (!site.ValidateIPAddress(ipAddress))
                {
                    return new GenericAuthenticationResult(EGenericAuthenticationStatus.Unsuccessful, "The IP address supplied, '{0}', is not in the trusted IP list for this site.".FormatWith(ipAddress));
                }

                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    site.RemoveIPAddressFromDatabase(sql, ipAddress);
                }
            }
            catch (KeyNotFoundException)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InvalidNetworkSite, "The site ID requested, '{0}', does not exist.".FormatWith(siteID.ToString()));
            }
            catch (Exception ex)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.UnknownError, ex.Message);
            }

            return new GenericAuthenticationResult(EGenericAuthenticationStatus.Successful);
        }

        [WebMethod]
        public ObjectAuthenticationResult<NetworkSite> v2GetNetworkSite(string adminkey, int siteid)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new ObjectAuthenticationResult<NetworkSite>(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            try
            {
                return new ObjectAuthenticationResult<NetworkSite>(EGenericAuthenticationStatus.Successful, NetworkSite.GetByID(siteid));
            }
            catch (KeyNotFoundException)
            {
                return new ObjectAuthenticationResult<NetworkSite>(EGenericAuthenticationStatus.InvalidNetworkSite);
            }
        }

        [WebMethod]
        public ObjectAuthenticationResult<NetworkSite[]> v2GetNetworkSites(int siteid)
        {
            string message;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return new ObjectAuthenticationResult<NetworkSite[]>(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }
            return new ObjectAuthenticationResult<NetworkSite[]>(EGenericAuthenticationStatus.Successful, NetworkSite.GetAll());

        }

        [WebMethod]
        public AddNetworkSiteResult v2AddNetworkSite(string adminkey,
                                     int id,
                                     string name,
                                     string ip,
                                     string referrer,
                                     string updateurl,
                                     string url,
                                     string iconurl,
                                     bool alteruser,
                                     string email)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new AddNetworkSiteResult(EAddNetworkSiteStatus.InsufficientPermissions, message);

            }

            if (!NetworkSite.TryValidateSiteInfo(name, ip, referrer, updateurl, url, iconurl, email, out message))
            {
                return new AddNetworkSiteResult(EAddNetworkSiteStatus.FailedValidation, message);
            }

            if (NetworkSite.ExistsByID(id))
            {
                return new AddNetworkSiteResult(EAddNetworkSiteStatus.AlreadyExists);
            }

            NetworkSite site = null;

            using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
            {
                try
                {
                    site = NetworkSite.Create(sql, id, name, ip, referrer, updateurl, url, iconurl, alteruser, email);
                }
                catch (Exception ex)
                {
                    return new AddNetworkSiteResult(EAddNetworkSiteStatus.UnknownError, ex.Message);
                }
            }

            if (site == null)
            {
                return new AddNetworkSiteResult(EAddNetworkSiteStatus.UnknownError);
            }
            else
            {
                return new AddNetworkSiteResult(EAddNetworkSiteStatus.Successful, site);
            }

        }

        [WebMethod]
        public GenericAuthenticationResult v2SetUserLevel(string adminkey, int uid, int level)
        {
            string message = null;

            if (!TryRequireAdminKey(adminkey, out message))
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InsufficientPermissions, message);
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(uid).SetLevel(sql, level);
                }
            }
            catch (KeyNotFoundException)
            {
                return new GenericAuthenticationResult(EGenericAuthenticationStatus.InvalidUser);
            }

            return new GenericAuthenticationResult(EGenericAuthenticationStatus.Successful);
        }

        [WebMethod]
        public DeleteUserResult v2AdminDeleteUser(int siteid, int userid)
        {
            string message;
            if (!TryCanHostAlterUser(siteid, out message))
            {
                return new DeleteUserResult(EDeleteUserStatus.InsufficientPermissions, message);
            }

            return DoDeleteUser(siteid, userid);
        }//v2AdminDeleteUser
        #endregion

        #region Subscription Management Methods

        [WebMethod]
        public bool setPremiumExpiration(int siteid, int userid, DateTime expiration, Byte level)
        {
            string message = null;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(userid).SetPremiumExpiration(sql, expiration, level);
                }

                return true;
            }
            catch (KeyNotFoundException)
            {
                return false;
            }
        }

        [WebMethod]
        public bool addSubscriptionCredit(int siteid, int userid, Byte months, Byte level, Byte type)
        {
            string message = null;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(userid).AddSubscriptionCredit(sql, months, level, type);
                }
            }
            catch (KeyNotFoundException)
            {
                return false;
            }

            return true;
        }

        [WebMethod]
        public bool terminateSubscription(int siteid, int userid, Byte type)
        {
            string message = null;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(userid).TerminateSubscription(sql, type);
                }
            }
            catch (KeyNotFoundException)
            {
                return false;
            }

            return true;
        }

        [WebMethod]
        public bool cancelPremium(int siteid, int userid, DateTime expiration)
        {
            string message = null;

            if (!TryIsHostNetworkSite(siteid, out message))
            {
                return false;
            }

            try
            {
                using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
                {
                    AuthUser.GetByID(userid).CancelPremium(sql, expiration);
                }
                return true;
            }
            catch (KeyNotFoundException)
            {
                return false;
            }
        }

        #endregion


        #endregion

        #region Non-Soap Methods

        [WebMethod]
        public string shareSessionByIP(string ipAddress, int siteid)
        {
            
            var session = "Unknown";

            if (siteid < 1)
            {
                return session;
            }

            try
            {
                if (!IsHostNetworkSite(siteid))
                {
                    return session;
                }

                var cookie = Context.Request.Cookies["networkcookie"];

                if (cookie == null || cookie.Value == null || !InputValidation.IsValidSession(cookie.Value) || cookie.Value.Length != AuthUser.SessionKeyLength)
                {
                    IPAddress newIPAddress;


                    if (!IPAddress.TryParse(ipAddress, out newIPAddress))
                    {
                        Logger.Trace("Failed to find a suitable IP address in shareSessionByIP.", new { Context.Request.UserHostAddress, Context.Request.UserAgent, Context.Request.UrlReferrer });
                        return session;
                    }

                    session = AuthUser.GenerateSessionKey(newIPAddress);
                    WriteSessionCookie(session);                    
                }
                else
                {
                    session = cookie.Value;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in shareSessionByIP");
            }

            return session;
        }

        private static string FixUrl(string url)
        {

            // Fix slashes in the wrong direction (this is insane...)
            if(url.Contains(@"\"))
            {
                url = url.Replace(@"\", @"/");
            }

            if (url.StartsWith("http:/", StringComparison.InvariantCultureIgnoreCase) && !url.StartsWith("http://", StringComparison.InvariantCultureIgnoreCase))
            {
                url = "http://" + url.Substring(6);
            }            
            else if (url.StartsWith("https:/", StringComparison.InvariantCultureIgnoreCase) && !url.StartsWith("https://", StringComparison.InvariantCultureIgnoreCase))
            {
                url = "https://" + url.Substring(7);
            }

            if (url.IndexOf("%2f", StringComparison.InvariantCultureIgnoreCase) > 0)
            {
                url = Uri.UnescapeDataString(url);
            }

            return url;
        }

        static bool IsTrustedDomain(string url, bool required, object logParams)
        {
            url = FixUrl(url);

            // Get the domain of the referrer and from address
            Uri fromUri;
            if (!Uri.TryCreate(url, UriKind.Absolute, out fromUri))
            {
                Logger.Info("[IsTrustedDomain] Invalid from parameter: " + url);
                return false;
            }

            return IsTrustedDomain(fromUri, required, logParams);
        }

        static bool IsTrustedDomain(Uri uri, bool required, object logParams)
        {
            if (uri == null && !required)
            {
                return true;
            }

            var domainName = uri.GetDomainName();

            if (string.IsNullOrEmpty(domainName))
            {
                return false;
            }

            if (!NetworkSite.IsTrustedDomain(domainName))
            {
                if (LogInvalidDomains && !LoggedInvalidDomains.ContainsKey(domainName))
                {                    
                    Logger.Warn("Untrusted domain attempted to make service call: " + domainName, new { logParams, uri });
                    LoggedInvalidDomains.TryAdd(domainName, true);
                }
                return false;
            }

            return true;
        }

        private static readonly ConcurrentDictionary<string, bool> LoggedInvalidDomains = new ConcurrentDictionary<string, bool>();

        [WebMethod]
        public void shareSession(string from)
        {
            if (string.IsNullOrEmpty(from))
            {
                return;
            }

            var cookie = Context.Request.Cookies["networkcookie"];
            var session = "Unknown";
            try
            {
                if (!IsTrustedDomain(from, true, new { from, HttpContext.Current.Request.UrlReferrer, method = "shareSession" }))
                {
                    if (ValidateReferrer)
                    {
                        return;
                    }
                }

                if (cookie == null || cookie.Value == null || !InputValidation.IsValidSession(cookie.Value) || cookie.Value.Length != AuthUser.SessionKeyLength)
                {
                    var ipAddress = HttpContext.Current.Request.GetClientIPAddress();

                    if (ipAddress != null)
                    {
                        session = AuthUser.GenerateSessionKey(ipAddress);
                        WriteSessionCookie(session);
                    }
                }
                else
                {
                    session = cookie.Value;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception in shareSession", new { Context.Request.UserHostAddress, Context.Request.UserAgent, Context.Request.UrlReferrer });
            }

            var redirectUrl = from + (from.Contains("?") ? "&" : "?") + "cookie=" + session;

            Context.Response.Redirect(redirectUrl);
        }

        private void WriteSessionCookie(string session)
        {            
            var cookie = new HttpCookie("networkcookie", session) { HttpOnly = true, Expires = DateTime.Now.AddYears(10) };
            Context.Response.SetCookie(cookie);            
        }

        [WebMethod]
        public void setSession(string sessionID, string redirectUrl)
        {
            if (string.IsNullOrEmpty(redirectUrl))
            {
                return;
            }

            try
            {
                if (!IsTrustedDomain(redirectUrl, true, new { redirectUrl, HttpContext.Current.Request.UrlReferrer, method = "setSession" }))
                {
                    if (ValidateReferrer)
                    {
                        return;
                    }
                }

                AuthUser.GetIDBySessionID(sessionID, true);
                var cookie = Context.Request.Cookies["networkcookie"];
                if (cookie == null)
                {
                    WriteSessionCookie(sessionID);
                }
            }
            catch (KeyNotFoundException) { }

            Context.Response.Redirect(redirectUrl, false);
        }

        #endregion
    }
}
