using System;
using System.Linq;
using System.Net;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Web.Caching;
using System.Xml;
using Curse.Caching;
using Curse.Extensions;
using Curse.Logging;

namespace Curse.Auth {
    
    /// <summary>
    /// Represents a network member site and the state associated with the
    /// site for caching purposes and similar. Used by the service in order
    /// to avoid regenerating cryptographic states and rechecking for valid
    /// IPs.
    /// </summary>    
    [Serializable]
    public class NetworkSite
    {
        #region Fields

        private byte[] _cyptoKey;
        private StringCipher _cipher;
        private List<string> _actualTrustedIPAddresses = new List<string>();
        private HashSet<string> _effectiveTrustedIPAddresses = new HashSet<string>();

        #endregion

        #region Constants

        const int iId = 0;
        const int iName = 1;
        const int iCryptoKey = 2;
        const int iUrl = 3;
        const int iIconUrl = 4;
        const int iMayAlterUser = 5;
        const int iReferrerHost = 6;
        const int iUpdateUrl = 7;
        const int iEmail = 8;

        #endregion

        #region Static Fields

        private static Random _random = new Random(DateTime.UtcNow.Second);

        // Trusted Domain Names
        private static readonly HashSet<string> _trustedDomains = new HashSet<string>(StringComparer.InvariantCultureIgnoreCase);

        private static readonly string[] AdditionalTopLevelDomains = { "local", "stg", "dev" };

        #endregion

        #region Constructor

        public NetworkSite() { }

        public NetworkSite(int id,
                           string name,
                           string url,
                           string iconurl,
                           string referrer,
                           string updateurl,
                           bool alteruser,
                           string email,
                           byte[] cryptokey)
        {
            ID = id;
            Name = name;
            Url = url;
            IconUrl = iconurl;
            CanAlterUser = alteruser;
            ReferrerHost = referrer;
            UpdateUrl = updateurl;
            Email = email;
            
            _cyptoKey = cryptokey;
            _cipher = new StringCipher(cryptokey);            
        }

        public NetworkSite(SqlConnection conn,                           
                           SqlDataReader reader)
        {


            ID = reader.GetInt32(iId);
            Name = reader.GetString(iName);            
            Url = reader.GetString(iUrl);
            IconUrl = reader.GetString(iIconUrl);
            CanAlterUser = reader.GetBoolean(iMayAlterUser);
            ReferrerHost = reader.GetString(iReferrerHost);
            UpdateUrl = reader.GetString(iUpdateUrl);
            Email = reader.GetString(iEmail);            

            var cryptokey = new byte[32];
            reader.GetBytes(iCryptoKey, 0, cryptokey, 0, 32);

            _cyptoKey = cryptokey;
            _cipher = new StringCipher(cryptokey);


            try
            {
                var referrerUri = new Uri("http://" + ReferrerHost, UriKind.Absolute);
                var domainNames = referrerUri.GetDomainNames(AdditionalTopLevelDomains);

                if (domainNames.Any())
                {
                    DomainName = domainNames.FirstOrDefault();
                    foreach (var domainName in domainNames)
                    {
                        _trustedDomains.Add(domainName);
                    }                    
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Invalid referrer: " + ReferrerHost);     
            }                     
            
            PopulateTrustedIPAddresses(conn);
        }

        #endregion

        #region Static Methods

        public static bool TryValidateSiteInfo(string name, string ip, string referrer, string updateUrl, string url, string iconUrl, string email, out string message)
        {
            if (!InputValidation.IsValidIp(ip))
            {
                message = "Invalid IP address.";
                return false;
            }

            if (!InputValidation.IsValidSiteName(name))
            {
                message = "Invalid network site name.";
                return false;
            }

            if (!InputValidation.IsValidHost(referrer))
            {
                message = "Invalid referrer.";
                return false;
            }

            if (!InputValidation.IsValidUpdateUrl(updateUrl))
            {
                message = "Invalid update URL.";
                return false;
            }

            if (!InputValidation.IsValidUrl(url))
            {
                message = "Invalid site URL.";
                return false;
            }

            if (!InputValidation.IsValidIconUrl(iconUrl))
            {
                message = "Invalid icon URL.";
                return false;
            }
            if (!InputValidation.IsValidEmail(email))
            {
                message = "Invalid email address.";
                return false;
            }
            message = "Successful";
            return true;
        }

        public static void Initialize(SqlConnection conn)
        {
            var sites = Sites;
            var sitesByID = SitesByID;

            Logger.Info("Loaded " + sites.Length + " sites!");
        }

        public static NetworkSite[] Sites
        {
            get
            {
                return CacheManager.GetOrAdd("Sites", () =>
                    {
                        using (var conn = DatabaseUtility.GetAuthConnection())
                        {
                            return GetNetworkSites(conn);
                        }

                    }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);
            }
        }

        public static Dictionary<int, NetworkSite> SitesByID
        {
            get
            {
                return CacheManager.GetOrAdd("SitesByID", () =>
                    {
                        var sites = Sites;
                        var dict = new Dictionary<int, NetworkSite>();
                        foreach (var site in sites)
                        {
                            dict[site.ID] = site;
                        }

                        return dict;

                    }, TimeSpan.FromDays(300), null, CacheItemPriority.NotRemovable);
            }
        }

        public static NetworkSite GetByID(int id)
        {            
            return SitesByID[id];            
        }

        public static NetworkSite[] GetAll()
        {
            return Sites;
        }

        public static bool ExistsByID(int id)
        {
            return SitesByID.ContainsKey(id);
        }

        public static NetworkSite Create(SqlConnection conn,
                                         int id,
                                         string name,
                                         string ip,
                                         string referrer,
                                         string updateUrl,
                                         string url,
                                         string iconurl,
                                         bool alterUser,
                                         string email)
        {


            byte[] cryptoKey = GetRandomCryptoKey();

            NetworkSite site = new NetworkSite(id, name, url, iconurl, referrer, updateUrl, alterUser, email, cryptoKey);

            SqlCommand cmd = conn.CreateCommand();
            cmd.Transaction = conn.BeginTransaction();

            try
            {
                cmd.CommandText =
                    "INSERT INTO sites " +
                    "(" +
                        "_id," +
                        "_name," +
                        "_cryptokey," +
                        "_url," +
                        "_iconUrl," +
                        "_mayAlterUser," +
                        "_referrerHost," +
                        "_updateUrl," +
                        "_email" +
                    ") " +
                    "VALUES " +
                    "(" +
                        "@id," +
                        "@name," +
                        "@key," +
                        "@url," +
                        "@iconurl," +
                        "@alter," +
                        "@referrer," +
                        "@updateurl," +
                        "@email" +
                    ")";

                cmd.Parameters.AddWithValue("id", site.ID);
                cmd.Parameters.AddWithValue("name", site.Name);
                cmd.Parameters.AddWithValue("key", site.CryptoKey);
                cmd.Parameters.AddWithValue("url", site.Url);
                cmd.Parameters.AddWithValue("iconurl", site.IconUrl);
                cmd.Parameters.AddWithValue("alter", site.CanAlterUser);
                cmd.Parameters.AddWithValue("referrer", site.ReferrerHost);
                cmd.Parameters.AddWithValue("updateurl", site.UpdateUrl);
                cmd.Parameters.AddWithValue("email", site.Email);
                cmd.ExecuteNonQuery();
                cmd.Parameters.Clear();

                site.AddIPAddressToDatabase(conn, ip, cmd);

                cmd.Transaction.Commit();
            }
            catch (Exception)
            {
                if (cmd.Transaction != null)
                {
                    cmd.Transaction.Rollback();
                }

                return null;
            }

            ExpireAll();

            return site;
        }
       

        private static void ExpireAll()
        {
            CacheManager.Expire("Sites");
            CacheManager.Expire("SitesByID");
        }

        public static NetworkSite[] GetNetworkSites(SqlConnection conn)
        {
            SqlCommand cmd = conn.CreateCommand();
            cmd.CommandText = 
                "SELECT " +
                    "_id," +
                    "_name," +
                    "_cryptokey," +
                    "_url," +
                    "_iconUrl," +
                    "_mayAlterUser," +
                    "_referrerHost," +
                    "_updateUrl," +
                    "_email " +
                "FROM sites";

            List<NetworkSite> sites = new List<NetworkSite>();

            using (SqlDataReader reader = cmd.ExecuteReader())
            {                                
                while (reader.Read())
                {                    
                    NetworkSite site = new NetworkSite(conn, reader);
                    sites.Add(site);
                }                
            }                       
            
            return sites.ToArray();
        }
        
        private static byte[] GetRandomCryptoKey()
        {
            byte[] key = new byte[32];
            _random.NextBytes(key);
            return key;
        }

        public static bool IsTrustedDomain(string domain)
        {
            return _trustedDomains.Contains(domain);
        }

        #endregion

        #region Public Methods
                            
        public void Delete(SqlConnection sql)
        {
            SqlCommand cmd = sql.CreateCommand();
            cmd.Transaction = sql.BeginTransaction();
            
            try
            {
                cmd.CommandText =  "DELETE FROM siteips WHERE _siteid = @siteid";
                cmd.Parameters.AddWithValue("siteid", ID);
                cmd.ExecuteNonQuery();
                
                cmd.CommandText = "DELETE FROM sites WHERE _id = @siteid";
                cmd.ExecuteNonQuery();
                cmd.Transaction.Commit();
            }
            catch(Exception e)
            {
                if (cmd.Transaction != null)
                {
                    cmd.Transaction.Rollback();
                }
                    
                throw e;
            }

            ExpireAll();

        }
        
        public bool ValidateIPAddress(string ip)
        {
            return _effectiveTrustedIPAddresses.Contains(ip);
        }                
        
        public override int GetHashCode()
        {
            return ID;
        }
        
        public string DecryptString(string str)
        {
            return _cipher.Decrypt(str);
        }
        
        public string EncryptString(string str)
        {
            return _cipher.Encrypt(str);
        }
        
        public void AddIPAddressToDatabase(SqlConnection sql, string ip, SqlCommand cmd)
        {
            IPAddress ipaddr = IPAddress.Parse(ip);
            if (cmd == null)
            {
                cmd = sql.CreateCommand();
            }

            cmd.CommandText =  "INSERT INTO siteips (_siteid, _ip) VALUES (@id, @ip)";
            cmd.Parameters.AddWithValue("id", ID);
            cmd.Parameters.AddWithValue("ip", ipaddr.GetAddressBytes());
            cmd.ExecuteNonQuery();

            ExpireAll();           
        }

        private void PopulateTrustedIPAddresses(SqlConnection conn)
        {
            _actualTrustedIPAddresses = new List<string>();
            _effectiveTrustedIPAddresses = new HashSet<string>();

            SqlCommand cmd = conn.CreateCommand();            
            cmd.CommandText = "SELECT _ip FROM siteips WHERE _siteid = @id";
            cmd.Parameters.Add(new SqlParameter("id", ID));

            using (SqlDataReader reader = cmd.ExecuteReader())
            {
                while (reader.Read())
                {
                    byte[] ipBytes = new byte[4];
                    reader.GetBytes(0, 0, ipBytes, 0, 4);
                    AddIPAddressToCache(new IPAddress(ipBytes));                                        
                }
            }
        }

        private void RecalculateEffectiveIPAddresses()
        {
            _effectiveTrustedIPAddresses.Clear();
            
            foreach (string actualIPString in _actualTrustedIPAddresses)
            {
                IPAddress ipAddress = IPAddress.Parse(actualIPString);
                byte[] ipBytes = ipAddress.GetAddressBytes();
                AddEffectiveIPAddressToCache(ipBytes);                
            }
        }

        public void AddEffectiveIPAddressToCache(byte[] ipBytes)
        {
            string ipString = null;

            if (ipBytes[3] == 0)
            {
                for (int i = 1; i < 255; i++)
                {
                    ipString = String.Format("{0}.{1}.{2}.{3}", ipBytes[0], ipBytes[1], ipBytes[2], i.ToString());
                    if(!_effectiveTrustedIPAddresses.Contains(ipString))
                    {
                        _effectiveTrustedIPAddresses.Add(ipString);
                    }
                }
            }
            else
            {
                ipString = String.Format("{0}.{1}.{2}.{3}", ipBytes[0], ipBytes[1], ipBytes[2], ipBytes[3]);
                if (!_effectiveTrustedIPAddresses.Contains(ipString))
                {
                    _effectiveTrustedIPAddresses.Add(ipString);
                }
            }
        }

        private void AddIPAddressToCache(IPAddress ipAddress)
        {
            
            string ipString = ipAddress.ToString();
            byte[] ipBytes = ipAddress.GetAddressBytes();

            _actualTrustedIPAddresses.Add(ipString);

            AddEffectiveIPAddressToCache(ipBytes);                                            
        }

        public void RemoveIPAddressFromDatabase(SqlConnection sql, string ip)
        {
            IPAddress ipaddr = IPAddress.Parse(ip);

            SqlCommand cmd = sql.CreateCommand();
            cmd.CommandText = "DELETE siteips WHERE _siteid = @id and _ip = @ip";
            cmd.Parameters.AddWithValue("id", ID);
            cmd.Parameters.AddWithValue("ip", ipaddr.GetAddressBytes());
            cmd.ExecuteNonQuery();

            ExpireAll();
        }

        #endregion

        #region Properties

        public int ID { get; set; }        

        public string Name { get; set; }

        public string Url { get; set; }

        public string IconUrl { get; set; }

        public string ReferrerHost { get; set; }

        public string UpdateUrl { get; set; }

        public bool CanAlterUser { get; set; }

        public string Email { get; set; }

        public string DomainName { get; set; }

        public List<string> TrustedIPAddresses
        {
            get
            {
                return _actualTrustedIPAddresses;
            }
            set
            {
                _actualTrustedIPAddresses = value;
            }
        }

        public byte[] CryptoKey { get { return _cyptoKey; } }

        #endregion

    }
}
