﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Data;

namespace Curse.Auth
{
    public class PasswordResetRequest
    {                
        // How long a token is good for
        private readonly static TimeSpan Lifespan = TimeSpan.FromDays(1);

        // How often a user can request a new password
        public readonly static TimeSpan Cooldown = TimeSpan.FromHours(1);

        private readonly static Random RandomProvider = new Random(DateTime.UtcNow.Second);

        private static readonly object SyncLock = new object();

        public PasswordResetRequest(Int32 userID)
        {
            Token = GenerateNewToken();
            DateCreated = DateTime.UtcNow;
            UserID = userID;
        }

        public PasswordResetRequest(SqlDataReader reader)
        {
            ID = (int)reader["_id"];
            Token = (int)reader["_token"];
            DateCreated = (DateTime)reader["_dateCreated"];
            UserID = (int)reader["_uid"];
        }

        private static Int32 GenerateNewToken()
        {
            return RandomProvider.Next(10000, Int32.MaxValue);            
        }

        #region Static Methods

        public static PasswordResetRequest Create(Int32 userID)
        {
            var token = new PasswordResetRequest(userID);
            for (int i = 0; i < 10; i++)
            {
                try
                {
                    token.SaveToDatabase();
                    return token;
                }
                catch (SqlException)
                {
                    token.Token = GenerateNewToken();
                }             
            }

            return null;
        }        

        public static void Prune()
        {
            lock (SyncLock)
            {
                using (var conn = DatabaseUtility.GetAuthConnection(true))
                {
                    using (var cmd = conn.CreateCommand())
                    {

                        cmd.CommandText = "DELETE FROM [passwordresetrequests] WHERE _dateCreated < @DateCreated";
                        cmd.Parameters.AddWithValue("@DateCreated", DateTime.UtcNow -Lifespan);
                        cmd.ExecuteNonQuery();
                    }

                }
            }
        }

        public void SaveToDatabase()
        {
            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    
                    cmd.CommandText = "INSERT INTO [passwordresetrequests] (_uid, _token, _dateCreated) OUTPUT INSERTED._id values (@UserID, @Token, @DateCreated)";                                    
                    cmd.Parameters.AddWithValue("@UserID", UserID);
                    cmd.Parameters.AddWithValue("@Token", Token);
                    cmd.Parameters.AddWithValue("@DateCreated", DateCreated);
                    ID = (Int32)cmd.ExecuteScalar();
                }

            }
        }

        public void Delete()
        {
            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "DELETE FROM [passwordresetrequests] WHERE _id = @ID";
                    cmd.Parameters.AddWithValue("@ID", ID);
                    cmd.ExecuteNonQuery();
                }
            }
        }

        public static PasswordResetRequest GetByUserID(int userID)
        {

            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "SELECT * FROM [passwordresetrequests] WHERE _uid = @UserID";
                    cmd.Parameters.AddWithValue("@UserID", userID);
           
                    using (var reader = cmd.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new PasswordResetRequest(reader);
                        }
                        else
                        {
                            return null;
                        }
                    }
                }
              
           }
        }

        public static PasswordResetRequest GetByToken(int token)
        {
            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "SELECT * FROM [passwordresetrequests] WHERE _token = @Token";
                    cmd.Parameters.AddWithValue("@Token", token);

                    using (var reader = cmd.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new PasswordResetRequest(reader);
                        }
                        else
                        {
                            return null;
                        }
                    }
                }

            }
        }              

        #endregion

        #region Properties

        public Int32 ID
        {
            get; 
            set;
        }

        public Int32 UserID
        {
            get;
            set;
        }

        public Int32 Token
        {
            get;
            set;
        }

        public DateTime DateCreated
        {
            get;
            set;
        }

        public DateTime DateExpires
        {
            get
            {
                return DateCreated.Add(Lifespan);
            }
            
        }
       
        public bool IsRenewable
        {
            get
            {
                return DateTime.UtcNow - DateCreated > Cooldown;
            }
        }

        public bool IsExpired
        {
            get
            {
                return DateExpires < DateTime.UtcNow;
            }
        }

        #endregion
    }
}
