using System;
using System.Data;
using System.Data.SqlClient;
using System.Text;
using System.Web;
using System.Security.Cryptography;
using System.Collections.Generic;
using System.Xml;
using Curse.Logging;
using Curse.Extensions;
using Curse.Caching;
using System.Net;

namespace Curse.Auth
{
    [Serializable]
    public class User
    {
        #region Constants

        const int iUid = 0;
        const int iUsername = 1;
        const int iPassword = 2;
        const int iSalt = 3;
        const int iUserLevel = 4;
        const string policyAcceptedColumn = "_policyAccepted";

        public const int SessionKeyLength = 48;
        private static readonly char[] _sessionCharacters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890".ToCharArray();

        #endregion

        #region Fields

        bool _sessionIsSecure = false;
        List<UserSubscripion> _subscriptions;
        List<UserLegacy> _legacy;
        string _plainTextPassword = null;
        byte[] _passwordHash = null;
        byte[] _salt = null;

        #endregion

        #region Static Fields

        private const int AnonymousLookupValue = 0;
        private const int NotFoundLookupValue = -1;

        private static readonly Random _randomProvider = new Random(DateTime.UtcNow.Second);

        #endregion

        #region Properties

        public int ID { get; set; }

        public string Name { get; set; }

        public UserProfile Profile { get; set; }

        public int Level { get; set; }

        public bool SessionIsSecure
        {
            get
            {
                return _sessionIsSecure;
            }
            set
            {
                _sessionIsSecure = value;
            }
        }

        public List<UserSubscripion> Subscriptions
        {
            get
            {
                return _subscriptions;
            }
        }

        public List<UserLegacy> Legacy
        {
            get
            {
                return _legacy;
            }
        }

        public UserEmail PrimaryEmail
        {
            get
            {
                return Profile.GetPrimaryEmail();
            }
        }

        public bool HasRenameCredit
        {
            get;
            set;
        }

        public List<UserEntitlement> Entitlements
        {
            get
            {
                return GetMergedEntitlements();

            }
        }

        private List<UserEntitlement> GetMergedEntitlements()
        {
            // Get the user's entitlements, from the new billing system
            var entitlements = BillingServiceHelper.GetAccountEntitlements(this.ID);

            // See if they have a legacy premium subscription
            var premiumSubscription = GetActiveSubscription(ESubscriptionType.Premium);

            // If they have a legacy premium subscription, add it.
            if (premiumSubscription != null)
            {
                bool hasActivePremiumSub = false;
                foreach (var entitlement in entitlements)
                {
                    if (entitlement.Type == ESubscriptionType.Premium && entitlement.Expires > DateTime.UtcNow)
                    {
                        hasActivePremiumSub = true;
                        break;
                    }
                }

                if (!hasActivePremiumSub)
                {
                    entitlements.Add(new UserEntitlement()
                    {
                        Expires = premiumSubscription.ExpirationDate,
                        ID = "CP",
                        Type = ESubscriptionType.Premium,
                        Active = true
                    });
                }
            }

            return entitlements;
        }

        public EUserStatus Status { get; set; }

        public DateTime? PolicyAcceptDate { get; set; }

        #endregion

        #region Static Methods

        public static User Create(SqlConnection conn,
                                  int siteID,
                                  string username,
                                  string plainTextPassword,
                                  string emailAddress,
                                  string firstName,
                                  string lastName,
                                  string gender,
                                  string country,
                                  string city,
                                  string region,
                                  bool newsletter)
        {

            byte[] salt = GenerateSalt();
            byte[] hashedPassword = GetPasswordHash(plainTextPassword, salt);

            using (SqlConnection sql = DatabaseUtility.GetAuthConnection(false))
            {
                SqlCommand cmd = sql.CreateCommand();
                cmd.Transaction = sql.BeginTransaction();
                try
                {
                    cmd.CommandText =
                        "INSERT INTO users " +
                        "(" +
                            "_username," +
                            "_password," +
                            "_salt," +
                            "_level" +
                        ") " +
                        "OUTPUT Inserted._uid " +
                        "VALUES " +
                        "(" +
                            "@username," +
                            "@password," +
                            "@salt," +
                            "@level" +
                        ")";

                    cmd.Parameters.AddWithValue("username", username);
                    cmd.Parameters.AddWithValue("password", hashedPassword);
                    cmd.Parameters.AddWithValue("salt", salt);
                    cmd.Parameters.AddWithValue("level", 100);
                    var userID = (int)cmd.ExecuteScalar();

                    cmd.Parameters.Clear();
                    cmd.CommandText =
                        "INSERT INTO useremails " +
                        "(" +
                            "_uid," +
                            "_email" +
                        ") " +
                        "OUTPUT Inserted._id " +
                        "VALUES " +
                        "(" +
                            "@uid," +
                            "@email" +
                        ")";
                    cmd.Parameters.AddWithValue("uid", userID);
                    cmd.Parameters.AddWithValue("email", emailAddress);
                    var emailid = (int)cmd.ExecuteScalar();

                    cmd.Parameters.Clear();
                    cmd.CommandText =
                        "INSERT INTO userprofiles " +
                        "(" +
                            "_uid," +
                            "_firstName," +
                            "_lastName," +
                            "_gender," +
                            "_defaultEmail," +
                            "_country," +
                            "_region," +
                            "_city," +
                            "_wantsmail," +
                            "_registeredVia" +
                        ") " +
                        "VALUES " +
                        "(" +
                            "@uid," +
                            "@first," +
                            "@last," +
                            "@gender," +
                            "@emailid," +
                            "@country," +
                            "@region," +
                            "@city," +
                            "@wantsmail," +
                            "@via" +
                        ")";

                    cmd.Parameters.AddWithValue("uid", userID);
                    cmd.Parameters.AddWithValue("first", firstName);
                    cmd.Parameters.AddWithValue("last", DatabaseUtility.GetOptionalString(lastName));
                    cmd.Parameters.AddWithValue("gender", DatabaseUtility.GetOptionalInt(gender));
                    cmd.Parameters.AddWithValue("emailid", emailid);
                    cmd.Parameters.AddWithValue("country", country);
                    cmd.Parameters.AddWithValue("region", DatabaseUtility.GetOptionalString(region));
                    cmd.Parameters.AddWithValue("city", DatabaseUtility.GetOptionalString(city));
                    cmd.Parameters.AddWithValue("wantsmail", newsletter);
                    cmd.Parameters.AddWithValue("via", siteID);
                    cmd.ExecuteNonQuery();

                    cmd.Transaction.Commit();


                    var user = CreateByID(userID, sql);

                    if (user == null)
                    {
                        Logger.Error("User " + userID + "created, but could not be found in database!");
                    }
                    else
                    {
                        user.ExpireAll();
                    }


                    return user;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create user!");

                    if (cmd.Transaction != null)
                    {
                        cmd.Transaction.Rollback();
                    }
                    return null;
                }

            }
        }

        public static User CreateLegacy(SqlConnection conn,
                          int siteID,
                          int legacyid,
                          string salt,
                          ELegacyEncryptionType encryptionType,
                          string username,
                          string passwordhash,
                          string emailAddress,
                          string firstName,
                          string lastName,
                          string gender,
                          string country,
                          string city,
                          string region,
                          bool newsletter,
                          DateTime dateRegistered)
        {

            int userID;

            using (SqlConnection sql = DatabaseUtility.GetAuthConnection(false))
            {
                SqlCommand cmd = sql.CreateCommand();
                cmd.Transaction = sql.BeginTransaction();
                try
                {
                    cmd.CommandText =
                        "INSERT INTO users " +
                        "(" +
                            "_username," +
                            "_level" +
                        ") " +
                        "OUTPUT Inserted._uid " +
                        "VALUES " +
                        "(" +
                            "@username," +
                            "@level" +
                        ")";

                    cmd.Parameters.AddWithValue("username", username);
                    cmd.Parameters.AddWithValue("level", 100);
                    userID = (int)cmd.ExecuteScalar();

                    cmd.Parameters.Clear();
                    cmd.CommandText =
                        "INSERT INTO useremails " +
                        "(" +
                            "_uid," +
                            "_email" +
                        ") " +
                        "OUTPUT Inserted._id " +
                        "VALUES " +
                        "(" +
                            "@uid," +
                            "@email" +
                        ")";
                    cmd.Parameters.AddWithValue("uid", userID);
                    cmd.Parameters.AddWithValue("email", emailAddress);
                    var emailid = (int)cmd.ExecuteScalar();

                    cmd.Parameters.Clear();
                    cmd.CommandText =
                        "INSERT INTO userprofiles " +
                        "(" +
                            "_uid," +
                            "_firstName," +
                            "_lastName," +
                            "_gender," +
                            "_defaultEmail," +
                            "_country," +
                            "_region," +
                            "_city," +
                            "_wantsmail," +
                            "_registeredVia, " +
                            "_registeredOn" +
                        ") " +
                        "VALUES " +
                        "(" +
                            "@uid," +
                            "@first," +
                            "@last," +
                            "@gender," +
                            "@emailid," +
                            "@country," +
                            "@region," +
                            "@city," +
                            "@wantsmail," +
                            "@via, " +
                            "@on" +
                        ")";

                    cmd.Parameters.AddWithValue("uid", userID);
                    cmd.Parameters.AddWithValue("first", firstName);
                    cmd.Parameters.AddWithValue("last", DatabaseUtility.GetOptionalString(lastName));
                    cmd.Parameters.AddWithValue("gender", DatabaseUtility.GetOptionalInt(gender));
                    cmd.Parameters.AddWithValue("emailid", emailid);
                    cmd.Parameters.AddWithValue("country", country);
                    cmd.Parameters.AddWithValue("region", DatabaseUtility.GetOptionalString(region));
                    cmd.Parameters.AddWithValue("city", DatabaseUtility.GetOptionalString(city));
                    cmd.Parameters.AddWithValue("wantsmail", newsletter);
                    cmd.Parameters.AddWithValue("via", siteID);
                    cmd.Parameters.AddWithValue("on", dateRegistered);
                    cmd.ExecuteNonQuery();

                    cmd.Transaction.Commit();
                }
                catch
                {
                    if (cmd.Transaction != null)
                    {
                        cmd.Transaction.Rollback();
                    }

                    return null;
                }
            }

            CacheEmailToUser(emailAddress, userID);
            CacheNameToUser(username, userID);

            //Create the legacy password entry
            var user = GetByID(userID);
            if (user.ChangeLegacyPassword(siteID, legacyid, passwordhash, salt, encryptionType, conn) == ESetUserPasswordResult.Success)
            {
                return user;
            }

            return null;
        }


        private void ExpireAll()
        {
            CacheManager.Expire(GetCacheKey(this.ID));
            CacheManager.Expire(GetCacheKeyByEmail(this.PrimaryEmail.Address));
            CacheManager.Expire(GetCacheKeyByUsername(this.Name));
        }

        private static string GetCacheKey(int id)
        {
            return string.Format(UserCacheKey, id);
        }

        private static string GetCacheKeyByUsername(string username)
        {
            return "UserByName:" + username.ToLowerInvariant();
        }

        private static string GetCacheKeyByEmail(string email)
        {
            return "UserByEmail:" + email.ToLowerInvariant();
        }

        private static string GetCacheKeyBySession(string session)
        {
            return "UserBySession:" + session.ToUpperInvariant();
        }

        public static User GetByID(int id)
        {
            // Figure out which site/service is passing -1 for the user ID
            if (id <= 0)
            {
                try
                {
                    
                    Logger.Warn("Attempt to get user by an invalid ID: " + id,
                        new { IPAddress = HttpContext.Current.Request.GetClientIPAddress(true).ToString(), HttpContext.Current.Request.UserAgent });
                }
                catch { }

                throw new KeyNotFoundException("No such user");
            }

            var cacheKey = GetCacheKey(id);

            return CacheManager.GetOrAdd(cacheKey, () =>
            {
                User user = null;
                using (var sql = DatabaseUtility.GetAuthConnection(false))
                {
                    user = CreateByID(id, sql);
                }

                if (user == null)
                {
                    Logger.Trace("User '" + id + "' was null after trying to get from cache and calling CreateByID");
                    throw new KeyNotFoundException("No such user");
                }

                return user;
            });
        }

        public static User GetByName(string name)
        {
            var userID = GetIDByName(name);

            if (userID > 0)
            {
                return GetByID(userID);
            }

            throw new KeyNotFoundException("No such username");
        }

        public static User GetByEmailAddress(string name)
        {
            var userID = GetIDByEmailAddress(name);

            if (userID > 0)
            {
                return GetByID(userID);
            }

            throw new KeyNotFoundException("No such e-mail address");

        }

        public static User GetBySessionID(string sessionID)
        {
            return GetBySessionID(sessionID, true);
        }

        public static User GetBySessionID(string sessionID, bool searchDatabase)
        {
            var userID = GetIDBySessionID(sessionID, searchDatabase);

            if (userID > AnonymousLookupValue)
            {
                return GetByID(userID);
            }

            throw new KeyNotFoundException("No such session");

        }

        public static int GetIDByName(string username)
        {

            var userID = GetNameToUser(username);

            if (userID >= AnonymousLookupValue)
            {
                return userID;
            }

            int effectiveUserID;

            using (var sql = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = new SqlCommand("spUserIDFromUsername", sql))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strUsername", SqlDbType.VarChar, 32);
                    cmd.Parameters["@strUsername"].Value = username;

                    var uid = (int?)cmd.ExecuteScalar();

                    if (!uid.HasValue)
                    {
                        return 0;
                    }

                    effectiveUserID = (int)uid;
                }
            }
            CacheNameToUser(username, effectiveUserID);
            return effectiveUserID;
        }

        public static User GetByOAuthAccount(int oAuthPlatform, string oAuthId)
        {
            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "select _uid from useroauth where _platformtype = @platformType and _platformId = @platformId";
                    cmd.Parameters.AddWithValue("@platformType", oAuthPlatform);
                    cmd.Parameters.AddWithValue("@platformId", oAuthId);
                    var uid = (int?)cmd.ExecuteScalar();
                    if (uid.HasValue)
                    {
                        return GetByID(uid.Value);
                    }
                }
            }

            return null;
        }

        public static int GetIDByEmailAddress(string emailAddress)
        {
            var userID = GetEmailToUser(emailAddress);

            if (userID >= AnonymousLookupValue)
            {
                return userID;
            }

            int effectiveUserID;

            using (var conn = DatabaseUtility.GetAuthConnection(true))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "spUserIDFromEmail";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strEmail", SqlDbType.VarChar, 64);
                    cmd.Parameters["@strEmail"].Value = emailAddress;

                    var uid = (int?)cmd.ExecuteScalar();

                    if (!uid.HasValue)
                    {
                        return 0;
                    }

                    effectiveUserID = (int)uid;
                }
            }

            CacheEmailToUser(emailAddress, effectiveUserID);
            return effectiveUserID;
        }

        private static void ClearSession(string sessionID)
        {
            CacheManager.Expire(GetCacheKeyBySession(sessionID));
        }

        private static void CacheSession(string sessionID, int userID)
        {
            var cacheKey = GetCacheKeyBySession(sessionID);
            CacheManager.Add(cacheKey, userID);
            CacheManager.ExpireRemotely(cacheKey);
        }

        private static void CacheEmailToUser(string email, int userID)
        {
            CacheManager.Add(GetCacheKeyByEmail(email), userID);
        }

        private static void ClearEmailToUser(string email)
        {
            CacheManager.Expire(GetCacheKeyByEmail(email));
        }

        private static int GetEmailToUser(string email)
        {
            int userID;

            if (CacheManager.TryGet(GetCacheKeyByEmail(email), out userID))
            {
                return userID;
            }

            return NotFoundLookupValue;
        }

        private const string UserCacheKey = "UserByID:{0}";

        public string CacheKey
        {
            get
            {
                return string.Format(UserCacheKey, ID);
            }
        }

        private static void ClearUser(int userID)
        {
            var cacheKey = GetCacheKey(userID);
            CacheManager.Expire(cacheKey);

        }

        public void ClearCache()
        {
            ClearUser(ID);
        }

        private static void CacheNameToUser(string name, int userID)
        {
            CacheManager.Add(GetCacheKeyByUsername(name), userID);
        }

        private static void ClearNameToUser(string name)
        {
            CacheManager.Expire(GetCacheKeyByUsername(name));
        }

        private static int GetNameToUser(string name)
        {
            int userID;

            if (CacheManager.TryGet(GetCacheKeyByUsername(name), out userID))
            {
                return userID;
            }

            return NotFoundLookupValue;
        }

        public static int GetIDBySessionID(string sessionID, bool searchDatabase)
        {

            int userID;

            if (CacheManager.TryGet(GetCacheKeyBySession(sessionID), out userID))
            {
                using (var conn = DatabaseUtility.GetAuthConnection(false))
                {
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "spUpdateSessionV2";
                        cmd.CommandType = CommandType.StoredProcedure;
                        cmd.Parameters.Add("@strSession", SqlDbType.VarChar, SessionKeyLength);
                        cmd.Parameters["@strSession"].Value = sessionID;
                        cmd.ExecuteNonQuery();
                    }
                }

                return userID;
            }

            if (!searchDatabase)
            {
                return 0;
            }

            var effectiveUserID = AnonymousLookupValue;

            using (var conn = DatabaseUtility.GetAuthConnection(false))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "spUserIDFromSessionV2";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strSession", SqlDbType.VarChar, SessionKeyLength);
                    cmd.Parameters["@strSession"].Value = sessionID;
                    var uid = (int?)cmd.ExecuteScalar();
                    if (uid.HasValue)
                    {
                        effectiveUserID = (int)uid;
                    }
                }
            }

            CacheSession(sessionID, effectiveUserID);
            return effectiveUserID;

        }

        public static string GetSessionFromUserID(int userID, int siteID)
        {
            using (var conn = DatabaseUtility.GetAuthConnection(false))
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "spGetSessionFromUserID";

                    cmd.CommandType = CommandType.StoredProcedure;
                    var userParam = cmd.Parameters.Add("@uid", SqlDbType.Int, userID);
                    userParam.Value = userID;

                    var siteParam = cmd.Parameters.Add("@siteid", SqlDbType.Int, siteID);
                    siteParam.Value = siteID;

                    var value = cmd.ExecuteScalar();
                    return (value == DBNull.Value) ? null : (string)value;
                }
            }

        }

        public static User CreateByID(int uid, SqlConnection conn)
        {

            User user = null;

            try
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "spUserFromID";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@numUserID", SqlDbType.Int);
                    cmd.Parameters["@numUserID"].Value = uid;

                    using (var reader = cmd.ExecuteReader())
                    {
                        user = CreateFromQueryResult(reader);
                    }

                    if (user == null)
                    {
                        return null;
                    }

                    user.Profile = new UserProfile(user.ID, conn);
                    user.LoadSubscriptions(conn);
                    user.LoadLegacyData(conn);
                }

                using (var cmd = conn.CreateCommand())
                {
                    cmd.Parameters.Add("@numUserID", SqlDbType.Int);
                    cmd.Parameters["@numUserID"].Value = uid;
                    cmd.CommandText = "select _uid from userrenames where _uid = @numUserID";
                    user.HasRenameCredit = cmd.ExecuteScalar() != null;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create user!");
                throw;
            }

            return user;
        }

        private static User CreateFromQueryResult(SqlDataReader from)
        {
            if (!from.Read())
            {
                return null;
            }

            var user = new User();
            user.ID = from.GetInt32(iUid);
            user.Name = from.GetString(iUsername);
            if (!from.IsDBNull(iPassword))
            {
                user._passwordHash = new byte[20];
                from.GetBytes(iPassword, 0, user._passwordHash, 0, 20);
                user._salt = new byte[5];
                from.GetBytes(iSalt, 0, user._salt, 0, 5);
            }
            user.Level = from.GetInt32(iUserLevel);
            user.PolicyAcceptDate = from.GetNullableValue<DateTime?>(policyAcceptedColumn);

            return user;
        }

        public static string GenerateSessionKey(IPAddress ipAddress)
        {
            try
            {
                if (ipAddress == null)
                {
                    throw new ArgumentNullException("ipAddress");
                }

                using (RandomNumberGenerator rng = new RNGCryptoServiceProvider())
                {
                    var ipAddressBytes = ipAddress.GetAddressBytes();
                    var numRandomBytes = SessionKeyLength - ipAddressBytes.Length;
                    var tokenData = new byte[numRandomBytes];
                    rng.GetBytes(tokenData);
                    Array.Resize(ref tokenData, numRandomBytes + ipAddressBytes.Length);
                    ipAddressBytes.CopyTo(tokenData, numRandomBytes);
                    var result = new StringBuilder(SessionKeyLength);

                    foreach (var b in tokenData)
                    {
                        result.Append(_sessionCharacters[b % (_sessionCharacters.Length)]);
                    }

                    return result.ToString();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to generate session key!");
                throw;
            }

        }

        public static byte[] GenerateSalt()
        {
            var salt = new byte[5];
            _randomProvider.NextBytes(salt);
            return salt;
        }

        public static bool TryValidateProfile(string firstname,
                                              string lastname,
                                              string gender,
                                              string country,
                                              string region,
                                              string city,
                                              string emailaddress,
                                              out string message)
        {
            if (!InputValidation.IsValidFirstName(firstname))
            {
                message = "Invalid first name.";
                return false;
            }

            if (!InputValidation.IsValidLastName(lastname))
            {
                message = "Invalid last name.";
                return false;
            }

            if (!InputValidation.IsValidGender(gender))
            {
                message = "Invalid gender.";
                return false;
            }

            if (!InputValidation.IsValidCountryCode(country))
            {
                message = "Invalid country code.";
                return false;
            }

            if (!InputValidation.IsValidRegionCode(region))
            {
                message = "Invalid region code.";
                return false;
            }

            if (!InputValidation.IsValidCity(city))
            {
                message = "Invalid city.";
                return false;
            }

            if (!InputValidation.IsValidEmail(emailaddress))
            {
                message = "Invalid email address.";
                return false;
            }

            message = "Success";
            return true;
        }

        #endregion

        #region Public Instance Methods

        public void Delete(SqlConnection conn)
        {
            // Delete from the database
            using (var cmd = conn.CreateCommand())
            {
                cmd.CommandText = "spDeleteUser";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("@UserID", SqlDbType.Int);
                cmd.Parameters["@UserID"].Value = ID;
                cmd.ExecuteNonQuery();
            }

            // Clear any cached data
            ClearUserSessions(conn);
            ClearNameToUser(Name);
            ClearEmailToUser(PrimaryEmail.Address);
            ClearUser(ID);
        }

        public void UpdateEmailAddress(string emailAddress)
        {
            var primaryEmail = PrimaryEmail;

            ClearEmailToUser(primaryEmail.Address);

            using (var sql = DatabaseUtility.GetAuthConnection(false))
            {
                primaryEmail.Update(sql, emailAddress);
            }

            CacheEmailToUser(emailAddress, ID);
        }

        public bool DeclineRename(SqlConnection conn)
        {

            using (var cmd = conn.CreateCommand())
            {
                cmd.Parameters.AddWithValue("id", ID);
                cmd.CommandText = "DELETE FROM userrenames WHERE _uid = @id;";
                cmd.ExecuteNonQuery();
            }

            HasRenameCredit = false;
            return true;
        }

        public bool Rename(string username, SqlConnection conn, bool isAdmin)
        {
            using (var cmd = conn.CreateCommand())
            {
                cmd.Transaction = conn.BeginTransaction();
                cmd.Parameters.AddWithValue("id", ID);

                if (!isAdmin)
                {
                    cmd.CommandText = "DELETE FROM userrenames WHERE _uid = @id;";
                    if (cmd.ExecuteNonQuery() == 0)
                    {
                        cmd.Transaction.Rollback();
                        return false;
                    }
                }

                if (username != Name.ToLower())
                {
                    cmd.CommandText = "UPDATE users SET _username = @username WHERE _uid = @id";
                    cmd.Parameters.AddWithValue("username", username);
                    cmd.ExecuteNonQuery();
                }

                cmd.Transaction.Commit();
            }

            // Clear the old user lookup
            ClearNameToUser(Name);
            Name = username;
            // Clear the new one
            ClearNameToUser(Name);

            HasRenameCredit = false;

            return true;
        }

        private bool NewProcessLegacyPassword(string plainTextPassword, SqlConnection conn)
        {

            string passwordHash = default(string), salt = default(string);
            var encryptionType = ELegacyEncryptionType.None;
            var pwds = new List<string>();

            using (var cmd = conn.CreateCommand())
            {
                cmd.CommandText = "SELECT _password, _salt, _encryption FROM userlegacy WHERE _uid = @uid";
                cmd.Parameters.Add("uid", SqlDbType.Int).Value = ID;
                using (var reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        passwordHash = reader.GetString(reader.GetOrdinal("_password"));
                        salt = reader.GetNullableValue<string>("_salt");
                        encryptionType = (ELegacyEncryptionType)reader.GetByte(reader.GetOrdinal("_encryption"));
                    }
                }
            }

            switch (encryptionType)
            {
                case ELegacyEncryptionType.D:
                    {
                        var sha1 = SHA1.Create();
                        pwds.Add(BitConverter.ToString(sha1.ComputeHash(Encoding.ASCII.GetBytes(Name.ToLower() + HttpUtility.HtmlDecode(plainTextPassword.Replace("\\", ""))))).Replace("-", "").ToLower());

                        if (salt != default(string))
                        {
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(CryptoServiceProvider.ComputeMD5String(plainTextPassword) + salt));
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(CryptoServiceProvider.ComputeMD5String(salt) + CryptoServiceProvider.ComputeMD5String(plainTextPassword)));
                        }
                        else
                        {
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(plainTextPassword).ToLower());
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(plainTextPassword + Name.ToLower()));
                            pwds.Add(CryptoServiceProvider.ComputeShaString(plainTextPassword).ToLower());
                            pwds.Add(plainTextPassword);
                        }

                        break;
                    }
                case ELegacyEncryptionType.ShaPasswordMD5Salt: //WowStead(9999002)
                    {
                        if (salt != default(string))
                        {
                            var saltString = (string)salt;
                            var shaPassword = CryptoServiceProvider.ComputeShaString(plainTextPassword).ToLower();
                            var md5Salt = CryptoServiceProvider.ComputeMD5String(saltString);
                            var finalPassword = CryptoServiceProvider.ComputeShaString(shaPassword + md5Salt).ToLower();

                            pwds.Add(finalPassword);
                        }
                        else
                        {
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(plainTextPassword));
                        }
                        break;
                    }
                case ELegacyEncryptionType.Sha1SaltWithPassword:
                    {
                        var shaPassword = CryptoServiceProvider.ComputeShaString(salt + plainTextPassword).ToLower();
                        pwds.Add(shaPassword);

                        break;
                    }
                case ELegacyEncryptionType.Pbkdf2:
                    {

                        var passwordKey = String.Empty;

                        using (var hmac = new HMACSHA256())
                        {
                            var df = new CryptoServiceProvider.Pbkdf2(hmac, plainTextPassword, salt, 10000);
                            passwordKey = Convert.ToBase64String(df.GetBytes(32));
                        }

                        pwds.Add(passwordKey);

                        break;
                    }
                case ELegacyEncryptionType.MD5Password: //WOR(19838304)
                    {
                        pwds.Add(CryptoServiceProvider.ComputeMD5String(plainTextPassword));
                        break;
                    }
                case ELegacyEncryptionType.MD5PasswordWithSalt: //AS(17395615)
                    {
                        if (salt != default(string))
                        {
                            var md5_password = CryptoServiceProvider.ComputeMD5String(plainTextPassword).ToLower();
                            pwds.Add(CryptoServiceProvider.ComputeMD5String(md5_password + salt).ToLower());
                        }
                        break;
                    }
                case ELegacyEncryptionType.MyBB:
                    {
                        var pwd_hash = CryptoServiceProvider.ComputeMD5String(plainTextPassword).ToLower();
                        var salt_hash = CryptoServiceProvider.ComputeMD5String(salt).ToLower();
                        var final_hash = CryptoServiceProvider.ComputeMD5String(salt_hash + pwd_hash).ToLower();
                        pwds.Add(final_hash);
                        break;
                    }
                case ELegacyEncryptionType.PHPBB2: //Darthhater(9999005)
                    {
                        if (passwordHash != null)
                        {
                            if (passwordHash.StartsWith("$2"))
                            {
                                if (BCrypt.Net.BCrypt.Verify(plainTextPassword, passwordHash))
                                {
                                    pwds.Add(passwordHash);
                                }
                            }
                            else
                            {
                                pwds.Add(CryptoServiceProvider.PhpBB2.ComputeHash(ASCIIEncoding.ASCII.GetBytes(plainTextPassword), passwordHash));
                            }
                        }

                        break;
                    }
                case ELegacyEncryptionType.PHPBB3: //Merriland(7777788)
                    {
                        if (passwordHash != null)
                        {
                            if (passwordHash.StartsWith("$2"))
                            {
                                if (BCrypt.Net.BCrypt.Verify(plainTextPassword, passwordHash))
                                {
                                    pwds.Add(passwordHash);
                                }
                            }
                            else
                            {
                                pwds.Add(CryptoServiceProvider.PhpBB3.ComputeHash(ASCIIEncoding.ASCII.GetBytes(plainTextPassword), passwordHash));
                            }
                        }

                        break;
                    }
                case ELegacyEncryptionType.vBulletin: //MTG Salvation
                    {
                        var md5_password = CryptoServiceProvider.ComputeMD5String(plainTextPassword).ToLower();
                        pwds.Add(CryptoServiceProvider.ComputeMD5String(md5_password + salt).ToLower());
                        break;
                    }
                case ELegacyEncryptionType.IPB:
                default:
                    {
                        Logger.Debug("ProcessLegacyPassword - Unknown encryption type. Calling old method");
                        bool rvalue = ProcessLegacyPassword(plainTextPassword, conn);
                        if (rvalue)
                        {
                            Logger.Debug("Old Method Succeeded: There is an issue with the new method.");
                        }
                        return rvalue;
                    }
            }

            #region
            // If the site isn't known, we can't process the password, fail as normal
            if (pwds.Count == 0)
            {
                Logger.Debug("ProcessLegacyPassword - No passwords. Calling old method");
                bool rvalue = ProcessLegacyPassword(plainTextPassword, conn);
                if (rvalue)
                {
                    Logger.Debug("Old Method Succeeded: There is an issue with the new method.");
                }
                return rvalue;
            }

            object renamed = null;

            foreach (var pwd in pwds)
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "SELECT _renamed FROM userlegacy WHERE _uid=@uid AND _password=@pwd";
                    cmd.Parameters.AddWithValue("uid", ID);
                    cmd.Parameters.AddWithValue("pwd", pwd);

                    // If the legacy user isn't known, we can't process the password, fail as normal
                    renamed = cmd.ExecuteScalar();

                    if (renamed != null)
                    {
                        break;
                    }
                }
            }

            if (renamed == null)
            {
                Logger.Debug("ProcessLegacyPassword - Renamed == null. Calling old method");
                bool rvalue = ProcessLegacyPassword(plainTextPassword, conn);
                if (rvalue)
                {
                    Logger.Debug("Old Method Succeeded: There is an issue with the new method.");
                }
                return rvalue;
            }

            // Password matches the legacy imported user
            // So now we'll create our curse hash of the decrypted password and import it with a salt
            // Update the in memory stuff that is loaded NULL for legacy users
            var rnd = new Random();
            _salt = new byte[5];
            rnd.NextBytes(_salt);
            _passwordHash = GetPasswordHash(plainTextPassword);
            _plainTextPassword = plainTextPassword;

            using (var cmd = conn.CreateCommand())
            {

                cmd.CommandText = "UPDATE users SET _password=@password,_salt=@salt WHERE _uid=@uid";
                cmd.Parameters.AddWithValue("password", _passwordHash);
                cmd.Parameters.AddWithValue("salt", _salt);
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.ExecuteNonQuery();
            }

            if ((bool)renamed && !HasRenameCredit)
            {
                using (var cmd = conn.CreateCommand())
                {
                    // The user was renamed, credit them to rename themselves
                    cmd.CommandText = "INSERT INTO userrenames VALUES(@uid)";
                    cmd.Parameters.AddWithValue("uid", ID);
                    cmd.ExecuteNonQuery();
                    HasRenameCredit = true;
                }
            }

            // Hereon the legacy user should be a fully fledged normal user, with the exception of rename credits
            return true;
            #endregion
        }

        private bool ProcessLegacyPassword(string plainTextPassword, SqlConnection conn)
        {
            SqlCommand cmd = null;
            var pwds = new List<string>();

            switch (Profile.RegisteredVia)
            {
                case 9999002: //WowStead
                    {
                        cmd = conn.CreateCommand();
                        cmd.CommandText = "SELECT _salt FROM userlegacy WHERE _uid=@uid";
                        cmd.Parameters.AddWithValue("uid", ID);
                        object salt = cmd.ExecuteScalar();
                        if (salt != null && salt != System.DBNull.Value)
                        {
                            MD5 md5 = MD5.Create();
                            SHA1 sha1 = SHA1.Create();

                            string saltString = (string)salt;
                            string shaPassword = BitConverter.ToString(sha1.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", string.Empty).ToLower();
                            string md5Salt = BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(saltString))).Replace("-", string.Empty).ToLower();
                            string finalPassword = BitConverter.ToString(sha1.ComputeHash(Encoding.ASCII.GetBytes(shaPassword + md5Salt))).Replace("-", string.Empty).ToLower();

                            pwds.Add(finalPassword);
                        }
                        else
                        {
                            MD5 md5 = MD5.Create();
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", string.Empty));
                        }
                        break;
                    }
                case 1: // DEV                
                case 19838304: // WOR
                    {
                        MD5 md5 = MD5.Create();
                        pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", ""));
                        break;
                    }
                case 17395615: // AS                    
                    {
                        cmd = conn.CreateCommand();
                        cmd.CommandText = "SELECT _salt FROM userlegacy WHERE _uid=@uid";
                        cmd.Parameters.AddWithValue("uid", ID);
                        object salt = cmd.ExecuteScalar();
                        if (salt != null)
                        {
                            MD5 md5 = MD5.Create();
                            string md5_password = BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", "").ToLower();
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(md5_password + (string)salt))).Replace("-", "").ToLower());
                        }
                        break;
                    }
                case 6666668: // WowAce
                    {
                        MD5 md5 = MD5.Create();
                        SHA1 sha1 = SHA1.Create();
                        pwds.Add(BitConverter.ToString(sha1.ComputeHash(Encoding.ASCII.GetBytes(Name.ToLower() + HttpUtility.HtmlDecode(plainTextPassword.Replace("\\", ""))))).Replace("-", "").ToLower());

                        cmd = conn.CreateCommand();
                        cmd.CommandText = "SELECT _salt FROM userlegacy WHERE _uid=@uid";
                        cmd.Parameters.AddWithValue("uid", ID);
                        object salt = cmd.ExecuteScalar();
                        if (salt != null)
                        {
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))) + (string)salt))));
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes((string)salt))) + BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword)))))));
                        }
                        else
                        {
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", "").ToLower());
                            pwds.Add(BitConverter.ToString(md5.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword + Name.ToLower()))).Replace("-", "").ToLower());
                            pwds.Add(BitConverter.ToString(sha1.ComputeHash(Encoding.ASCII.GetBytes(plainTextPassword))).Replace("-", "").ToLower());
                            pwds.Add(plainTextPassword);
                        }

                        break;
                    }
                case 7777788: // Marriland
                case 9999005: // Darth Hater
                    {
                        cmd = conn.CreateCommand();
                        cmd.CommandText = "select _password from userlegacy where _uid = @uid";
                        cmd.Parameters.AddWithValue("uid", ID);
                        object passwordHash = cmd.ExecuteScalar();
                        if (passwordHash != null)
                        {
                            if (((string)passwordHash).StartsWith("$2"))
                            {
                                if (BCrypt.Net.BCrypt.Verify(plainTextPassword, (string)passwordHash))
                                {
                                    pwds.Add((string)passwordHash);
                                }
                            }//blowfish test
                            else
                            {
                                if (Profile.RegisteredVia == 9999005)
                                {
                                    pwds.Add(CryptoServiceProvider.PhpBB2.ComputeHash(ASCIIEncoding.ASCII.GetBytes(plainTextPassword), (string)passwordHash));
                                }
                                else if (Profile.RegisteredVia == 7777788)
                                {
                                    pwds.Add(CryptoServiceProvider.PhpBB3.ComputeHash(ASCIIEncoding.ASCII.GetBytes(plainTextPassword), (string)passwordHash));
                                }
                            }//non-blowfish test

                        }//check the passwords

                        break;
                    }
                default:
                    break;
            }
            // If the site isn't known, we can't process the password, fail as normal
            if (pwds.Count == 0)
            {
                return false;
            }

            object renamed = null;
            foreach (string pwd in pwds)
            {
                cmd = conn.CreateCommand();
                cmd.CommandText = "SELECT _renamed FROM userlegacy WHERE _uid=@uid AND _password=@pwd";
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.Parameters.AddWithValue("pwd", pwd);

                // If the legacy user isn't known, we can't process the password, fail as normal
                renamed = cmd.ExecuteScalar();

                if (renamed != null)
                {
                    break;
                }
            }

            if (renamed == null)
            {
                return false;
            }

            // Password matches the legacy imported user
            // So now we'll create our curse hash of the decrypted password and import it with a salt
            // Update the in memory stuff that is loaded NULL for legacy users
            var rnd = new Random();
            _salt = new byte[5];
            rnd.NextBytes(_salt);
            _passwordHash = GetPasswordHash(plainTextPassword);
            _plainTextPassword = plainTextPassword;

            cmd.Parameters.Clear();
            cmd.CommandText = "UPDATE users SET _password=@password,_salt=@salt WHERE _uid=@uid";
            cmd.Parameters.AddWithValue("password", _passwordHash);
            cmd.Parameters.AddWithValue("salt", _salt);
            cmd.Parameters.AddWithValue("uid", ID);
            cmd.ExecuteNonQuery();

            if ((bool)renamed && !HasRenameCredit)
            {
                // The user was renamed, credit them to rename themselves
                cmd.Parameters.Clear();
                cmd.CommandText = "INSERT INTO userrenames VALUES(@uid)";
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.ExecuteNonQuery();
                HasRenameCredit = true;
            }

            ExpireRemotely();

            // Hereon the legacy user should be a fully fledged normal user, with the exception of rename credits
            return true;

        }

        private void ExpireRemotely()
        {
            CacheManager.ExpireRemotely(CacheKey);
        }


        public bool CreditRename(SqlConnection conn)
        {
            try
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "INSERT INTO userrenames VALUES(@uid)";
                    cmd.Parameters.AddWithValue("uid", ID);
                    cmd.ExecuteNonQuery();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Credit rename failed");
                return false;
            }

            HasRenameCredit = true;
            ExpireRemotely();
            return true;
        }

        public bool ValidatePassword(string plainTextPassword, int siteID, SqlConnection conn)
        {

            if (_passwordHash == null && !NewProcessLegacyPassword(plainTextPassword, conn))
            {
                return false;
            }

            if (plainTextPassword == _plainTextPassword)
            {
                return true;
            }

            // Get the hash of the plain text password, based on the salt
            var hash = GetPasswordHash(plainTextPassword);

            // If any bytes do not match, return false
            for (int i = 0; i < hash.Length; ++i)
            {
                if (hash[i] != _passwordHash[i])
                {
                    return false;
                }
            }

            // Cache the plain text password for faster future lookups
            _plainTextPassword = plainTextPassword;

            return true;
        }

        public void ChangePassword(string plainTextPassword, SqlConnection sql)
        {
            if (_salt == null)
            {
                ChangePasswordAndSalt(plainTextPassword, sql);
                return;
            }

            var pwd = GetPasswordHash(plainTextPassword);

            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "UPDATE users SET _password = @pwd where _uid = @uid";
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.Parameters.AddWithValue("pwd", pwd);
                cmd.ExecuteNonQuery();
            }

            _passwordHash = pwd;
            _plainTextPassword = plainTextPassword;

            ExpireRemotely();
        }

        public void ChangePasswordAndSalt(string password, SqlConnection sql)
        {
            _salt = GenerateSalt();
            _passwordHash = GetPasswordHash(password);
            _plainTextPassword = password;

            using (var cmd = sql.CreateCommand())
            {
                cmd.Parameters.Clear();
                cmd.CommandText = "UPDATE users SET _password=@password,_salt=@salt WHERE _uid=@uid";
                cmd.Parameters.AddWithValue("password", _passwordHash);
                cmd.Parameters.AddWithValue("salt", _salt);
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.ExecuteNonQuery();
            }

            ExpireRemotely();
        }

        public ESetUserPasswordResult ChangeLegacyPassword(int siteid, int legacyid, string passwordHash, string salt, ELegacyEncryptionType encyptionType, SqlConnection sql)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "spSetUserLegacyPassword";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.AddWithValue("@SiteId", siteid);
                cmd.Parameters.AddWithValue("@LegacyId", legacyid);
                cmd.Parameters.AddWithValue("@UserId", ID);
                cmd.Parameters.AddWithValue("@Password", passwordHash);
                cmd.Parameters.AddWithValue("@Encryption", encyptionType);

                if (!string.IsNullOrEmpty(salt))
                {
                    cmd.Parameters.AddWithValue("@Salt", salt);
                }

                try
                {
                    cmd.ExecuteNonQuery();
                    return ESetUserPasswordResult.Success;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to update or create the user password.");
                    return ESetUserPasswordResult.Unsuccessful;
                }
                finally
                {
                    ExpireRemotely();
                }
            }
        }

        public void Serialize(XmlWriter to)
        {
            to.WriteStartElement("user");

            to.WriteStartAttribute("id");
            to.WriteValue(ID);
            to.WriteEndAttribute();

            to.WriteStartAttribute("name");
            to.WriteValue(Name);
            to.WriteEndAttribute();

            to.WriteStartAttribute("level");
            to.WriteValue(Level);
            to.WriteEndAttribute();

            Profile.Serialize(to);

            to.WriteEndElement();
        }

        public void SerializeSubscriptions(XmlWriter to)
        {
            to.WriteStartElement("subscriptions");
            foreach (UserSubscripion subscription in _subscriptions)
            {
                subscription.serialize(to);
            }
            to.WriteEndElement();
        }

        public void SetPremiumExpiration(SqlConnection sql, DateTime expiration, Byte level)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "spSetSubscriptionExpiration";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("@numUserID", SqlDbType.Int);

                cmd.Parameters["@numUserID"].Value = ID;
                cmd.Parameters.Add("@dtExpirationDate", SqlDbType.DateTime);

                cmd.Parameters["@dtExpirationDate"].Value = expiration;
                cmd.Parameters.Add("@numLevel", SqlDbType.TinyInt);

                // Subscription Level: Monthly, Quarterly, Annual
                cmd.Parameters["@numLevel"].Value = level;
                cmd.Parameters.Add("@numType", SqlDbType.SmallInt);

                // Subscription Type: Always premium for now
                cmd.Parameters["@numType"].Value = ESubscriptionType.Premium;
                cmd.ExecuteNonQuery();
            }

            ExpireRemotely();

            // Re-pop the subscriptions:
            LoadSubscriptions(sql);
        }

        public void AddSubscriptionCredit(SqlConnection sql, Byte months, Byte level, Byte type)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "spAddSubscriptionCredit";
                cmd.CommandType = CommandType.StoredProcedure;

                // User ID
                cmd.Parameters.Add("@numUserID", SqlDbType.Int);
                cmd.Parameters["@numUserID"].Value = ID;

                // Months
                cmd.Parameters.Add("@numMonths", SqlDbType.TinyInt);
                cmd.Parameters["@numMonths"].Value = months;

                // Subscription Level: Monthly, Quarterly, Annual
                cmd.Parameters.Add("@numLevel", SqlDbType.TinyInt);
                cmd.Parameters["@numLevel"].Value = level;

                // Subscription Type: Always premium for now
                cmd.Parameters.Add("@numType", SqlDbType.SmallInt);
                cmd.Parameters["@numType"].Value = type;
                cmd.ExecuteNonQuery();
            }

            ExpireRemotely();

            // Re-pop the subscriptions:
            LoadSubscriptions(sql);
        }

        public void TerminateSubscription(SqlConnection sql, Byte type)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "spTerminateSubscription";
                cmd.CommandType = CommandType.StoredProcedure;

                // User ID
                cmd.Parameters.Add("@numUserID", SqlDbType.Int);
                cmd.Parameters["@numUserID"].Value = ID;

                // Subscription Type: Always premium for now
                cmd.Parameters.Add("@numType", SqlDbType.SmallInt);
                cmd.Parameters["@numType"].Value = type;

                cmd.ExecuteNonQuery();
            }

            ExpireRemotely();

            // Re-pop the subscriptions:
            LoadSubscriptions(sql);
        }

        public void CancelPremium(SqlConnection sql, DateTime expiration)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "spCancelSubscription";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("@numUserID", SqlDbType.Int);
                cmd.Parameters["@numUserID"].Value = ID;
                cmd.Parameters.Add("@dtExpirationDate", SqlDbType.DateTime);
                cmd.Parameters["@dtExpirationDate"].Value = expiration;
                cmd.Parameters.Add("@numType", SqlDbType.SmallInt);
                cmd.Parameters["@numType"].Value = ESubscriptionType.Premium;
                cmd.ExecuteNonQuery();
            }

            ExpireRemotely();

            // Re-pop the subscriptions:
            LoadSubscriptions(sql);
        }

        public string CreateSession(SqlConnection sql, string session, int siteID)
        {

            if (string.IsNullOrEmpty(session))
            {
                throw new ArgumentException("session cannot be null", "session");
            }

            // Insert the new session into the database
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.CommandText = "spSetUserSessionV2";
                cmd.Parameters.AddWithValue("session", session);
                cmd.Parameters.AddWithValue("uid", this.ID);
                cmd.Parameters.AddWithValue("siteid", siteID);
                cmd.ExecuteNonQuery();
            }

            // Update the cache            
            CacheSession(session, ID);
            ExpireRemotely();
            return session;
        }

        public void ClearUserSessions(SqlConnection sql)
        {

            using (var cmd = sql.CreateCommand())
            {
                // Find any session which may exist for this user in the databse
                cmd.CommandText = "SELECT _session from usersessions WHERE _uid = @uid";
                cmd.Parameters.AddWithValue("uid", this.ID);
                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var oldSession = reader.GetString(0);

                        // If a session exists in the DB, remove it from our cache
                        if (oldSession != null)
                        {
                            ClearSession(oldSession);
                        }
                    }
                }

                cmd.Parameters.Clear();
                cmd.CommandText = "DELETE FROM usersessions WHERE _uid = @uid";
                cmd.Parameters.AddWithValue("uid", this.ID);
                cmd.ExecuteNonQuery();
            }

        }

        public void SetLevel(SqlConnection sql, int level)
        {
            using (var cmd = sql.CreateCommand())
            {

                cmd.CommandText = "UPDATE users SET _level = @level WHERE _uid = @uid";
                cmd.Parameters.AddWithValue("level", level);
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.ExecuteNonQuery();
            }

            Level = level;
            ExpireRemotely();
        }

        public bool GetSubscriptionStatus(ESubscriptionType type)
        {
            foreach (UserSubscripion sub in _subscriptions)
            {
                if (sub.Type == type)
                {
                    return sub.Active;
                }
            }
            return false;
        }

        public UserSubscripion GetActiveSubscription(ESubscriptionType type)
        {
            foreach (UserSubscripion sub in _subscriptions)
            {
                if (sub.Type == type && sub.Active)
                {
                    return sub;
                }
            }
            return null;
        }

        public Byte GetPremiumLevel()
        {
            return GetSubscriptionLevel(ESubscriptionType.Premium);
        }

        public void AddUseroAuthData(int userId, int oAuthPlatform, string oAuthId, SqlConnection sql)
        {
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "INSERT into useroauth (_uid, _platformType, _platformID) VALUES (@userId, @oAuthPlatform, @oAuthId);";
                cmd.Parameters.AddWithValue("@userId", userId);
                cmd.Parameters.AddWithValue("@oAuthPlatform", oAuthPlatform);
                cmd.Parameters.AddWithValue("@oAuthId", oAuthId);
                cmd.ExecuteNonQuery();
            }
        }

        public void AcceptPolicy(SqlConnection sql)
        {
            var policyAccepted = DateTime.UtcNow;
            using (var cmd = sql.CreateCommand())
            {
                cmd.CommandText = "UPDATE users SET _policyAccepted=@policyAccepted WHERE _uid=@uid";
                cmd.Parameters.AddWithValue("@uid", ID);
                cmd.Parameters.AddWithValue("@policyAccepted", policyAccepted);
                cmd.ExecuteNonQuery();
            }

            PolicyAcceptDate = policyAccepted;
            ExpireRemotely();
        }

        #endregion

        #region Private Instance Methods

        private void LoadLegacyData(SqlConnection conn)
        {
            _legacy = UserLegacy.GetAllByUserID(ID, conn);
        }

        private byte[] GetPasswordHash(string password)
        {
            return GetPasswordHash(password, _salt);
        }

        private static byte[] GetPasswordHash(string password, byte[] salt)
        {
            var strbytes = Encoding.UTF8.GetBytes(password);
            var combined = new byte[strbytes.Length + salt.Length];

            Buffer.BlockCopy(salt, 0, combined, 0, salt.Length);
            Buffer.BlockCopy(strbytes, 0, combined, 5, strbytes.Length);

            var sha1 = new SHA1Managed();
            var hash = sha1.ComputeHash(combined);
            return hash;
        }

        private void LoadSubscriptions(SqlConnection conn)
        {
            _subscriptions = UserSubscripion.GetUserSubscriptions(ID, conn);
        }

        private Byte GetSubscriptionLevel(ESubscriptionType type)
        {
            foreach (UserSubscripion sub in _subscriptions)
            {
                if (sub.Type == type)
                {
                    return sub.Level;
                }
            }
            return 0;
        }

        #endregion
    }
}
