using System;
using System.Data;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Xml;

namespace Curse.Auth
{
    public class UserEmail
    {
        const int iId = 0;
        const int iAddress = 1;
        const int iVerified = 2;

        private static readonly Dictionary<int, UserEmail> _emailsByID = new Dictionary<int, UserEmail>();
        private static readonly Dictionary<string, int> _emailIDByEmailAddress = new Dictionary<string, int>(StringComparer.InvariantCultureIgnoreCase);
        
        public UserEmail() { }
        

        public UserEmail(int id,
                         string address,
                         bool verified)
        {
            ID = id;
            Address = address;
            Verified = verified;
        }

        static public List<UserEmail> GetUserEmails(int uid, SqlConnection from)
        {
            var emails = new List<UserEmail>();
            
            SqlCommand cmd = new SqlCommand("spEmailFromUserID", from);
            cmd.CommandType = CommandType.StoredProcedure;
            cmd.Parameters.Add("@numUserID", SqlDbType.Int);
            cmd.Parameters["@numUserID"].Value = uid;

            using (SqlDataReader reader = cmd.ExecuteReader())
            {                
                while (reader.Read())
                {
                    UserEmail email =
                        new UserEmail(reader.GetInt32(iId),
                                      reader.GetString(iAddress),
                                      reader.GetBoolean(iVerified));
                    emails.Add(email);
                }                
            }

            return emails;
        }


        public static UserEmail GetByID(int id)
        {

            UserEmail email;
            lock (_emailsByID)
            {
                if (_emailsByID.TryGetValue(id, out email))
                {
                    return email;
                }
            }

            using (SqlConnection sql = DatabaseUtility.GetAuthConnection())
            {
                email = CreateByID(id, sql);
            }

            if (email == null)
            {
                throw new KeyNotFoundException("No such e-mail");
            }

            lock (_emailsByID)
            {
                _emailsByID[id] = email;
            }

            return email;
        }

        public static UserEmail GetByEmailAddress(string name)
        {
            int emailID = GetIDByEmailAddress(name);

            if (emailID > 0)
            {
                return GetByID(emailID);
            }
            else
            {
                throw new KeyNotFoundException("No such e-mail address");
            }
        }

        public static int GetIDByEmailAddress(string emailAddress)
        {

            lock (_emailIDByEmailAddress)
            {
                if (_emailIDByEmailAddress.ContainsKey(emailAddress))
                {
                    return _emailIDByEmailAddress[emailAddress];
                }
            }

            int? emailID;

            using (SqlConnection conn = DatabaseUtility.GetAuthConnection())
            {
                SqlCommand cmd = new SqlCommand("spEmailIDFromAddress", conn);
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("@strEmail", SqlDbType.VarChar, 64);
                cmd.Parameters["@strEmail"].Value = emailAddress;
                emailID = (int?)cmd.ExecuteScalar();

                if (emailID == null)
                {
                    _emailIDByEmailAddress[emailAddress] = 0;
                }
                else
                {
                    _emailIDByEmailAddress[emailAddress] = (int)emailID;
                }
            }

            return _emailIDByEmailAddress[emailAddress];
        }
        
        private static UserEmail CreateByID(int id, SqlConnection from)
        {            

            SqlCommand cmd = from.CreateCommand();
            cmd.CommandText = "SELECT _id, _email, _verified FROM useremails WHERE _id = @id";
            cmd.Parameters.AddWithValue("id", id);

            UserEmail email;                
            using (SqlDataReader reader = cmd.ExecuteReader())
            {
                if (!reader.Read())
                {
                    return null;
                }
                
                email = new UserEmail(reader.GetInt32(iId), reader.GetString(iAddress), reader.GetBoolean(iVerified));                
            }            
            return email;
        }
        
        public void Serialize(XmlWriter to)
        {
            to.WriteStartElement("email");

            to.WriteStartAttribute("id");
            to.WriteValue(ID);
            to.WriteEndAttribute();

            to.WriteStartAttribute("address");
            to.WriteValue(Address);
            to.WriteEndAttribute();

            to.WriteStartAttribute("verified");
            to.WriteValue(Verified);
            to.WriteEndAttribute();

            to.WriteEndElement();
        }
        
        public void Update(SqlConnection sql, string address)
        {
            SqlCommand cmd = sql.CreateCommand();
            cmd.CommandText = 
                "UPDATE useremails SET " +
                    "_email = @email " +
                "WHERE _id = @id";
            cmd.Parameters.AddWithValue("email", address);
            cmd.Parameters.AddWithValue("id", ID);
            cmd.ExecuteNonQuery();
            
            Address = address;
        }
        
        public void verify(SqlConnection sql)
        {
            SqlCommand cmd = sql.CreateCommand();
            cmd.CommandText = 
                "UPDATE useremails " +
                "SET _verified = 1 " +
                "WHERE _id = @id";
            cmd.Parameters.AddWithValue("id", ID);
            cmd.ExecuteNonQuery();

            Verified = true;
        }

        public string Address { get; set; }
        public bool Verified { get; set; }
        public int ID { get; set; }
               
    }
}
