using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data.SqlClient;
using System.Web.UI;
using System.Xml;

namespace Curse.Auth
{
    /// <summary>
    /// Global user profile. Used for the profile which is shared between
    /// all network sites in the Curse network.
    /// </summary>
    public class UserProfile
    {
        const int iFirstName = 0;
        const int iLastName = 1;
        const int iGender = 2;
        const int iDefaultEmail = 3;
        const int iCountry = 4;
        const int iRegion = 5;
        const int iCity = 6;
        const int iWantsMail = 7;
        const int iRegisteredVia = 8;
        const int iRegisteredOn = 9;

        public int ID { get; set; }
        public string FirstName { get; set; }
        public string LastName { get; set; }
        public DateTime? BirthDate { get; set; }
        public int? Gender { get; set; }
        public List<UserEmail> Emails { get; set; }
        public int PrimaryEmail { get; set; }
        public string Country { get; set; }
        public string Region { get; set; }
        public string City { get; set; }
        public bool WantsEmail { get; set; }
        public int RegisteredVia { get; set; }
        public DateTime RegisteredOn { get; set; }

        public UserProfile(){}

        public UserProfile(int id, SqlConnection conn)
        {

            using (var cmd = conn.CreateCommand())
            {
                cmd.CommandText =
                "SELECT " +
                        "_firstName, " +
                        "_lastName, " +
                        "_gender," +
                        "_defaultEmail," +
                        "_country," +
                        "_region," +
                        "_city," +
                        "_wantsmail," +
                        "_registeredVia," +
                        "_registeredOn " +
                "FROM userprofiles " +
                "WHERE _uid = @uid";

                cmd.Parameters.AddWithValue("uid", id);

                using (var reader = cmd.ExecuteReader())
                {
                    if (!reader.Read())
                        throw new KeyNotFoundException("No such user.");

                    ID = id;
                    FirstName = reader.IsDBNull(iFirstName) ? null : reader.GetString(iFirstName);
                    LastName = reader.IsDBNull(iLastName) ? null : reader.GetString(iLastName);
                    Gender = reader.IsDBNull(iGender) ? null : (byte?)reader.GetByte(iGender);                                       
                    Country = reader.GetString(iCountry);
                    Region = reader.IsDBNull(iRegion) ? null : reader.GetString(iRegion);
                    City = reader.IsDBNull(iCity) ? null : reader.GetString(iCity);
                    WantsEmail = reader.GetBoolean(iWantsMail);
                    RegisteredVia = reader.GetInt32(iRegisteredVia);
                    RegisteredOn = reader.GetDateTime(iRegisteredOn);
                    PrimaryEmail = reader.GetInt32(iDefaultEmail);
                }
            }                        

            Emails = UserEmail.GetUserEmails(id, conn);
        }
        
        public void Serialize(XmlWriter to)
        {
            to.WriteStartElement("profile");

            if (FirstName != null)
            {
                to.WriteStartElement("firstName");
                to.WriteValue(FirstName);
                to.WriteEndElement();
            }

            if(LastName != null)
            {
                to.WriteStartElement("lastName");
                to.WriteValue(LastName);
                to.WriteEndElement();
            }
            
            if(Gender != null)
            {
                to.WriteStartElement("gender");
                to.WriteValue(Gender);
                to.WriteEndElement();
            }

            to.WriteStartElement("country");
            to.WriteValue(Country);
            to.WriteEndElement();

            if(Region != null)
            {
                to.WriteStartElement("region");
                to.WriteValue(Region);
                to.WriteEndElement();
            }
            
            if(City != null)
            {
                to.WriteStartElement("city");
                to.WriteValue(City);
                to.WriteEndElement();
            }
            
            to.WriteStartElement("wantsEmail");
            to.WriteValue(WantsEmail);
            to.WriteEndElement();
            
            to.WriteStartElement("registeredVia");
            to.WriteValue(RegisteredVia);
            to.WriteEndElement();
            
            to.WriteStartElement("registeredOn");
            //Y-m-dT%H:%M:%S
            to.WriteValue(RegisteredOn);
            to.WriteEndElement();

            to.WriteStartElement("emails");
            
            to.WriteStartAttribute("primary");
            to.WriteValue(PrimaryEmail);
            to.WriteEndAttribute();
            
            foreach(UserEmail i in Emails)
                i.Serialize(to);
            
            to.WriteEndElement();

            to.WriteEndElement();
        }
        
        public void Update(SqlConnection sql,
                           string firstname,
                           string lastname,
                           string gender,
                           string country,
                           string region,
                           string city,
                           int primaryemail,
                           bool newsletter)
        {

            using (var cmd = sql.CreateCommand())
            {

                cmd.CommandText =
                    "UPDATE userprofiles SET " +
                        "_firstName = @fname," +
                        "_lastName = @lname," +
                        "_gender = @gender," +
                        "_defaultEmail = @email," +
                        "_country = @country," +
                        "_region = @region," +
                        "_city = @city," +
                        "_wantsmail = @newsletter " +
                    "WHERE _uid = @uid";

                cmd.Parameters.AddWithValue("fname", firstname);
                cmd.Parameters.AddWithValue("lname", DatabaseUtility.GetOptionalString(lastname));
                cmd.Parameters.AddWithValue("gender", gender);
                cmd.Parameters.AddWithValue("email", primaryemail);
                cmd.Parameters.AddWithValue("country", country);
                cmd.Parameters.AddWithValue("region", DatabaseUtility.GetOptionalString(region));
                cmd.Parameters.AddWithValue("city", DatabaseUtility.GetOptionalString(city));
                cmd.Parameters.AddWithValue("newsletter", newsletter);
                cmd.Parameters.AddWithValue("uid", ID);
                cmd.ExecuteNonQuery();
            }            
        }
        
        public UserEmail GetEmailByID(int id)
        {
            UserEmail found = null;
            foreach(UserEmail email in Emails)
            {
                if(email.ID == id)
                {
                    found = email;
                    break;
                }
            }
            
            return found;        
        }
       
        internal UserEmail GetPrimaryEmail()
        {
            return GetEmailByID(PrimaryEmail);
        }
        
        [Obsolete]
        public void SwitchPrimaryEmail(SqlConnection sql, int id)
        {
            if(GetEmailByID(id) == null)
                throw new KeyNotFoundException("New primary email " +
                                               "address doesn't exist.");
            
            SqlCommand cmd = sql.CreateCommand();
            cmd.CommandText =
                "UPDATE userprofiles SET _defaultEmail = @id " +
                "WHERE _uid = @uid";
            cmd.Parameters.AddWithValue("id", id);
            cmd.Parameters.AddWithValue("uid", ID);
            cmd.ExecuteNonQuery();
        }
    }
}
