﻿using System;
using System.Data.SqlClient;
using System.IO;
using System.ServiceModel;
using System.ServiceModel.Web;
using System.Text;
using System.Configuration;
using Curse.AzerothService.Caching;
using Curse.AzerothService.Managers;
using Curse.Extensions;
using Curse.ServiceModels.Configuration;
using Curse.ServiceModels.Authentication;

namespace Curse.AzerothService
{
    [ServiceBehavior(Name = "AzerothService", Namespace = "http://azeroth.curse.com/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.PerSession)]
    public class AzerothService : IAzerothService
    {
        static object _startLock = new object();
        static bool _serviceStarted;

        static AzerothService()
        {
            lock (_startLock)
            {
                _serviceStarted = false;
                try
                {
                    var config = ServiceConfiguration.Instance;
                    
                    #region Init Logger
                    string logLevel = config.Logging.LogLevel;
                    switch (logLevel.ToLower())
                    {
                        case "debug":
                            Logger.SetLogLevel = ELogLevel.Debug;
                            break;
                        case "access":
                            Logger.SetLogLevel = ELogLevel.Access;
                            break;
                        case "error":
                            Logger.SetLogLevel = ELogLevel.Error;
                            break;
                        case "info":
                            Logger.SetLogLevel = ELogLevel.Info;
                            break;
                        case "warning":
                            Logger.SetLogLevel = ELogLevel.Warning;
                            break;
                    }
                    Logger.SetAutoGrowthMegabytes = 50;
                    Logger.SetLogPath = config.Logging.LogPath;
                    #endregion

                    Logger.Log("Initializing the UpdateCache...", ELogLevel.Info);
                    UpdateCache.Instance.Initialize();

                    Logger.Log("Initializing the ParserManager...", ELogLevel.Info);
                    ParserManager.Instance.Initialize();

                    Logger.Log("Service Starting...", ELogLevel.Info);
                    _serviceStarted = true;
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to start service. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                }
            }
        }

        #region Service Methods

        public void UploadUpdate(Stream input)
        {
            if (!_serviceStarted)
            {
                return;
            }

            try
            {
                int userId = 0;
                bool authenticationRequired = bool.Parse(ConfigurationManager.AppSettings["AuthenticationRequired"]);
                if (authenticationRequired)
                {
                    string sessionId = WebOperationContext.Current.IncomingRequest.Headers["session"];
                    var status = ServiceAuthentication.ValidateSessionKeyToUserId(sessionId, out userId);
                    if (status != StatusCode.Ok)
                    {
                        Logger.Log("ValidateSessionKeyToUserId failed with status {0}", ELogLevel.Access, status);
                        return;
                    }
                }
                Logger.Log("Received {0} bytes for update file stream", ELogLevel.Debug, WebOperationContext.Current.IncomingRequest.ContentLength);

                byte[] data = new BinaryReader(input).ReadBytes((int)WebOperationContext.Current.IncomingRequest.ContentLength);
                UpdateCache.Instance.QueueUpdate(new Package(Encoding.UTF8, data), userId);
            }
            catch(Exception exc)
            {
                Logger.Log("UploadUpdate Exception: {0}", ELogLevel.Error, exc.GetExceptionDetails());
            }
        }

        public string HealthCheck()
        {
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["AzerothService"].ConnectionString))
            {
                conn.Open();
            }

            return "Success";
        }
        #endregion
    }
}
