﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using Curse.ServiceModels.Models;
using System.Collections.Concurrent;
using Curse.Wow.DataLib.Extraction;
using Curse.Extensions;
using System.Configuration;
using System.Data.SqlClient;
using Curse.Wow.DataLib.DataFiles;

namespace Curse.AzerothService.Caching
{
    public class DbCache
    {
        ConcurrentQueue<WowFile> _dbQueue;
        ConcurrentDictionary<int, DateTime> _updateTimes;
        WorkerThread<bool> _processThread;
        WorkerThread<bool> _fileUpdateThread;
        List<string> _acceptedDownloads;
        List<int> _trustedUsers;
        object _acceptedDownloadLock = new object();
        DateTime _lastUpdateTime = DateTime.Parse("01/01/2012");
        string _wowServiceConnection = string.Empty;
        string _azerothServiceConnection = string.Empty;
        double _rateLimit;
        int _maxQueuedUpdates;
        bool _backupDbFiles = false;

        static DbCache _instance = new DbCache();
        public static DbCache Instance { get { return _instance; } }
        public List<string> AcceptedFileTypes { get { return _acceptedDownloads; } }

        public DbCache()
        {
            _dbQueue = new ConcurrentQueue<WowFile>();
            _updateTimes = new ConcurrentDictionary<int, DateTime>();
            _acceptedDownloads = new List<string>();

            _processThread = new WorkerThread<bool>(() => ProcessQueueThread(), TimeSpan.FromMilliseconds(1000), true);
            _fileUpdateThread = new WorkerThread<bool>(() => UpdateFileNameCache(), TimeSpan.FromMilliseconds(30000), true);

            _wowServiceConnection = ConfigurationManager.ConnectionStrings["WowService"].ConnectionString;
            _azerothServiceConnection = ConfigurationManager.ConnectionStrings["AzerothService"].ConnectionString;
            _rateLimit = double.Parse(ConfigurationManager.AppSettings["RateLimit"]);
            _maxQueuedUpdates = int.Parse(ConfigurationManager.AppSettings["MaxQueuedUpdates"]);
            _backupDbFiles = bool.Parse(ConfigurationManager.AppSettings["BackupDbFiles"]);

            _trustedUsers = new List<int>();
            var userIds = ConfigurationManager.AppSettings["TrustedUSers"].Split(',');
            foreach (string userId in userIds)
            {
                _trustedUsers.Add(Int32.Parse(userId));
            }
        }

        public void Initialize()
        {
        }

        bool UpdateFileNameCache()
        {
            try
            {
                List<string> acceptedDownloads = new List<string>();
                using (var conn = new SqlConnection(_azerothServiceConnection))
                {
                    var currTime = DateTime.UtcNow.AddMinutes(-5);
                    var cmd = new SqlCommand("SELECT FileName FROM DbFileTypes WHERE AcceptUploads = 1", conn);

                    conn.Open();
                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var fileName = reader.GetString(reader.GetOrdinal("FileName"));
                            acceptedDownloads.Add(fileName);
                        }
                    }

                    lock (_acceptedDownloadLock)
                    {
                        _acceptedDownloads = acceptedDownloads;
                        _lastUpdateTime = currTime;
                    }
                }
            }
            catch (Exception exc)
            {
                Logger.Log("UpdateFileNameCache - Failed to pull filenames from the DB. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
            }
            return true;
        }
        bool ProcessQueueThread()
        {
            WowFile dbFile = null;
            
            try
            {
                // Try to get an update
                lock (_dbQueue)
                {
                    if (!_dbQueue.TryDequeue(out dbFile))
                    {
                        return false;
                    }
                }

                try
                {
                    dbFile.LoadFormatFromDatabase();
                    dbFile.LoadDataFromFile();
                    dbFile.ExportToDatabase(true, false);
                }
                catch (Exception ex)
                {
                    LogErrorToDatabase(dbFile.WowFileBackupID, ex);
                }
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Debug, null, "ProcessUpdateThread Exception: {0}", exc.GetExceptionDetails());
            }
            return true;
        }

        public void QueueUpload(int userId, string fileName, Stream stream)
        {

            int rowID = 0;
            // Check the Rate Limit
            if (IsRateLimitExceeded(userId))
            {
                return;
            }

            if (_backupDbFiles)
            {
                rowID = SaveFileToDatabase(userId, fileName, stream);
            }

            try
            {
                var extension = string.Empty;
                if (fileName.Contains('.'))
                {
                    extension = fileName.Substring(fileName.LastIndexOf('.'));
                }

                var dbFile = new WowFile(stream, extension, _wowServiceConnection) { WowFileBackupID = rowID };
                lock (_dbQueue)
                {
                    if (_dbQueue.Count < _maxQueuedUpdates || IsTrustedUser(userId))
                    {
                        _dbQueue.Enqueue(dbFile);
                    }
                }
            }
            catch (SilentException exc)
            {
                Logger.Log("Invalid Locale. Details: {0}", ELogLevel.Info, exc.Message);
            }
            catch (Exception exc)
            {
                LogErrorToDatabase(rowID, exc);
                Logger.Log("Failed to Queue DbFile. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
            }
        }

        public bool IsRateLimitExceeded(int userId)
        {
            if (IsTrustedUser(userId))
            {
                return false;
            }
            DateTime now = DateTime.UtcNow;
            DateTime lastupdate;

            if (!_updateTimes.TryGetValue(userId, out lastupdate) || now.Subtract(lastupdate).TotalSeconds > _rateLimit)
            {
                _updateTimes[userId] = DateTime.UtcNow;
                return false;
            }
            else
            {
                Logger.IgnoredAction(userId, "", "Update", "Exceeded update rate limit");
                return true;
            }
        }

        bool IsTrustedUser(int userId)
        {
            //return _trustedUsers.Contains(userId);
            return true;
        }

        int SaveFileToDatabase(int userId, string fileName, Stream stream)
        {
            byte[] buffer = new byte[stream.Length];
            stream.Read(buffer, 0, buffer.Length);
            stream.Position = 0;

            int rowID = 0;
            
            using (var conn = new SqlConnection(_azerothServiceConnection))
            {
                var cmd = new SqlCommand("INSERT INTO DbFileBackup (UserID, FileName, FileData) VALUES (@UserID, @FileName, @FileData) SET @ID = SCOPE_IDENTITY()", conn);
                cmd.Parameters.Add("ID", System.Data.SqlDbType.Int).Direction = System.Data.ParameterDirection.Output;
                cmd.Parameters.Add("UserID", System.Data.SqlDbType.Int).Value = userId;
                cmd.Parameters.Add("FileName", System.Data.SqlDbType.VarChar).Value = fileName;
                cmd.Parameters.Add("FileData", System.Data.SqlDbType.VarBinary).Value = buffer;

                try
                {
                    conn.Open();
                    
                    try
                    {
                        cmd.ExecuteNonQuery();
                        rowID = Convert.ToInt32(cmd.Parameters["ID"].Value);
                    }
                    catch (Exception ex)
                    {
                        Logger.Log("Converting RowID from output ID when saving uploaded file - Failed. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                }
                catch (Exception exc)
                {
                    Logger.Log("SaveFileToDatabase - Failed. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                }
                return rowID;
            }
        }
        void LogErrorToDatabase(int rowId, Exception exc)
        {
            using (var conn = new SqlConnection(_azerothServiceConnection))
            {
                var cmd = new SqlCommand("UPDATE DbFileBackup SET ParsingError = @ParsingError WHERE ID = @ID", conn);
                cmd.Parameters.Add("ID", System.Data.SqlDbType.Int).Value = rowId;
                cmd.Parameters.Add("ParsingError", System.Data.SqlDbType.VarChar).Value = exc.Message + " ----- " + exc.StackTrace;

                conn.Open();
                cmd.ExecuteNonQuery();
            }
        }
    }
}