﻿using System;
using System.Collections.Generic;
using System.Collections.Concurrent;
using System.Configuration;
using Curse.AzerothService.Managers;
using Curse.AzerothService.Models;
using Curse.Extensions;
using Curse.ServiceModels.Models;
using Curse.Azeroth.LuaImporter;

namespace Curse.AzerothService.Caching
{
    public class UpdateCache
    {
        ConcurrentQueue<IUpdate> _updateQueue;
        WorkerThread<bool> _updateThread;
        Dictionary<int, DateTime> _updateTimes;

        int _rateLimit;
        long _minProfilerVersion;
        int[] _clientVersions;
        string[] _supportedLocals;
        int _maxQueuedUpdates;

        static UpdateCache _instance = new UpdateCache();
        public static UpdateCache Instance { get { return _instance; } }
        public UpdateCache()
        {
            _updateQueue = new ConcurrentQueue<IUpdate>();
            _updateTimes = new Dictionary<int, DateTime>();

            // TODO: Need to pull the rest of these in
            _rateLimit = int.Parse(ConfigurationManager.AppSettings["RateLimit"]);
            _minProfilerVersion = long.Parse(ConfigurationManager.AppSettings["MinProfilerVersion"]);
            _clientVersions = new int[] { 1, 2 };
            _supportedLocals = new string[] { "en" };
            _maxQueuedUpdates = int.Parse(ConfigurationManager.AppSettings["MaxQueuedUpdates"]);

            _updateThread = new WorkerThread<bool>(() => ProcessUpdateThread(), TimeSpan.FromMilliseconds(1000), true);
        }
        public void Initialize() { }

        bool ProcessUpdateThread()
        {
            IUpdate update = null;

            try
            {
                // Try to get an update
                lock (_updateQueue)
                {
                    if (!_updateQueue.TryDequeue(out update))
                    {
                        return false;
                    }
                }

                // Check the Rate Limit
                if (IsRateLimitExceeded(update))
                {
                    return false;
                }

                CPUMonitor monitorTotal = new CPUMonitor("ProcessUpdateThread for User {0}", update.UserId);
                ParserEngine parser = ParserManager.Instance.GetParser();
                if (parser == null)
                {
                    throw new Exception("ProcessUpdateThread: Failed to get parser");
                }
                parser.Parse(update.UpdateData, update.UserId);

                Logger.Log(ELogLevel.Debug, update.UserHost, monitorTotal.ToString());
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Debug, null, "ProcessUpdateThread Exception: {0}", exc.GetExceptionDetails());
            }
            return true;
        }
        bool IsRateLimitExceeded(IUpdate update)
        {
            if (update.IsTrustedUser)
            {
                return false;
            }
            DateTime now = DateTime.UtcNow;
            DateTime lastupdate;

            if (!_updateTimes.TryGetValue(update.UserId, out lastupdate) || now.Subtract(lastupdate).TotalSeconds > _rateLimit)
            {
                _updateTimes[update.UserId] = DateTime.UtcNow;
                return false;
            }
            else
            {
                Logger.IgnoredAction(update.UserId, update.UserHost, "Update", "Exceeded update rate limit");
                return true;
            }  
        }

        public void QueueUpdate(Package package, int userId)
        {
            try
            {
                var update = new GameUpdate(_minProfilerVersion, _clientVersions, _supportedLocals, userId, IsTrustedUser(userId));
                if (!update.Read(package))
                {
                    switch (update.UpdateStatus)
                    {
                        case StatusCode.InvalidStream:
                            Logger.Log(ELogLevel.Error, update.UserHost, "Invalid update file stream");
                            return;
                        case StatusCode.InvalidGameClientVersion:
                            Logger.Log(ELogLevel.Error, update.UserHost, "Unsupported game client version: " + update.ClientVersion.Value);
                            return;
                        case StatusCode.InvalidProfilerVersion:
                            Logger.Log(ELogLevel.Error, update.UserHost, "Out of date profiler version: " + update.ProfilerVersion);
                            return;
                    }
                }

                lock (_updateQueue)
                {
                    if (_updateQueue.Count >= _maxQueuedUpdates && !update.IsTrustedUser)
                    {
                        update.Clear();
                    }
                    _updateQueue.Enqueue(update);
                }
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to Queue Update. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
            }
        }
        bool IsTrustedUser(int userId)
        {
            // TODO:  make this run off the config
            return true;
        }
    }
}