﻿using System;
using System.Linq;
using Curse.Azeroth.DB;
using System.Data.SqlClient;
using Curse.Azeroth.Models;
using System.Data;
using System.Collections.Generic;
using System.Configuration;
using Curse.Azeroth.Models.WoW;
using Curse.Azeroth.Models.Extensions;

namespace Curse.Azeroth.DbObjects.Wow
{
    public class NpcDb : DbObject
    {
        public override void Save(IUpdate update, SqlConnection conn)
        {
            if (update is WowUpdate)
            {
                var wowUpdate = update as WowUpdate;
                var bulkCopy = new SqlBulkCopy(conn);

                #region Table Declarations
                var vendorItemTable = new DataTable();
                var npcStatTable = new DataTable();
                var npcSpellTable = new DataTable();
                var npcDropTable = new DataTable();
                var npcDropCountTable = new DataTable();
                var npcReputationTable = new DataTable();
                var npcLocationTable = new DataTable();
                var npcReactionTable = new DataTable();
                var tradeSkillRequirements = new DataTable();

                // Load Schemas
                DbUtilities.LoadSchema(ref vendorItemTable, "VendorItems", conn);
                DbUtilities.LoadSchema(ref npcStatTable, "NpcStats", conn);
                DbUtilities.LoadSchema(ref npcSpellTable, "NpcSpell", conn);
                DbUtilities.LoadSchema(ref npcDropTable, "[Drop]", conn);
                DbUtilities.LoadSchema(ref npcDropCountTable, "DropCount", conn);
                DbUtilities.LoadSchema(ref npcReputationTable, "NpcReputation", conn);
                DbUtilities.LoadSchema(ref npcLocationTable, "Location", conn);
                DbUtilities.LoadSchema(ref npcReactionTable, "NpcReaction", conn);
                DbUtilities.LoadSchema(ref tradeSkillRequirements, "TradeSkillRequirement", conn);
                #endregion

                // Populate IDs
                PopulateExtendedCostIDs(ref wowUpdate);
                PopulateFactionIDs(ref wowUpdate);
                PopulateClassIDs(ref wowUpdate);
                PopulateProfessionIDs(ref wowUpdate);

                foreach (var npc in wowUpdate.Npcs)
                {
                    #region Vendor Items
                    foreach (var item in npc.VendorItems)
                    {
                        var itemRow = vendorItemTable.NewRow();
                        itemRow["Build"] = wowUpdate.Build;
                        itemRow["NpcID"] = npc.Id;
                        itemRow["ItemID"] = item.Id;
                        itemRow["Stack"] = item.StackSize;
                        itemRow["LimitedAvailability"] = item.LimitedAvailability;
                        itemRow["Price"] = item.Price;
                        itemRow["ExtendedCost"] = item.ExtendedCostID;
                        itemRow["DateCreated"] = DateTime.UtcNow;

                        vendorItemTable.Rows.Add(itemRow);
                    }
                    #endregion

                    #region Reaction
                    if (npc.Reaction != null)
                    {
                        var reactionRow = npcReactionTable.NewRow();
                        reactionRow["Build"] = wowUpdate.Build;
                        reactionRow["NpcID"] = npc.Id;
                        reactionRow["SideID"] = npc.Reaction.Faction;
                        reactionRow["Reaction"] = npc.Reaction.ReactionType;
                        reactionRow["DateCreated"] = DateTime.UtcNow;

                        npcReactionTable.Rows.Add(reactionRow);
                    }
                    #endregion

                    #region Encounter Data
                    foreach (var encounter in npc.Encounters)
                    {
                        #region Npc Stats
                        foreach (var stat in encounter.Stats)
                        {
                            #region Stats
                            var statRow = npcStatTable.NewRow();
                            statRow["Build"] = wowUpdate.Build;
                            statRow["InstanceDifficultyID"] = encounter.InstanceDifficulty;
                            statRow["NpcID"] = npc.Id;
                            statRow["Repairs"] = npc.Repairs;
                            statRow["FactionID"] = npc.FactionID;
                            statRow["Health"] = stat.MaxHealth;
                            statRow["PowerTypeID"] = stat.PowerType;
                            statRow["Power"] = stat.PowerValue;
                            statRow["Level"] = stat.Level;
                            statRow["PvPFlagged"] = npc.IsPvP;
                            statRow["DateCreated"] = DateTime.UtcNow;

                            npcStatTable.Rows.Add(statRow);
                            #endregion

                            #region Npc Reputation
                            foreach (var reputation in stat.Reputations)
                            {
                                var repRow = npcReputationTable.NewRow();
                                repRow["Build"] = wowUpdate.Build;
                                repRow["NpcID"] = npc.Id;
                                repRow["FactionID"] = reputation.FactionID;
                                repRow["Standing"] = reputation.PlayerStanding;
                                repRow["Amount"] = reputation.Amount;
                                repRow["DateCreated"] = DateTime.UtcNow;

                                npcReputationTable.Rows.Add(repRow);
                            }
                            #endregion

                            #region Locations
                            foreach (var location in stat.Locations)
                            {
                                foreach (var coord in location.Coordinates)
                                {
                                    var locRow = npcLocationTable.NewRow();
                                    locRow["Build"] = wowUpdate.Build;
                                    locRow["EntityTypeID"] = int.Parse(ConfigurationManager.AppSettings["NpcEntityID"]);
                                    locRow["EntityID"] = npc.Id;
                                    locRow["InstanceDifficultyID"] = encounter.InstanceDifficulty;
                                    locRow["ZoneID"] = location.LocationID;
                                    locRow["MapLevel"] = coord.MapLevel;
                                    locRow["X"] = coord.X;
                                    locRow["Y"] = coord.Y;
                                    locRow["DateCreated"] = DateTime.UtcNow;

                                    npcLocationTable.Rows.Add(locRow);
                                }
                            }
                            #endregion
                        }
                        #endregion

                        #region Npc Spells
                        foreach (var spellId in encounter.SpellIds)
                        {
                            var spellRow = npcSpellTable.NewRow();
                            spellRow["Build"] = wowUpdate.Build;
                            spellRow["InstanceDifficultyID"] = encounter.InstanceDifficulty;
                            spellRow["NpcID"] = npc.Id;
                            spellRow["SpellID"] = spellId;
                            spellRow["DateCreated"] = DateTime.UtcNow;

                            npcSpellTable.Rows.Add(spellRow);
                        }
                        #endregion

                        #region Npc Drops
                        foreach (var drop in encounter.Drops)
                        {
                            var dropRow = npcDropTable.NewRow();
                            dropRow["Build"] = wowUpdate.Build;
                            dropRow["InstanceDifficultyID"] = encounter.InstanceDifficulty;
                            dropRow["DropID"] = drop.DropId;
                            dropRow["DropTypeID"] = drop.DropType;
                            dropRow["SourceID"] = npc.Id;
                            dropRow["SourceTypeID"] = DropSourceType.NPC;
                            dropRow["Quantity"] = drop.Quantity;
                            dropRow["DateCreated"] = DateTime.UtcNow;

                            npcDropTable.Rows.Add(dropRow);
                        }
                        #endregion

                        #region Npc Loot Count
                        if (npc.LootCount != default(int))
                        {
                            var countRow = npcDropCountTable.NewRow();
                            countRow["Build"] = wowUpdate.Build;
                            countRow["InstanceDifficultyID"] = encounter.InstanceDifficulty;
                            countRow["SourceID"] = npc.Id;
                            countRow["SourceTypeID"] = DropSourceType.NPC;
                            countRow["TimesLooted"] = npc.LootCount;
                            countRow["DateCreated"] = DateTime.UtcNow;

                            // I only want one of these but have to put this here because of the instance level ID
                            if (npc.LootCount > 0)
                            {
                                npcDropCountTable.Rows.Add(countRow);
                            }
                        }
                        #endregion
                    }
                    #endregion

                    #region Teaches
                    foreach (var tradeSkill in npc.TradeSkillRequirements)
                    {
                        var tradeRow = tradeSkillRequirements.NewRow();
                        tradeRow["Build"] = wowUpdate.Build;
                        tradeRow["ClassID"] = tradeSkill.ClassID;
                        tradeRow["ProfessionID"] = tradeSkill.ProfessionID;
                        tradeRow["SkillID"] = tradeSkill.SkillID;
                        tradeRow["RequiredLevel"] = tradeSkill.RequiredLevel;
                        tradeRow["RequiredSkill"] = tradeSkill.RequiredSkill;
                        tradeRow["DateCreated"] = DateTime.UtcNow;

                        tradeSkillRequirements.Rows.Add(tradeRow);
                    }
                    #endregion
                }

                #region Bulk Copy
                bulkCopy.DestinationTableName = "VendorItems";
                bulkCopy.WriteToServer(vendorItemTable);

                bulkCopy.DestinationTableName = "NpcStats";
                bulkCopy.WriteToServer(npcStatTable);

                bulkCopy.DestinationTableName = "NpcSpell";
                bulkCopy.WriteToServer(npcSpellTable);

                bulkCopy.DestinationTableName = "[Drop]";
                bulkCopy.WriteToServer(npcDropTable);

                bulkCopy.DestinationTableName = "DropCount";
                bulkCopy.WriteToServer(npcDropCountTable);

                bulkCopy.DestinationTableName = "NpcReputation";
                bulkCopy.WriteToServer(npcReputationTable);

                bulkCopy.DestinationTableName = "Location";
                bulkCopy.WriteToServer(npcLocationTable);

                bulkCopy.DestinationTableName = "NpcReaction";
                bulkCopy.WriteToServer(npcReactionTable);

                bulkCopy.DestinationTableName = "TradeSkillRequirement";
                bulkCopy.WriteToServer(tradeSkillRequirements);
                #endregion
            }
        }

        private void PopulateExtendedCostIDs(ref WowUpdate update)
        {
            var currencyIds = new Dictionary<string, int>();
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["WowService"].ConnectionString))
            {
                var cmd = new SqlCommand("SELECT Id FROM CurrencyTypes WHERE Icon = @Icon", conn);
                cmd.Parameters.Add("Icon", SqlDbType.VarChar);

                conn.Open();
                foreach (var npc in update.Npcs)
                {
                    foreach (var vendorItem in npc.VendorItems)
                    {
                        if (vendorItem.ExtendedCost == null)
                        {
                            vendorItem.ExtendedCostID = 0;
                        }
                        else
                        {
                            // CurrencyIds first
                            foreach (var currency in vendorItem.ExtendedCost.Currencys)
                            {
                                if (!currencyIds.ContainsKey(currency.CurrenctName))
                                {
                                    cmd.Parameters["Icon"].Value = currency.CurrenctName;
                                    var currencyValue = cmd.ExecuteScalar();
                                    if (currencyValue != null)
                                    {
                                        currencyIds.Add(currency.CurrenctName, (int)currencyValue);
                                    }
                                    else currencyIds.Add(currency.CurrenctName, 0);
                                }

                                currency.CurrencyID = currencyIds[currency.CurrenctName];
                            }

                            // ExtendedCostId
                            var itemCmd = vendorItem.ExtendedCost.GetMatchingCommand(conn);
                            var extendedValue = itemCmd.ExecuteScalar();
                            if (extendedValue != null)
                            {
                                vendorItem.ExtendedCostID = (int)extendedValue;
                            }
                            else vendorItem.ExtendedCostID = 0;
                        }
                    }
                }
            }
        }
        private void PopulateFactionIDs(ref WowUpdate update)
        {
            var factionIds = new Dictionary<string, int>();
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["WowService"].ConnectionString))
            {
                var cmd = new SqlCommand("SELECT id FROM Faction WHERE Name = @FactionName", conn);
                cmd.Parameters.Add("FactionName", SqlDbType.VarChar);

                conn.Open();
                foreach (var npc in update.Npcs)
                {
                    if (!factionIds.ContainsKey(npc.Faction))
                    {
                        cmd.Parameters["FactionName"].Value = npc.Faction;
                        var factionValue = cmd.ExecuteScalar();
                        if (factionValue != null)
                        {
                            factionIds.Add(npc.Faction, (int)factionValue);
                        }
                        else factionIds.Add(npc.Faction, 0);
                    }
                    npc.FactionID = factionIds[npc.Faction];

                    foreach (var encounter in npc.Encounters)
                    {
                        foreach (var stat in encounter.Stats)
                        {
                            foreach (var reputation in stat.Reputations)
                            {
                                if (!factionIds.ContainsKey(reputation.FactionName))
                                {
                                    cmd.Parameters["FactionName"].Value = reputation.FactionName;
                                    var factionValue = cmd.ExecuteScalar();
                                    if (factionValue != null)
                                    {
                                        factionIds.Add(reputation.FactionName, (int)factionValue);
                                    }
                                    else factionIds.Add(reputation.FactionName, 0);
                                }
                                reputation.FactionID = factionIds[reputation.FactionName];
                            }
                        }
                    }
                }
            }
        }
        private void PopulateClassIDs(ref WowUpdate update)
        {
            var classIds = new Dictionary<string, int>();
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["WowService"].ConnectionString))
            {
                var cmd = new SqlCommand("SELECT id FROM ChrClasses WHERE InternalName = @ClassName", conn);
                cmd.Parameters.Add("ClassName", SqlDbType.VarChar);

                conn.Open();
                foreach (var npc in update.Npcs)
                {
                    foreach (var tradeSkill in npc.TradeSkillRequirements)
                    {
                        if (!classIds.ContainsKey(tradeSkill.ClassName))
                        {
                            cmd.Parameters["ClassName"].Value = tradeSkill.ClassName;
                            var classValue = cmd.ExecuteScalar();
                            if (classValue != null)
                            {
                                classIds.Add(tradeSkill.ClassName, (int)classValue);
                            }
                            else classIds.Add(tradeSkill.ClassName, 0);
                        }
                        tradeSkill.ClassID = classIds[tradeSkill.ClassName];
                    }
                }
            }
        }
        private void PopulateProfessionIDs(ref WowUpdate update)
        {
            var professionIds = new Dictionary<string, int>();
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["Azeroth"].ConnectionString))
            {
                var cmd = new SqlCommand("SELECT ExternalId FROM WowSkill WHERE Name = @Name", conn);
                cmd.Parameters.Add("Name", SqlDbType.VarChar);

                conn.Open();
                foreach (var npc in update.Npcs)
                {
                    foreach (var tradeSkill in npc.TradeSkillRequirements)
                    {
                        if (!professionIds.ContainsKey(tradeSkill.ProfessionName))
                        {
                            cmd.Parameters["Name"].Value = tradeSkill.ProfessionName;
                            var professionValue = cmd.ExecuteScalar();
                            if (professionValue != null)
                            {
                                professionIds.Add(tradeSkill.ProfessionName, (int)professionValue);
                            }
                            else professionIds.Add(tradeSkill.ProfessionName, 0);
                        }
                        tradeSkill.ProfessionID = professionIds[tradeSkill.ProfessionName];
                    }
                }
            }
        }
    }
}