﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Threading;

namespace Curse.ClientService
{
    public abstract class BaseLookupCache
    {
        private const int RELOAD_LOOKUP_INTERVAL = 5000;
        protected string _databaseConnectionString;
        protected Dictionary<CustomKey, Int32> _lookupCache = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());
        protected bool _lookupsLoaded = false;
        private string[] _lookupTables;
        private Thread _reloadLookupThread = null;

        protected BaseLookupCache(string connectionString, string[] lookupTables)
        {
            _databaseConnectionString = connectionString;
            _lookupTables = lookupTables;
            _lookupsLoaded = ReloadLookupCache();
            _reloadLookupThread = new Thread(ReloadLookupCacheThread) { IsBackground = true };
            _reloadLookupThread.Start();

        }
        private void ReloadLookupCacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(RELOAD_LOOKUP_INTERVAL);
                try
                {
                    ReloadLookupCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _reloadLookupThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               this.GetType().Name + " Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info,
                               null,
                               this.GetType().Name +  " Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private bool ReloadLookupCache()
        {
            try
            {
                using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
                {
                    conn.Open();
                    SqlCommand command = conn.CreateCommand();
                    foreach (string tableName in _lookupTables)
                    {
                        command.CommandText = "select * from " + tableName;
                        using (SqlDataReader reader = command.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                _lookupCache[GetLookupKey(tableName, (string)reader["LookupLabel"])] = (int)reader["LookupID"];
                            }
                        }
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.Log("CClientUsageCache - Failed to load lookup cache. Message: {0}", ELogLevel.Error, ex.Message);
                return false;
            }
        }

        protected CustomKey GetLookupKey(string tableName, string nameValue)
        {
            return new CustomKey(tableName.ToLowerInvariant(), nameValue.ToLowerInvariant());
        }

        protected int GetLookupID(SqlConnection conn, string tableName, string labelValue)
        {
            if (labelValue == null || labelValue.Length == 0)
            {
                labelValue = "Unknown";
            }
            labelValue = labelValue.Trim();
            CustomKey lookupKey = GetLookupKey(tableName, labelValue);
            if (_lookupCache.ContainsKey(lookupKey))
            {
                return _lookupCache[lookupKey];
            }
            SqlCommand command = conn.CreateCommand();
            command.CommandText = "insert into " + tableName + "(LookupLabel) output inserted.LookupID values(@LookupLabel)";
            SqlParameter param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
            param.Value = labelValue;
            int id = (int)command.ExecuteScalar();
            _lookupCache[lookupKey] = id;
            return id;
        }
    }
}
