﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;

using Curse.Extensions;
using Curse.ClientService.com.curse.auth;
using System.Configuration;
using System.ServiceModel;
using Curse.ClientService.Models;
using System.Web.Caching;
using System.ServiceModel.Channels;
using System.ServiceModel.Web;
using System.Threading;


namespace Curse.ClientService
{
    public static class CClientAuthentication
    {
        private static readonly int _siteID;
        private static readonly bool _freePremium = false;

        private static readonly object _loginsLock;
        private static Dictionary<string, DateTime> _logins;

        private static readonly object _sessionsLock;
        private static readonly Dictionary<string, CSession> _sessions;
        
        private static readonly StringCipher _outgoingCipher;
        private static readonly StringCipher _incomingCipher;

        private static readonly string _apiKey;
        
        private const int SubscriptionSeedPremium = 181756351;
        private const int SubscriptionSeedNonPremium = 76172335;
        private const string IncomingCipherKey = "31ED549B43318C9CFAE926ADE1720D2202C2DEFFAB06AE9D78FB6F16DF2D70A4";

        private static void CacheLogin(string username, string encryptedPassword)
        {
            string token = GetLoginToken(username, encryptedPassword);

            lock (_loginsLock)
            {
                if (!_logins.ContainsKey(token))
                {
                    _logins.Add(token, DateTime.UtcNow);
                }
            }
        }

        private static void CacheSession(string username, CSession session)
        {
            
            lock (_sessionsLock)
            {
                _sessions[username] = session;
            }
        }

        private static CSession GetSession(string username)
        {
            CSession session;
            
            lock (_sessionsLock)
            {
                _sessions.TryGetValue(username, out session);
            }

            return session;
        }

        public static string ApiKey
        {
            get
            {
                return _apiKey;
            }
        }

        private static bool LoginExists(string username, string plainTextPassword)
        {
            string token = GetLoginToken(username, plainTextPassword);

            lock (_loginsLock)
            {
                return _logins.ContainsKey(token);
            }
        }

        static CClientAuthentication()
        {
            _siteID = int.Parse(ConfigurationManager.AppSettings["AuthenticationId"]);
            _outgoingCipher = new StringCipher(ConfigurationManager.AppSettings["AuthenticationKey"]);
            _incomingCipher = new StringCipher(IncomingCipherKey);
            _freePremium = bool.Parse(ConfigurationManager.AppSettings["FreePremium"]);
            _apiKey = ConfigurationManager.AppSettings["ApiKey"];

            _loginsLock = new object();
            _sessionsLock = new object();

            _logins = new Dictionary<string, DateTime>();
            _sessions = new Dictionary<string, CSession>(StringComparer.InvariantCultureIgnoreCase);

            new Thread(CleanupSessionThread) { IsBackground = true }.Start();
        }


        private static void CleanupSessionThread()
        {
            while (true)
            {
                Thread.Sleep(5000);
                try
                {
                    CleanupSessions();
                }
                catch (Exception ex)
                {
                    Logger.Log("Unable to cleanup sessions! Message: {0}", ELogLevel.Error, ex.Message);
                }
            }
        }


        private static void CleanupSessions()
        {
            DateTime expiredSessionDate = DateTime.UtcNow.AddMinutes(-20);
            
            Dictionary<string, DateTime> loginsCopy;

            lock (_loginsLock)
            {
                loginsCopy = new Dictionary<string, DateTime>(_logins);
            }

            string[] expiredKeys = loginsCopy.Where(p => p.Value < expiredSessionDate).Select(p => p.Key).ToArray();

            if (expiredKeys.Length == 0)
            {
                return;
            }

            foreach (string expiredKey in expiredKeys)
            {
                loginsCopy.Remove(expiredKey);
            }

            lock (_loginsLock)
            {
                _logins = loginsCopy;
            }
        }

        public static List<CSession> GetAllSessions()
        {
            lock (_sessionsLock)
            {
                return _sessions.Values.ToList();
            }
        }

        public static CSession GetSessionByUsername(string username, bool refreshSubscriptionStatus)
        {
            CSession session = GetSession(username);
            
            if (session == null)
            {
                Logger.Log("Unable to find session for username: {0}", ELogLevel.Debug, username);
                return null;
            }

            if (refreshSubscriptionStatus && session.IsExpired)
            {                                
                try
                {
                    bool hasPremium = session.ActualPremiumStatus;

                    using (NetworkService authService = new NetworkService())
                    {
                        hasPremium = authService.v2GetSubscriptionStatus(_siteID, session.UserID, ESubscriptionType.Premium);
                        
                    }

                    session.ActualPremiumStatus = hasPremium;
                    session.EffectivePremiumStatus = GetEffectivePremiumStatus(hasPremium);
                    session.SubscriptionToken = GetSubscriptionToken(session.UserID, hasPremium);
                    session.DateRefreshed = DateTime.UtcNow;
                }
                catch (Exception ex)
                {
                    Logger.Log("Unable to refresh user's premium status. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }               
            }
            return session;
        }

        public static User CurrentUserProfile
        {
            get
            {
                return GetUserProfile(CurrentSession.UserID);
            }
        }

        public static CAuthToken CurrentAuthToken
        {
            get
            {
                WebOperationContext webContext = WebOperationContext.Current;
                OperationContext operationContext = OperationContext.Current;

                if (operationContext != null && webContext != null)
                {

                    MessageVersion messageVersion = operationContext.IncomingMessageVersion;

                    if (messageVersion == MessageVersion.None) // Soap 11 Version - Check standard Http headers
                    {
                        string username = webContext.IncomingRequest.Headers["x-curse-username"];
                        string password = webContext.IncomingRequest.Headers["x-curse-password"];
                        if (username != null && password != null)
                        {
                            return new CAuthToken(username, password);
                        }
                    }
                    else if (operationContext.IncomingMessageHeaders.FindHeader(CAuthToken.HeaderName, CAuthToken.HeaderNamespace) != -1) // Soap 12 Version - Check standard Soap headers
                    {
                        return operationContext.IncomingMessageHeaders.GetHeader<CAuthToken>(CAuthToken.HeaderName, CAuthToken.HeaderNamespace);
                    }
                }
                return new CAuthToken() { IsAnonymous = true };
            }
        }

        public static CSession RefreshedSession
        {
            get
            {
                return GetSessionByUsername(CurrentAuthToken.Username, true);
            }
        }


        public static CSession CurrentSession
        {
            get
            {
                return GetSessionByUsername(CurrentAuthToken.Username, false);        
            }                    
        }

        public static void RecordUserActivity(string username)
        {
            CSession session = GetSessionByUsername(username, false);

            if (session != null)
            {
                session.RecordActivity();
            }
        }

        private static string GetLoginToken(string username, string password)
        {
            return username.ToLowerInvariant() + "-" + password;
        }

        public static EAuthenticationStatus AuthenticateUser(string username, string encryptedPassword)
        {
            string planTextPassword = _incomingCipher.Decrypt(encryptedPassword);

            LoginResult result = null;

            if (LoginExists(username, planTextPassword))
            {
                RecordUserActivity(username);
                return EAuthenticationStatus.Success;
            }

            bool actualPremiumStatus = false;
            using (NetworkService authService = new NetworkService())
            {

                try
                {
                    result = authService.v2ValidateClientUser(_siteID, username, _outgoingCipher.Encrypt(planTextPassword));

                    if (result.Status == ELoginStatus.Success)
                    {
                        actualPremiumStatus = authService.v2GetSubscriptionStatus(_siteID, result.UserId, ESubscriptionType.Premium);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("Exception occurred while validating user! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    Logger.Log("Username: {0}, plainTextPassword: {1}", ELogLevel.Error, username, planTextPassword);
                    return EAuthenticationStatus.UnknownError;
                }                
            }
            
            if (result.Status == ELoginStatus.Success)
            {                                                
                bool effectivePremiumStatus = GetEffectivePremiumStatus(actualPremiumStatus);

                CSession session = new CSession()
                {
                    UserID = result.UserId,
                    SessionID = result.SessionId,
                    ActualPremiumStatus = actualPremiumStatus,
                    EffectivePremiumStatus = effectivePremiumStatus,
                    SubscriptionToken = GetSubscriptionToken(result.UserId, effectivePremiumStatus),
                    DateLastActive = DateTime.UtcNow
                };

                CacheLogin(username, planTextPassword);
                CacheSession(username, session);

                return EAuthenticationStatus.Success;
            }
            else
            {
                Logger.Log("Unable to authenticate user with username: {0} and plain text password: {1}", ELogLevel.Debug, username, planTextPassword);
                return ToAuthenticationStatus(result.Status);
            }           
        }

        private static EAuthenticationStatus ToAuthenticationStatus(ELoginStatus status)
        {
            switch (status)
            {
                case ELoginStatus.InvalidPassword:
                    return EAuthenticationStatus.InvalidPassword;
                case ELoginStatus.InvalidSession:
                    return EAuthenticationStatus.InvalidSession;
                case ELoginStatus.Success:
                    return EAuthenticationStatus.Success;
                case ELoginStatus.UnauthorizedLogin:
                    return EAuthenticationStatus.UnauthorizedLogin;
                case ELoginStatus.UnknownEmail:
                    return EAuthenticationStatus.UnknownEmail;
                case ELoginStatus.UnknownUsername:
                    return EAuthenticationStatus.UnknownUsername;
                case ELoginStatus.Unsuccessful:
                default:
                    return EAuthenticationStatus.Unsuccessful;
            }
        }

        private static bool GetEffectivePremiumStatus(bool hasPremium)
        {
            if (_freePremium || hasPremium)
            {
                return true;
            }
            else
            {
                return false;

            }
        }       
       
        public static int GetSubscriptionToken(int userID, bool hasPremium)
        {
            return userID ^ (hasPremium ? SubscriptionSeedPremium : SubscriptionSeedNonPremium);
        }        
        
        private static string GetUserProfileCacheKey(int userID)
        {
            return "UserProfile-" + userID.ToString();
        }

        private static User GetUserProfile(int userID)
        {
            string cacheKey = GetUserProfileCacheKey(userID);

            User profile = HttpRuntime.Cache.Get(cacheKey) as User;

            if (profile == null)
            {
                using (NetworkService authService = new NetworkService())
                {
                    profile = authService.v2GetUserProfile(_siteID, userID);
                }

                HttpRuntime.Cache.Insert(cacheKey, profile, null, Cache.NoAbsoluteExpiration, TimeSpan.FromMinutes(10));
            }
            return profile;
        }               
        
        public static bool IsUsernameAvailable(string username)
        {
            using (NetworkService authService = new NetworkService())
            {
                return authService.v2IsUsernameAvailable(_siteID, username);
            }
        }

        public static CRegisterUserResult RegisterUser(string username, string password, string firstname, string lastname, string birthdate, string gender, string country, string region, string city, string email, bool newsletter)
        {
            AddUserResult result = null;            
            using (NetworkService authService = new NetworkService())
            {
                try
                {
                    result = authService.v2AddUser(_siteID,
                        username,
                        _outgoingCipher.Encrypt(password),
                        firstname,
                        lastname,
                        birthdate,
                        gender,
                        country,
                        region,
                        city,
                        email,
                        newsletter,
                        null,
                        null,
                        null);
                }
                catch
                {
                    Logger.Log("CClientAuthentication - Failed to register user.", ELogLevel.Error);
                }
            }

            CRegisterUserResult registerUserResult = new CRegisterUserResult();            

            if (result == null)
            {
                Logger.Log("CClientAuthentication - v2AddUser returned empty result", ELogLevel.Error);
                registerUserResult.Status = ERegisterUserStatus.UnknownError;
            }
            else
            {
                registerUserResult.Status = (ERegisterUserStatus)result.Status;
                registerUserResult.UserID = result.UserID;
                registerUserResult.SessionID = result.SessionID;
            }

            return registerUserResult;
        }

        
        
        

    }
}
