﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Configuration;
using System.Threading;
using System.Data.SqlClient;
using System.Data;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO.Compression;
using System.Collections;
using System.Runtime.Serialization;

using Curse.AddOns;
using Curse.ClientService.Extensions;
using Curse.Extensions;
using ICSharpCode.SharpZipLib.BZip2;
using System.Web.Script.Serialization;
using System.Runtime.Serialization.Json;

namespace Curse.ClientService
{
    public class CAddOnCache
    {
        private Dictionary<int, CAddOn> _addOnCache = new Dictionary<int, CAddOn>();
        private Dictionary<string, CRepositoryMatch> _slugCache = new Dictionary<string, CRepositoryMatch>();
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;
        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        private Dictionary<EFeedTimespan, Int32> _feedLifespans = new Dictionary<EFeedTimespan, Int32>();
        private string _staticFileDestination = null;
        private bool _createFeedFiles = false;

        private static readonly CAddOnCache _instance = new CAddOnCache();

        public static CAddOnCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public CAddOnCache()
        {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());

#if DEBUG
            _createFeedFiles = true;
#endif
            // Populate feed lifespans
            _feedLifespans.Add(EFeedTimespan.Complete, (int)TimeSpan.FromHours(3).TotalSeconds);
            _feedLifespans.Add(EFeedTimespan.Weekly, (int)TimeSpan.FromHours(2).TotalSeconds);
            _feedLifespans.Add(EFeedTimespan.Daily, (int)TimeSpan.FromHours(1).TotalSeconds);
            _feedLifespans.Add(EFeedTimespan.Hourly, (int)TimeSpan.FromMinutes(10).TotalSeconds);

           
        }

        public bool IsCacheUpdating
        {
            get
            {
                return _isCacheUpdating;
            }
        }

        public bool IsCacheBuilt
        {
            get
            {
                return _isCacheBuilt;
            }
            set
            {
                _isCacheBuilt = value;
            }
        }

        public void Initialize()
        {            
            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Normal;
            _updateThread.Start();
        }

        public void ResetCache()
        {
            _lastQueryTime = new DateTime(1979, 5, 17);
        }

        public CRepositoryMatch GetRepositoryMatchFromSlug(string gameSlug, string addOnSlug)
        {
            string slug = GetSlugKey(gameSlug, addOnSlug);
            if (_slugCache.ContainsKey(slug))
            {
                return _slugCache[slug];
            }
            else
            {
                return null;
            }
        }

        public CAddOn GetCachedAddOn(int pId)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            if (_addOnCache.ContainsKey(pId))
            {
                return _addOnCache[pId];
            }
            else
            {
                return null;
            }
        }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
               
                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                finally
                {
                    _isCacheUpdating = true;
                    Thread.Sleep(_updateThreadInterval);
                }
                
            }
        }

        public void ResetSingleAddonCache(int id)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand command = conn.CreateCommand();
                command.CommandText = "update Project set DateSynced = GETUTCDATE() where Project.ID = @ProjectID";
                SqlParameter param = command.Parameters.Add("@ProjectID", SqlDbType.Int);
                param.Value = id;
                command.ExecuteNonQuery();
            }
        }

        private void UpdateCache()
        {
            DateTime startTime = DateTime.UtcNow;
            Dictionary<int, CAddOn> addOnCache = new Dictionary<int, CAddOn>(_addOnCache);
            List<CAddOn> completeAddOnList = new List<CAddOn>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Error, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString(), ex.Message);
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                DateTime changeDate = _lastQueryTime.AddMinutes(-1);

                // All user surrogates
                DataTable gameTable = new DataTable();
                gameTable.Columns.Add("ID", typeof(Int32));        

                // Add the user rows to the table
                foreach (var game in CGameCache.Instance.Games)
                {
                    gameTable.Rows.Add(game.ID);
                }
                

                // Get a cache of all authors
                Logger.Log("Building Author Cache...", ELogLevel.Info);
                Dictionary<int, List<CAddOnAuthor>> authorCache = CAuthorCache.Instance.GetAuthorCache(conn, changeDate);

                // Get a cache of all categories
                Logger.Log("Building Category Cache...", ELogLevel.Info);
                Dictionary<int, List<CAddOnCategory>> categoryCache = CCategoryCache.Instance.GetCategoryCache(conn, changeDate);

                // Get a cache of the files:
                Logger.Log("Building File Cache...", ELogLevel.Info);
                Dictionary<int, List<CAddOnFile>> fileCache = CFileCache.Instance.GetFileCache(conn, changeDate, gameTable);

                // Get a cache of the individual fingerprints:
                Logger.Log("Building Fingerprint Cache...", ELogLevel.Info);
                Dictionary<int, List<CIndividualFileFingerprint>> individualFingerprintCache = CFingerprintCache.Instance.GetIndividualFingerprintCache(conn, changeDate, gameTable);

                HashSet<int> validGameIDs = new HashSet<int>(CGameCache.Instance.Games.Select(p => p.ID));

                // Populate the cache with recently modified addons:
                Logger.Log("Beginning Addon DB Call...", ELogLevel.Info);

                using (SqlCommand command = new SqlCommand("curseService_GetAddOnListv3", conn))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = changeDate;
                    command.Parameters.Add("@GameIDs", SqlDbType.Structured).Value = gameTable;

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        Logger.Log("Completed Addon DB Call...", ELogLevel.Info);

                        int addonCount = 0;

                        Logger.Log("Reading addon database...", ELogLevel.Info);

                        while (reader.Read())
                        {
                            int addOnId = reader.GetInt32(reader.GetOrdinal("addon_id"));

                            Dictionary<int, CAddOnFile> existingFiles;
                            if (addOnCache.ContainsKey(addOnId))
                            {
                                existingFiles = addOnCache[addOnId].Files;
                            }
                            else
                            {
                                existingFiles = new Dictionary<int, CAddOnFile>();
                            }

                            CAddOn addon = new CAddOn();
                            addon.SetFromDataReader(reader, individualFingerprintCache, authorCache, categoryCache, fileCache, existingFiles);

                            addOnCache[addOnId] = addon;

                            if ((++addonCount % 1000) == 0)
                            {
                                Logger.Log(string.Format("Processed 1,000 Addons, current total: {0}", addOnCache.Count.ToString("###,##0")), ELogLevel.Debug);
                            }
                        }
                    }
                }

                Logger.Log("Rebuilding fingerprint index...", ELogLevel.Info);
                CFingerprintCache.Instance.Rebuild(addOnCache);

                // Construct the complete list:
                Logger.Log("Constructing complete list...", ELogLevel.Info);
                foreach (KeyValuePair<int, CAddOn> kvp in addOnCache.Where(p => p.Value.IsAvailable))
                {
                    completeAddOnList.Add(kvp.Value);
                }

                // Do this for complete addon count count
                Logger.Log("Total Addons: {0}", ELogLevel.Info, addOnCache.Count.ToString("###,##0"));
                
                foreach (var game in CGameCache.Instance.Games)
                {
                    var totalCount = addOnCache.Values.Count(p => p.GameId == game.ID);
                    var availableCount = addOnCache.Values.Count(p => p.GameId == game.ID && p.IsAvailable);
                    Logger.Log(string.Format("{0} - Total: {1}, Avaiable: {2}", game.Name, totalCount.ToString("###,##0"), availableCount.ToString("###,##0")), ELogLevel.Info);
                }

                Dictionary<string, CRepositoryMatch> slugCache = new Dictionary<string, CRepositoryMatch>();
                Dictionary<int, string> gameStages = new Dictionary<int, string>();
                foreach (CGame game in CGameCache.Instance.Games)
                {
                    gameStages.Add(game.ID, game.StageName.ToLower());
                }

                // Construct the slug cache
                foreach (KeyValuePair<int, CAddOn> kvp in addOnCache)
                {
                    CAddOn addon = kvp.Value;
                    string gameStage = gameStages[addon.GameId];
                    slugCache[GetSlugKey(gameStage, addon.Slug.ToLower())] = new CRepositoryMatch()
                    {
                        Id = addon.Id,
                        LatestFiles = addon.LatestFiles
                    };
                }

                lock (_slugCache)
                {
                    _slugCache = slugCache;
                }

                lock (_addOnCache)
                {
                    _addOnCache = addOnCache;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt)
            {
                Logger.Log("AddOn Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;

            Logger.Log("Addon cache built in {0} seconds", ELogLevel.Info, (DateTime.UtcNow - startTime).TotalSeconds);

            CreateFeedFiles(completeAddOnList);

            GC.Collect();

        }

        private Dictionary<EFeedTimespan, DateTime> _feedTimestamps = new Dictionary<EFeedTimespan, DateTime>();

        private bool IsFeedExpired(EFeedTimespan feedTimespan)
        {
            DateTime timestamp;
            Int32 lifespan = _feedLifespans[feedTimespan];

            if (_feedTimestamps.TryGetValue(feedTimespan, out timestamp))
            {
                int age = (int)DateTime.UtcNow.Subtract(timestamp).TotalSeconds;
                if (age > lifespan)
                {
                    Logger.Log("Feed has expired: {0}", ELogLevel.Info, feedTimespan);
                    return true;
                }
                else
                {
                    return false;
                }

            }
            else
            {
                Logger.Log("Feed has not yet been created: {0}", ELogLevel.Info, feedTimespan);
                return true;
            }

        }

        public string GetSlugKey(string gameSlug, string addonSlug)
        {
            return gameSlug + "_" + addonSlug;
        }

        private void CreateFeedFiles(List<CAddOn> completeAddOnList)
        {
            if (!_createFeedFiles)
            {
                return;
            }

            foreach (EFeedTimespan timespan in System.Enum.GetValues(typeof(EFeedTimespan)))
            {
                try
                {
                    CreateFeedFile(timespan, completeAddOnList);
                }
                catch (Exception ex)
                {
                    Logger.Log("Unable to create feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
            }
        }

        private void CreateFeedFile(EFeedTimespan feedTimespan, List<CAddOn> completeList)
        {
            if (!IsFeedExpired(feedTimespan))
            {
                return;
            }

            _feedTimestamps[feedTimespan] = DateTime.UtcNow;
            int hours = (int)feedTimespan;
            List<CAddOn> incrementalList = null;
            if (hours > 0)
            {
                TimeSpan t = new TimeSpan(hours, 0, 0);
                DateTime date = DateTime.UtcNow.Subtract(t);
                incrementalList = completeList.Where(p => p.DateModified >= date).ToList();
            }
            else
            {
                incrementalList = completeList;
            }
            SaveBinaryToDisk(feedTimespan, incrementalList);
            SaveJsonToDisk(feedTimespan, incrementalList);
            SaveXmlToDisk(feedTimespan, incrementalList);
        }

        private void SaveXmlToDisk(EFeedTimespan feedTimespan, List<CAddOn> incrementalList)
        {
            byte[] uncompressedBytes = null;

            using (MemoryStream uncompressedStream = new MemoryStream())
            {
                DataContractSerializer xs = new DataContractSerializer(incrementalList.GetType());
                xs.WriteObject(uncompressedStream, incrementalList);
                uncompressedStream.Position = 0;
                uncompressedBytes = uncompressedStream.ToArray();
            }

            byte[] compressedBytes = null;

            using (MemoryStream compressedStream = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(compressedStream))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = compressedStream.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, feedTimespan.ToString() + ".xml.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                if (File.Exists(currentfile))
                {
                    File.Replace(tempfile, currentfile, backupfile, true);
                }
                else
                {
                    File.Copy(tempfile, currentfile, true);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("CAddOnCache - SaveXmlToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            finally
            {
                File.Delete(tempfile);
                if (File.Exists(backupfile))
                {
                    File.Delete(backupfile);
                }
            }
        }

        private void SaveJsonToDisk(EFeedTimespan feedTimespan, List<CAddOn> incrementalList)
        {
            byte[] uncompressedBytes = null;

            using (MemoryStream uncompressedStream = new MemoryStream())
            {
                DataContractJsonSerializer xs = new DataContractJsonSerializer(incrementalList.GetType());
                xs.WriteObject(uncompressedStream, incrementalList);
                uncompressedStream.Position = 0;
                uncompressedBytes = uncompressedStream.ToArray();
            }
            byte[] compressedBytes = null;

            using (MemoryStream compressedStream = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(compressedStream))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = compressedStream.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, feedTimespan.ToString() + ".json.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                if (File.Exists(currentfile))
                {
                    File.Replace(tempfile, currentfile, backupfile, true);
                }
                else
                {
                    File.Copy(tempfile, currentfile, true);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("CAddOnCache - SaveXmlToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            finally
            {
                File.Delete(tempfile);
                if (File.Exists(backupfile))
                {
                    File.Delete(backupfile);
                }
            }
        }

        private void SaveBinaryToDisk(EFeedTimespan feedTimespan, List<CAddOn> incrementalList)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, incrementalList);

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, feedTimespan.ToString() + ".zip");
                using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
                {
                    binWriter.Write(compressed);
                }
                try
                {
                    if (File.Exists(currentfile))
                    {
                        File.Replace(tempfile, currentfile, backupfile, true);
                    }
                    else
                    {
                        File.Copy(tempfile, currentfile, true);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("CAddOnCache - SaveBinaryToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally
                {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile))
                    {
                        File.Delete(backupfile);
                    }
                }
            }
        }

     
    }
}
