﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Configuration;
using System.Threading;
using System.Data.SqlClient;
using System.Data;

using Curse.Extensions;
using Curse.ClientService.Extensions;
using Curse.ClientService.Models;
using Curse.ClientService.GeoCoding;
using System.Net;
using System.IO;

namespace Curse.ClientService
{
    public class CCampaignCache
    {
        private static readonly CCampaignCache _instance = new CCampaignCache();

        private readonly string _databaseConnectionString;
        private readonly int _updateThreadInterval = 30 * 1000; // 30 Seconds
        private readonly string _themePackDllPath;
        private readonly string _themePackDestinationPath;
        private readonly string _staticFileDestination = null;
        private readonly bool _createFeedFiles = false;

        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private Thread _updateThread = null;
        private string _themePackDllHash = null;
        

        private List<CCampaign> _campaigns;
        private Dictionary<int, CCampaign> _campaignByRegion;        
        private Dictionary<string, List<int>> _regionsByCountry;

        public static CCampaignCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public void Initialize() { }

        public string ThemePackHash
        {
            get
            {
                return _themePackDllHash;
            }
        }

        public CCampaign GetActiveCampaign(IPAddress ipAddress)
        {
            if (_regionsByCountry == null || _regionsByCountry.Count == 0)
            {
                return null;
            }

            Int64 ipRangeValue = ipAddress.ToInt64();

            // Get the country for this IP
            CountryIPRange range = CountryIPRange.GetRange(ipRangeValue);

            if (range == null)
            {
                return null;
            }

            // Get the regions for the country
            List<int> regions;
            if (!_regionsByCountry.TryGetValue(range.CountryCode, out regions))
            {
                return null;
            }

            // Get the campaign for this region
            CCampaign campaign = null;

            foreach (int region in regions)
            {
                if (_campaignByRegion.TryGetValue(region, out campaign))
                {
                    break;
                }
            }

            return campaign;
        }

        public void PopulateGeoData()
        {
            List<CountryIPRange> countryIPRanges = new List<CountryIPRange>();
            Dictionary<string, List<int>> regionsByCountry = new Dictionary<string, List<int>>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception exc)
                {
                    Logger.Log(ELogLevel.Error, "localhost", "Unable to establish connection to database:" + exc.Message);
                    return;
                }

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "select * from CountryIpRange order by CountryCode";

                using (SqlDataReader reader = command.ExecuteReader())
                {

                    while (reader.Read())
                    {
                        countryIPRanges.Add(new CountryIPRange(reader));
                    }
                }

                CountryIPRange.BuildRanges(countryIPRanges);
                
                command.CommandText = "select * from CountryRegion order by CountryCode";

                using (SqlDataReader reader = command.ExecuteReader())
                {

                    while (reader.Read())
                    {
                        string countryCode = (string)reader["CountryCode"];
                        int regionID = (int)reader["RegionID"];

                        if (!regionsByCountry.ContainsKey(countryCode))
                        {
                            regionsByCountry[countryCode] = new List<int>();
                        }

                        regionsByCountry[countryCode].Add(regionID);
                    }
                }
            }            

            _regionsByCountry = regionsByCountry;
            
        }

        private CCampaignCache()
        {
            _updateThreadInterval = 1000 * 5;
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _themePackDllPath = ConfigurationManager.AppSettings["ThemePackDllSourcePath"];
            _themePackDestinationPath = Path.Combine(_staticFileDestination, "ThemePack.zip");
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());

#if DEBUG
            _createFeedFiles = true;
#endif
            _campaigns = new List<CCampaign>();
            _campaignByRegion = new Dictionary<int, CCampaign>();
            _regionsByCountry = new Dictionary<string, List<int>>();

            try
            {
                PopulateGeoData();
            }
            catch (Exception ex)
            {
                Logger.Log("Error populating geo data: " + ex.Message, ELogLevel.Error);
            }

            try
            {
                UpdateCache();
            }
            catch (Exception ex)
            {
                Logger.Log("Error updating cache: " + ex.Message, ELogLevel.Error);
            }

            try
            {
                var latestThemePackDll = GetLatestThemePackDll();
                if (latestThemePackDll != null)
                {
                    ProcessThemePackDll(latestThemePackDll);
                }
            }
            catch (Exception ex)
            {
               Logger.Log("Unable to create theme pack feed file: " + ex.Message, ELogLevel.Error);
            }

            try
            {
                FileSystemWatcher watcher = new FileSystemWatcher(_themePackDllPath, "*.dll");
                watcher.Created += new FileSystemEventHandler(Watcher_Created);
                watcher.EnableRaisingEvents = true;
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to establish file watcher for theme pack: " + ex.Message, ELogLevel.Error);
            }

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        private string GetLatestThemePackDll()
        {
            DirectoryInfo info = new DirectoryInfo(_themePackDllPath);
            FileInfo[] files = info.GetFiles("*.dll");

            if(files.Length == 0)
            {
                Logger.Log("CCampaignCache - No theme pack DLLs found!", ELogLevel.Error);
                return null;
            }

            return files.OrderByDescending(p => p.CreationTimeUtc).FirstOrDefault().FullName;                        
        }

        private int _retryDelaySeconds = 10;
        private int _retryMaximumSeconds = 90;

        void Watcher_Created(object sender, FileSystemEventArgs e)
        {

            DateTime fileReceived = DateTime.UtcNow;

            while (true)
            {
                if (this.DllCreationCompleted(e.FullPath))
                {
                    this.ProcessThemePackDll(e.FullPath);
                    break;
                }

                // Calculate the elapsed time and stop if the maximum retry
                // period has been reached.
                TimeSpan timeElapsed = DateTime.UtcNow - fileReceived;

                if (timeElapsed.TotalSeconds > _retryMaximumSeconds)
                {
                    Logger.Log("CCampaignCache - ProcessThemePackDll Exception! Unable to ", ELogLevel.Error);                    
                    break;
                }

                Thread.Sleep(_retryDelaySeconds);
            }

        }

        private bool DllCreationCompleted(string filePath)
        {
            // If the file can be opened for exclusive access it means that the file
            // is no longer locked by another process.
            try
            {
                using (FileStream inputStream = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.None))
                {
                    return true;
                }
            }
            catch (IOException)
            {
                return false;
            }
        }

        private void ProcessThemePackDll(string filePath)
        {            
            byte[] dllBytes = File.ReadAllBytes(filePath);
            string dllHash = dllBytes.ComputeHash();

            if (_themePackDllHash == dllHash)
            {
                return;
            }

            if(!_createFeedFiles)
            {
                _themePackDllHash = dllHash;
                return;
            }

            byte[] compressed = Utility.GetCompressedBytes(dllBytes);

            string tempfile = Path.Combine(_staticFileDestination, Utility.UniqueNumber + ".zip");
            string backupfile = Path.Combine(_staticFileDestination, Utility.UniqueNumber + ".zip");
            string currentfile = _themePackDestinationPath;

            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressed);
            }

            try
            {
                if (File.Exists(currentfile))
                {
                    File.Replace(tempfile, currentfile, backupfile, true);
                }
                else
                {
                    File.Copy(tempfile, currentfile);
                }

                _themePackDllHash = dllHash;
            }
            catch (Exception ex)
            {
                Logger.Log("CCampaignCache - ProcessThemePackDll Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            finally
            {
                File.Delete(tempfile);
                if (File.Exists(backupfile))
                {
                    File.Delete(backupfile);
                }
            }


        }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info,
                               null,
                               "CCampainCache Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void UpdateCache()
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                // Get a copy of the cached campaign list
                List<CCampaign> campaigns = new List<CCampaign>(_campaigns);
                
                SqlCommand command = new SqlCommand("select * from Campaign with(nolock) where DateModified > @DateModified", conn);
                SqlParameter param = command.Parameters.Add("@DateModified", SqlDbType.DateTime);
                param.Value = _lastQueryTime;

                using (SqlDataReader reader = command.ExecuteReader())
                {
                    _lastQueryTime = DateTime.UtcNow;

                    while (reader.Read())
                    {
                        CCampaign campaign = new CCampaign();
                        campaign.SetFromDataReader(reader, conn);
                        campaigns.RemoveAll(p => p.ID == campaign.ID);
                        campaigns.Add(campaign);
                    }
                }

                // Remove all expired campaigns
                campaigns.RemoveAll(p => p.EndDate <= DateTime.UtcNow || p.StartDate > DateTime.UtcNow);

                Dictionary<int, CCampaign> campaignsByRegion = new Dictionary<int, CCampaign>();

                foreach (CCampaign campaign in campaigns.OrderBy(p => p.StartDate))
                {
                    foreach (int regionID in campaign.Regions)
                    {
                        campaignsByRegion[regionID] = campaign;
                    }
                }

                _campaigns = campaigns;
                _campaignByRegion = campaignsByRegion;
            }
        }
    }
}
