﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using Curse.AddOns;
using System.Data;
using Curse.ClientService.Extensions;

namespace Curse.ClientService
{
    public class CFileCache
    {
        private Dictionary<int, string> _changeLogs = new Dictionary<int, string>();
        public Dictionary<int, string> ChangeLogs
        {
            get { return _changeLogs; }
            set { _changeLogs = value; }
        }

        private static readonly CFileCache _instance = new CFileCache();

        private Dictionary<int, int> _addonIDByFileID = new Dictionary<int, int>();

        public static CFileCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public int GetAddonIDByFileID(int fileID)
        {
            int projectID = 0;
            if (_addonIDByFileID.TryGetValue(fileID, out projectID))
            {
                return projectID;
            }
            else
            {
                return 0;
            }
        }

        public Dictionary<int, List<CAddOnFile>> GetFileCache(SqlConnection conn, DateTime changeDate, DataTable gameTable)
        {
            Dictionary<int, List<CAddOnFileDependency>> fileDependencyCache = new Dictionary<int, List<CAddOnFileDependency>>();
            Dictionary<int, List<string>> fileGameVersionCache = new Dictionary<int, List<string>>();
            Dictionary<int, List<CAddOnFile>> fileCache = new Dictionary<int, List<CAddOnFile>>();

            //Get the file dependancies
            Logger.Log("File Cache: Getting File Dependencies...", ELogLevel.Debug);
            using (SqlCommand dependencyCommand = new SqlCommand("curseService_GetAllFileDependenciesv2", conn))
            {
                dependencyCommand.CommandType = CommandType.StoredProcedure;
                dependencyCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                dependencyCommand.Parameters["@LastUpdated"].Value = changeDate;
                dependencyCommand.Parameters.Add("@GameIDs", SqlDbType.Structured).Value = gameTable;
                dependencyCommand.CommandTimeout = 300;

                Logger.Log("File Cache: Beginning Dependency Call...", ELogLevel.Debug);
                using (SqlDataReader dependencyReader = dependencyCommand.ExecuteReader())
                {
                    Logger.Log("File Cache: Dependency Call Complete...", ELogLevel.Debug);
                    while (dependencyReader.Read())
                    {
                        int fileId = dependencyReader.GetInt32(dependencyReader.GetOrdinal("addon_file_id"));
                        int dependencyId = dependencyReader.GetInt32(dependencyReader.GetOrdinal("addon_dependency_id"));
                        byte dependencyType = dependencyReader.GetByte(dependencyReader.GetOrdinal("addon_dependency_type"));

                        if (!fileDependencyCache.ContainsKey(fileId))
                        {
                            fileDependencyCache.Add(fileId, new List<CAddOnFileDependency>());
                        }//if there isnt an entry for this file ID already

                        fileDependencyCache[fileId].Add(new CAddOnFileDependency(dependencyId, dependencyType));
                    }
                }
            }

            Logger.Log("File Cache: Getting GameVersions...", ELogLevel.Debug);

            using (SqlCommand command = conn.CreateCommand())
            {
                command.CommandText = "curseService_GetProjectFileGameVersions";
                command.CommandType = CommandType.StoredProcedure;
                command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = changeDate;

                Logger.Log("File Cache: Executing game version query...", ELogLevel.Debug);
                using (SqlDataReader gameVersionReader = command.ExecuteReader())
                {
                    Logger.Log("File Cache: Query completed. Building dictionary...", ELogLevel.Debug);
                    while (gameVersionReader.Read())
                    {
                        int fileId = gameVersionReader.GetInt32(gameVersionReader.GetOrdinal("ProjectFileID"));
                        string versionName = gameVersionReader.GetString(gameVersionReader.GetOrdinal("Name"));

                        if (!fileGameVersionCache.ContainsKey(fileId))
                        {
                            fileGameVersionCache.Add(fileId, new List<string>());
                        }

                        fileGameVersionCache[fileId].Add(versionName);
                    }
                }
            }
            

            // Get a cache of all files
            Logger.Log("File Cache: Getting File Cache...", ELogLevel.Debug);
            using (SqlCommand fileCommand = new SqlCommand("curseService_GetAllAddOnFiles", conn))
            {
                fileCommand.CommandType = CommandType.StoredProcedure;
                fileCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fileCommand.Parameters["@LastUpdated"].Value = changeDate;
                fileCommand.CommandTimeout = 300;

                Logger.Log("File Cache: Beginning File Call...", ELogLevel.Debug);
                using (SqlDataReader fileReader = fileCommand.ExecuteReader())
                {
                    Logger.Log("File Cache: File Call Complete...", ELogLevel.Debug);
                    while (fileReader.Read())
                    {
                        int projectID = fileReader.GetInt32(fileReader.GetOrdinal("addon_id"));
                        if (!fileCache.ContainsKey(projectID))
                        {
                            fileCache.Add(projectID, new List<CAddOnFile>());
                        }

                        CAddOnFile file = new CAddOnFile();
                        file.SetFromDataReader(fileReader, fileDependencyCache, fileGameVersionCache);
                        fileCache[projectID].Add(file);
                        _addonIDByFileID[file.Id] = projectID;
                    }
                }
            }


            // Get the addon fingerprints
            Logger.Log("File Cache: Getting File Fingerprints...", ELogLevel.Debug);
            using (SqlCommand fingerprintCommand = new SqlCommand("curseService_GetAllAddOnFileFingerprints", conn))
            {
                fingerprintCommand.CommandType = CommandType.StoredProcedure;
                fingerprintCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fingerprintCommand.Parameters["@LastUpdated"].Value = changeDate;
                fingerprintCommand.CommandTimeout = 600;

                Logger.Log("File Cache: Beginning Fingerprint Call...", ELogLevel.Debug);
                using (SqlDataReader fingerprintReader = fingerprintCommand.ExecuteReader())
                {
                    Logger.Log("File Cache: Fingerprint Call Complete...", ELogLevel.Debug);
                    while (fingerprintReader.Read())
                    {
                        int projectID = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("addon_id"));

                        if (!fileCache.ContainsKey(projectID))
                        {
                            continue;
                        }

                        int fileId = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("file_id"));
                        long fingerprint = fingerprintReader.GetInt64(fingerprintReader.GetOrdinal("fingerprint"));

                        string foldername = null;
                        if (fingerprintReader["folder"] != System.DBNull.Value)
                        {
                            foldername = fingerprintReader.GetString(fingerprintReader.GetOrdinal("folder"));
                        }

                        List<CAddOnFile> files = fileCache[projectID];
                        CAddOnFile file = files.Where(p => p.Id == fileId).FirstOrDefault();

                        if (file == null) { continue; }

                        file.Fingerprints.Add(fingerprint);
                        if (foldername != null)
                        {
                            CAddOnModule module = new CAddOnModule()
                            {
                                Fingerprint = fingerprint,
                                Foldername = foldername
                            };
                            file.Modules.Add(module);
                        }
                    }
                }
            }

            return fileCache;
        }

    }
}
