﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Threading;
using System.Reflection;

namespace Curse.ClientService
{
    public class CHardwareReportCache : BaseLookupCache
    {
        private static readonly CHardwareReportCache _instance = new CHardwareReportCache(); 

        private const int WRITE_INTERVAL = 5000;        
        private readonly Thread _writeToDatabaseThread = null;        
        private List<CHardwareProfile> _hardwareProfiles;                        

        public static CHardwareReportCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CHardwareReportCache()
            : base(ConfigurationManager.ConnectionStrings["ClientHardwareReporting"].ConnectionString, new[]
        {
            "CpuManufacturer",
            "CpuModel",
            "DirectXVersion",
            "GpuDriverName",
            "GpuDriverVersion",
            "GpuManufacturer",
            "GpuModel",
            "OperatingSystemVersion",
            "Language",
            "MonitorModel",
            "RamManufacturer",
            "RamType",
            "ServicePackVersion",
            "SoundModel"
        })
        {
            
            
            _hardwareProfiles = new List<CHardwareProfile>();

            _writeToDatabaseThread = new Thread(WriteToDatabaseThread) { IsBackground = true };
            _writeToDatabaseThread.Start();
        }              

        public void Initialize() {}
        
        public bool SaveReport(int pUserID, CHardwareProfile pHardwareProfile)
        {
            if (!_lookupsLoaded)
            {
                return false;
            }

            pHardwareProfile.UserID = pUserID;
            lock (_hardwareProfiles)
            {
                _hardwareProfiles.Add(pHardwareProfile);
            }
            return true;
        }
               
        private void WriteToDatabase()
        {
            List<CHardwareProfile> profiles = null;

            lock (_hardwareProfiles)
            {
                profiles = new List<CHardwareProfile>(_hardwareProfiles);
                _hardwareProfiles.Clear();
            }
            
            if (profiles.Count == 0)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                foreach (CHardwareProfile profile in profiles)
                {
                    try
                    {
                        SaveProfileToDatabase(conn, profile);
                    }
                    catch (Exception ex)
                    {
                        PropertyInfo[] fields = profile.GetType().GetProperties(BindingFlags.Instance | BindingFlags.Public);
                        string objectDump = string.Empty;
                        foreach (PropertyInfo info in fields)
                        {
                            object x = info.GetValue(profile, null);
                            objectDump += info.Name + ": " + x.ToString() + Environment.NewLine;
                        }                        
                        Logger.Log("CHardwareProfile - Failed to save hardware profile. Message: {0}. Object Dump: {1}", ELogLevel.Error, ex.Message, objectDump);
                    }
                }
            }
        }

        private void WriteToDatabaseThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(WRITE_INTERVAL);
                try
                {
                    WriteToDatabase();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _writeToDatabaseThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "CHardwareProfile Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info,
                               null,
                               "CHardwareProfile SaveThread Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void SaveProfileToDatabase(SqlConnection pConn, CHardwareProfile pHardwareProfile)
        {            
            SqlCommand command = new SqlCommand("spUpdateHardwareProfile", pConn);
            command.CommandType = CommandType.StoredProcedure;

            SqlParameter param = command.Parameters.Add("@UserID", SqlDbType.Int);
            param.Value = pHardwareProfile.UserID;

            param = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar, 256);
            param.Value = pHardwareProfile.ComputerName;

            param = command.Parameters.Add("@LanguageID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "Language", pHardwareProfile.LanguageName);

            param = command.Parameters.Add("@IsLaptop", SqlDbType.Bit);
            param.Value = pHardwareProfile.IsLaptop;

            param = command.Parameters.Add("@DirectXVersionID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "DirectXVersion", pHardwareProfile.DirectXVersionName);

            // OS
            param = command.Parameters.Add("@OSVersionID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "OperatingSystemVersion", GetNormalizedOSName(pHardwareProfile.OSVersionName));

            param = command.Parameters.Add("@OsIs64Bit", SqlDbType.Bit);
            param.Value = pHardwareProfile.OSIs64Bit;

            param = command.Parameters.Add("@OSServicePackID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "ServicePackVersion", pHardwareProfile.OSServicePackName);

            param = command.Parameters.Add("@OSMajorVersion", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.OSVersionMajor;

            param = command.Parameters.Add("@OSMajorRevision", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.OSVersionMajorRevision;

            param = command.Parameters.Add("@OSMinorVersion", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.OSVersionMinor;

            param = command.Parameters.Add("@OSMinorRevision", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.OSVersionMinorRevision;

            //RAM
            param = command.Parameters.Add("@RamManufacturerID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "RamManufacturer", pHardwareProfile.RamManufacturerName);

            param = command.Parameters.Add("@RamTypeID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "RamType", pHardwareProfile.RamTypeName);

            param = command.Parameters.Add("@RamSize", SqlDbType.SmallInt);
            if (pHardwareProfile.RamSize > Int16.MaxValue) {
                param.Value = Int16.MaxValue;
            }
            else {
                param.Value = pHardwareProfile.RamSize;
            }

            param = command.Parameters.Add("@RamFrequency", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.RamFrequency;

            // CPU
            param = command.Parameters.Add("@CpuManufacturerID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "CpuManufacturer", pHardwareProfile.CpuManufacturerName);

            param = command.Parameters.Add("@CpuFrequency", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.CpuFrequency;

            param = command.Parameters.Add("@CpuModelID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "CpuModel", pHardwareProfile.CpuDescription);

            param = command.Parameters.Add("@CpuCount", SqlDbType.TinyInt);
            param.Value = pHardwareProfile.CpuCount;

            // GPU
            param = command.Parameters.Add("@GpuManufacturerID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "GpuManufacturer", pHardwareProfile.GpuManufacturerName);

            param = command.Parameters.Add("@GpuModelID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "GpuModel", pHardwareProfile.GpuDescription);

            param = command.Parameters.Add("@GpuMemorySize", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.GpuMemorySize;

            param = command.Parameters.Add("@GpuCount", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.GpuCount;

            param = command.Parameters.Add("@GpuDriverNameID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "GpuDriverName", pHardwareProfile.GpuDriverName);

            param = command.Parameters.Add("@GpuDriverVersionID", SqlDbType.Int);
            param.Value = GetLookupID_DriverVersion(pConn, "GpuDriverVersion", pHardwareProfile.GpuDriverVersionName, pHardwareProfile.GpuDriverDate);

            // Display                
            param = command.Parameters.Add("@MonitorModelID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "MonitorModel", pHardwareProfile.MonitorDescription);

            param = command.Parameters.Add("@MonitorResolutionWidth", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.MonitorResolutionWidth;

            param = command.Parameters.Add("@MonitorResolutionHeight", SqlDbType.SmallInt);
            param.Value = pHardwareProfile.MonitorResolutionHeight;

            param = command.Parameters.Add("@MonitorCount", SqlDbType.TinyInt);
            param.Value = pHardwareProfile.MonitorCount;

            // Sound
            param = command.Parameters.Add("@SoundIsPresent", SqlDbType.Bit);
            param.Value = pHardwareProfile.SoundIsPresent;

            param = command.Parameters.Add("@SoundIsPci", SqlDbType.Bit);
            param.Value = pHardwareProfile.SoundIsPci;

            param = command.Parameters.Add("@SoundIsHwAccel", SqlDbType.Bit);
            param.Value = pHardwareProfile.SoundIsHwAccel;

            param = command.Parameters.Add("@SoundModelID", SqlDbType.Int);
            param.Value = GetLookupID(pConn, "SoundModel", pHardwareProfile.SoundDescription);

            // HDD
            param = command.Parameters.Add("@HDDCount", SqlDbType.TinyInt);
            param.Value = pHardwareProfile.HDDCount;

            param = command.Parameters.Add("@HDDTotalSpace", SqlDbType.Int);
            param.Value = pHardwareProfile.HDDTotalSpace;

            param = command.Parameters.Add("@HDDFreeSpace", SqlDbType.Int);
            param.Value = pHardwareProfile.HDDFreeSpace;

            command.ExecuteNonQuery();            
        }

        private string GetNormalizedOSName(string name)
        {
            string loweredName = name.ToLowerInvariant();
            if (loweredName.Contains(" 7"))
            {
                return "Windows 7";
            }
            else if (loweredName.Contains(" vista"))
            {
                return "Windows Vista";
            }
            else if (loweredName.Contains(" server") && loweredName.Contains("2008"))
            {
                return "Windows Server 2008";
            }
            else if (loweredName.Contains(" server") && loweredName.Contains("2003"))
            {
                return "Windows Server 2003";
            }
            else if (loweredName.Contains(" xp"))
            {
                return "Windows XP";
            }            
            else if (loweredName.Contains(" 2000"))
            {
                return "Windows 2000";
            }
            else if (loweredName.Contains(" me"))
            {
                return "Windows ME";
            }
            else if (loweredName.Contains(" 98"))
            {
                return "Windows 98";
            }
            else if (loweredName.Contains(" 95"))
            {
                return "Windows 95";
            }
            
            return "Unknown";
        }
              
        private int GetLookupID_DriverVersion(SqlConnection conn, string tableName, string labelValue, DateTime driverDate)
        {
            DateTime date = driverDate;
            if (labelValue == null || labelValue.Length == 0 || date.Year == DateTime.MinValue.Year)
            {
                labelValue = "Unknown";
                date = new DateTime(2000, 1, 1);
            }
            labelValue = labelValue.Trim();

            CustomKey lookupKey = GetLookupKey(tableName, labelValue);
            if (_lookupCache.ContainsKey(lookupKey))
            {
                return _lookupCache[lookupKey];
            }
            SqlCommand command = conn.CreateCommand();
            command.CommandText = "insert into " + tableName + "(LookupLabel, DriverDate) output inserted.LookupID values(@LookupLabel, @DriverDate)";
            SqlParameter param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
            param.Value = labelValue;
            param = command.Parameters.Add("@DriverDate", System.Data.SqlDbType.DateTime);
            param.Value = date;
            int id = (int)command.ExecuteScalar();
            _lookupCache[lookupKey] = id;
            return id;
        }
    }
}
