﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Configuration;
using System.Threading;
using System.Data.SqlClient;
using Curse.ClientService.Models;
using Curse.ClientService.Extensions;
using System.Data;

namespace Curse.ClientService
{
    public class CNotificationCache
    {
        private static readonly CNotificationCache _instance = new CNotificationCache();

        private readonly string _databaseConnectionString;        
        private readonly int _updateThreadInterval = 30000; // 30 Seconds
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private List<CNotification> _notificationCache;
        private Thread _updateThread = null;
        
        private CNotificationCache()
        {            
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _notificationCache = new List<CNotification>();

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();  
            
        }

        public static CNotificationCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info,
                               null,
                               "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void UpdateCache()
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                List<CNotification> notifications = new List<CNotification>(_notificationCache);

                // Remove all expired notifications
                notifications.RemoveAll(p => p.DateExpires < DateTime.UtcNow);

                SqlCommand command = new SqlCommand("select * from Notification where DateModified > @DateModified and DateExpires > GETUTCDATE()", conn);
                SqlParameter param = command.Parameters.Add("@DateModified", SqlDbType.DateTime);
                param.Value = _lastQueryTime;
                                
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    _lastQueryTime = DateTime.UtcNow;

                    while (reader.Read())
                    {
                        CNotification notification = new CNotification();
                        notification.SetFromDataReader(reader);
                        notifications.RemoveAll(p => p.ID == notification.ID);                        
                        notifications.Add(notification);
                    }
                }
                _notificationCache = notifications;
            }
        }


        public List<CNotification> GetNotifications(int notificationID)
        {
            return _notificationCache.Where(p => p.ID > notificationID).ToList();
        }
    }
}
