﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Threading;
using System.Reflection;

using Curse.Extensions;
using System.Web.Caching;
using Curse.ClientService.AdminModels;

namespace Curse.ClientService
{
    public class CUsageReportingCache : BaseLookupCache
    {
        private static readonly CUsageReportingCache _instance = new CUsageReportingCache();

        private const int WRITE_INTERVAL = 11000;
        private Thread _writeToDatabaseThread = null;
        private List<CUsageProfile> _userProfiles = new List<CUsageProfile>();
        
        private object _userProfilesLock = new object();

        public static CUsageReportingCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public void Initialize() { }


        private CUsageReportingCache()
            : base(ConfigurationManager.ConnectionStrings["ClientUsageReporting"].ConnectionString, new[]
        {
            "ClientVersion",
            "OperatingSystemVersion",
            "OperatingSystemPlatform"
        })
        {
            _writeToDatabaseThread = new Thread(WriteToDatabaseThread) { IsBackground = true };
            _writeToDatabaseThread.Start();
        }

        public bool SaveReport(int userID, CUsageProfile usageProfile)
        {
            if (!_lookupsLoaded || usageProfile == null || usageProfile.InstalledAddons == null)
            {
                return false;
            }

            usageProfile.UserID = userID;
            usageProfile.Timestamp = DateTime.UtcNow;

            lock (_userProfilesLock)
            {
                _userProfiles.Add(usageProfile);
            }
            return true;
        }

        private void WriteToDatabase()
        {
            List<CUsageProfile> profiles = null;

            lock (_userProfilesLock)
            {
                profiles = new List<CUsageProfile>(_userProfiles);
                _userProfiles.Clear();
            }            

            if (profiles.Count == 0)
            {
                return;
            }

            List<Int32> userIDs = (from p in profiles select p.UserID).ToList();

            try
            {
                SaveUsers(profiles);
            }
            catch (SqlException ex)
            {
                Logger.Log("SQL Exception occurred while saving users. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            catch (Exception ex)
            {
                Logger.Log("Unknown Exception occurred while saving users. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }

            try
            {
                SaveInstalledAddons(profiles);
            }
            catch (SqlException ex)
            {
                Logger.Log("SQL Exception occurred while saving addons. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            catch (Exception ex)
            {
                Logger.Log("Unknown Exception occurred while saving addons. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }

            try
            {
                SaveGameMetrics(profiles);
            }
            catch (SqlException ex)
            {
                Logger.Log("SQL Exception occurred while saving Game Metrics. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            catch (Exception ex)
            {
                Logger.Log("Unknown Exception occurred while saving Game Metrics. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }

            try
            {
                SaveGameTabData(profiles);
            }
            catch (SqlException ex)
            {
                Logger.Log("SQL Exception occurred while saving Game Tab Metrics. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            catch (Exception ex)
            {
                Logger.Log("Unknown Exception occurred while saving Game Tab Metrics. Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
        }

        private void WriteToDatabaseThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(WRITE_INTERVAL);
                try
                {
                    WriteToDatabase();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _writeToDatabaseThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "CHardwareProfile Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info, null, "CHardwareProfile SaveThread Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        public List<AddonInstallCount> GetInstallCounts()
        {

            if (HttpRuntime.Cache["InstallCounts"] != null)
            {
                return (List<AddonInstallCount>)HttpRuntime.Cache["InstallCounts"];
            }

            List<AddonInstallCount> installCounts = new List<AddonInstallCount>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand command = conn.CreateCommand();
                command.CommandText = "select AddonID, count(0) as InstallCount from InstalledAddon with(nolock) group by AddonID;";
                command.CommandTimeout = 300;
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        installCounts.Add(new AddonInstallCount()
                        {
                            AddonID = reader.GetInt32(0),
                            InstallCount = reader.GetInt32(1)
                        });
                    }
                }
            }
            HttpRuntime.Cache.Add("InstallCounts", installCounts, null, Cache.NoAbsoluteExpiration, TimeSpan.FromMinutes(15), CacheItemPriority.High, null);
            return installCounts;
        }

        private void SaveInstalledAddons(List<CUsageProfile> usageProfiles)
        {

            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();

                DataTable addonTable = new DataTable();
                addonTable.Columns.Add("AddonID", typeof(int));
                addonTable.Columns.Add("UserID", typeof(int));
                addonTable.BeginLoadData();

                Dictionary<int, List<int>> addonsProcessed = new Dictionary<int, List<int>>();

                foreach (CUsageProfile usageProfile in usageProfiles)
                {
                    //addonsProcessed.Clear();
                    foreach (int addonID in usageProfile.InstalledAddons)
                    {
                        if (addonsProcessed.ContainsKey(usageProfile.UserID))
                        {
                            if (addonsProcessed[usageProfile.UserID].Contains(addonID))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            addonsProcessed.Add(usageProfile.UserID, new List<int>());
                        }

                        DataRow row = addonTable.NewRow();
                        row["AddonID"] = addonID;
                        row["UserID"] = usageProfile.UserID;
                        addonTable.Rows.Add(row);
                        addonsProcessed[usageProfile.UserID].Add(addonID);
                    }
                }

                cmd = conn.CreateCommand();
                cmd.CommandText = "CREATE TABLE #InstalledAddon (AddonID int, UserID int);";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#InstalledAddon";
                    bulk.WriteToServer(addonTable);
                }

                SqlTransaction transaction = conn.BeginTransaction();
                SqlCommand updateCommand = new SqlCommand("delete from InstalledAddon where InstalledAddon.UserID in(select UserID from #InstalledAddon);" +
                "insert into InstalledAddon select * from #InstalledAddon;", conn, transaction);
                updateCommand.CommandTimeout = 600;
                updateCommand.ExecuteNonQuery();
                transaction.Commit();
            }
        }

        private void SaveUsers(List<CUsageProfile> usageProfiles)
        {
            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();

                DataTable userTable = new DataTable();
                userTable.Columns.Add("UserID", typeof(int));
                userTable.Columns.Add("DateFirstSeen", typeof(DateTime));
                userTable.Columns.Add("DateLastSeen", typeof(DateTime));
                userTable.Columns.Add("ClientVersionID", typeof(int));
                userTable.Columns.Add("OperatingSystemVersionID", typeof(int));
                userTable.Columns.Add("OperatingSystemPlatformID", typeof(int));
                userTable.Columns.Add("MainWindowHeight", typeof(float));
                userTable.Columns.Add("MainWindowWidth", typeof(float));
                userTable.Columns.Add("ViewShowSidebar", typeof(bool));
                userTable.Columns.Add("ViewShowGameHeader", typeof(bool));
                userTable.Columns.Add("ViewShowToolbarLabel", typeof(bool));
                userTable.Columns.Add("DefaultGameId", typeof(int));
                userTable.Columns.Add("RoundedCorners", typeof(bool));
                userTable.Columns.Add("SoftwareRendering", typeof(bool));
                userTable.Columns.Add("DotNetVersion", typeof(float));
                userTable.BeginLoadData();

                foreach (CUsageProfile usageProfile in usageProfiles)
                {
                    DataRow row = userTable.NewRow();
                    row["UserID"] = usageProfile.UserID;
                    row["DateFirstSeen"] = usageProfile.Timestamp;
                    row["DateLastSeen"] = usageProfile.Timestamp;
                    row["ClientVersionID"] = GetLookupID(conn, "ClientVersion", usageProfile.ClientVersion);
                    row["OperatingSystemVersionID"] = GetLookupID(conn, "OperatingSystemVersion", usageProfile.OperatingSystemVersion);
                    row["OperatingSystemPlatformID"] = GetLookupID(conn, "OperatingSystemPlatform", usageProfile.OperatingSystemPlatform);
                    row["MainWindowHeight"] = (int)usageProfile.MainWindowHeight;
                    row["MainWindowWidth"] = (int)usageProfile.MainWindowWidth;
                    row["ViewShowSidebar"] = usageProfile.ViewShowSidebar;
                    row["ViewShowGameHeader"] = usageProfile.ViewShowGameHeader;
                    row["ViewShowToolbarLabel"] = usageProfile.ViewShowToolbarLabel;
                    row["DefaultGameId"] = usageProfile.DefaultGameID;
                    row["RoundedCorners"] = usageProfile.RoundedCorners;
                    row["SoftwareRendering"] = usageProfile.SoftwareRendering;
                    row["DotNetVersion"] = usageProfile.DotNetVersion;
                    userTable.Rows.Add(row);
                }

                // Save out the users:
                cmd.CommandText = "create table #ClientUser (UserID int, DateFirstSeen datetime, DateLastSeen datetime, ClientVersionID int, OperatingSystemVersionID int, OperatingSystemPlatformID int,"
                + " MainWindowHeight float, MainWindowWidth float, ViewShowSidebar bit, ViewShowGameHeader bit, ViewShowToolbarLabel bit, DefaultGameId int, RoundedCorners bit, SoftwareRendering bit, DotNetVersion float);";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#ClientUser";
                    bulk.WriteToServer(userTable);
                }

                // Update existing
                cmd.CommandText = "update ClientUser"
                    + " set DateLastSeen = tmp.DateLastSeen,"
                    + " ClientVersionID = tmp.ClientVersionID,"
                    + " OperatingSystemVersionID = tmp.OperatingSystemVersionID,"
                    + " OperatingSystemPlatformID = tmp.OperatingSystemPlatformID,"
                    + " MainWindowHeight = tmp.MainWindowHeight,"
                    + " MainWindowWidth = tmp.MainWindowWidth,"
                    + " ViewShowSidebar = tmp.ViewShowSidebar,"
                    + " ViewShowGameHeader = tmp.ViewShowGameHeader,"
                    + " ViewShowToolbarLabel = tmp.ViewShowToolbarLabel,"
                    + " DefaultGameId = tmp.DefaultGameId," 
                    + " RoundedCorners = tmp.RoundedCorners,"
                    + " SoftwareRendering = tmp.SoftwareRendering, "
                    + " DotNetVersion = tmp.DotNetVersion"
                    + " from ClientUser, #ClientUser as tmp where ClientUser.UserID = tmp.UserID;";

                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();

                //Insert new
                cmd.CommandText = "insert into ClientUser select tmp.UserID, tmp.DateFirstSeen, tmp.DateLastSeen, tmp.ClientVersionID, tmp.OperatingSystemVersionID,"
                    + " tmp.OperatingSystemPlatformID, tmp.MainWindowHeight, tmp.MainWindowWidth, tmp.ViewShowSidebar, tmp.ViewShowGameHeader, tmp.ViewShowToolbarLabel, NULL,"
                    + " tmp.DefaultGameId, tmp.RoundedCorners, tmp.SoftwareRendering, tmp.DotNetVersion"
                    + " from #ClientUser as tmp where not exists(select top 1 1 from ClientUser where ClientUser.UserID = tmp.UserID)";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
            }
        }

        private void SaveGameMetrics(List<CUsageProfile> usageProfiles)
        {
            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();

                DataTable metricsTable = new DataTable();
                metricsTable.Columns.Add("UserID", typeof(int));
                metricsTable.Columns.Add("GameID", typeof(int));
                metricsTable.Columns.Add("TimesSelected", typeof(int));
                metricsTable.Columns.Add("InfoClickWebsite", typeof(int));
                metricsTable.Columns.Add("InfoClickChangelog", typeof(int));
                metricsTable.Columns.Add("InfoClickDescription", typeof(int));
                metricsTable.Columns.Add("InfoClickDonate", typeof(int));
                //InfoClickCategories
                //InfoClickAuthor
                metricsTable.Columns.Add("PlayButtonClick", typeof(int));
                metricsTable.Columns.Add("LocalSavedCount", typeof(int));
                metricsTable.Columns.Add("DateCreated", typeof(DateTime));
                metricsTable.Columns.Add("DateModified", typeof(DateTime));
                //userTable.Columns.Add("AddedType", typeof(int));
                metricsTable.Columns.Add("InstanceCount", typeof(int));
                metricsTable.Columns.Add("SnapshotCount", typeof(int)); 
                metricsTable.Columns.Add("ColumnCount", typeof(int));
                metricsTable.BeginLoadData();

                foreach (CUsageProfile usageProfile in usageProfiles)
                {
                    if (usageProfile.GameMetrics == null)
                    {
                        continue;
                    }
                    
                    foreach (var metric in usageProfile.GameMetrics)
                    {
                        DataRow row = metricsTable.NewRow();
                        row["UserID"] = usageProfile.UserID;
                        row["GameID"] = metric.GameId;
                        row["TimesSelected"] = metric.TimesSelected;
                        row["InfoClickWebsite"] = metric.InfoClickWebsite;
                        row["InfoClickChangelog"] = metric.InfoClickChangeLog;
                        row["InfoClickDescription"] = metric.InfoClickDescription;
                        row["InfoClickDonate"] = metric.InfoClickDonate;
                        row["PlayButtonClick"] = metric.PlayButtonClick;
                        row["LocalSavedCount"] = metric.LocalSavesCount;
                        row["DateCreated"] = usageProfile.Timestamp;
                        row["DateModified"] = usageProfile.Timestamp;
                        row["InstanceCount"] = metric.InstanceCount;
                        row["SnapshotCount"] = metric.SnapshotCount;
                        row["ColumnCount"] = metric.ColumnCount;

                        metricsTable.Rows.Add(row);
                    }
                }

                // Save out the users:
                cmd.CommandText = "CREATE TABLE #GameMetrics (UserID int, GameID int, TimesSelected int, InfoClickWebsite int, InfoClickChangelog int, InfoClickDescription int, InfoClickDonate int, PlayButtonClick int, LocalSavedCount int, DateCreated datetime, DateModified datetime, InstanceCount int, SnapshotCount int, ColumnCount int)";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#GameMetrics";
                    bulk.WriteToServer(metricsTable);
                }

                // Update existing
                cmd.CommandText = "UPDATE GameMetrics " +
                    "SET UserID = tmp.UserID, " +
                    "	GameID = tmp.GameID, " +
                    "   TimesSelected = (GameMetrics.TimesSelected + tmp.TimesSelected), " +
                    "	InfoClickWebsite = (GameMetrics.InfoClickWebsite + tmp.InfoClickWebsite), " +
                    "	InfoClickChangelog = (GameMetrics.InfoClickChangelog  + tmp.InfoClickChangelog), " +
                    "	InfoClickDescription = (GameMetrics.InfoClickDescription + tmp.InfoClickDescription), " +
                    "	InfoClickDonate = (GameMetrics.InfoClickDonate + tmp.InfoClickDonate), " +
                    "	PlayButtonClick = (GameMetrics.PlayButtonClick + tmp.PlayButtonClick), " +
                    "   LocalSavesCount = tmp.LocalSavedCount, " +
                    "   DateModified = tmp.DateModified, " +
                    "   InstanceCount = tmp.InstanceCount, " +
                    "   SnapshotCount = GameMetrics.SnapshotCount + tmp.SnapshotCount, " +
                    "   ColumnCount = tmp.ColumnCount " +
                    "FROM GameMetrics, #GameMetrics as tmp " +
                    "WHERE GameMetrics.UserID = tmp.UserID AND GameMetrics.GameID = tmp.GameID";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();

                //Insert new
                cmd.CommandText = "INSERT INTO GameMetrics (UserID, GameID, TimesSelected, InfoClickWebsite, InfoClickChangelog, InfoClickDescription, InfoClickDonate, PlayButtonClick, LocalSavesCount, DateCreated, DateModified, InstanceCount, SnapshotCount, ColumnCount) " +
                    "SELECT tmp.UserID, " +
                    "	tmp.GameID, " +
                    "	tmp.TimesSelected, " +
                    "	tmp.InfoClickWebsite, " +
                    "	tmp.InfoClickChangelog, " +
                    "	tmp.InfoClickDescription, " +
                    "	tmp.InfoClickDonate, " +
                    "	tmp.PlayButtonClick, " +
                    "	tmp.LocalSavedCount, " +
                    "	tmp.DateCreated, " +
                    "	tmp.DateModified, " +
                    "	tmp.InstanceCount, " +
                    "	tmp.SnapshotCount, " +
                    "	tmp.ColumnCount " +
                    "FROM #GameMetrics AS tmp " +
                    "WHERE NOT EXISTS(SELECT TOP 1 1 FROM GameMetrics WHERE GameMetrics.UserID = tmp.UserID AND GameMetrics.GameID = tmp.GameID)";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
            }
        }

        private void SaveGameTabData(List<CUsageProfile> usageProfiles)
        {
            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();

                DataTable gameTabTable = new DataTable();
                gameTabTable.Columns.Add("GameMetricID", typeof(int));
                gameTabTable.Columns.Add("GameTabID", typeof(int));
                gameTabTable.Columns.Add("TimesSelected", typeof(int));
                gameTabTable.Columns.Add("DateCreated", typeof(DateTime));
                gameTabTable.Columns.Add("DateModified", typeof(DateTime));
                gameTabTable.BeginLoadData();

                foreach (CUsageProfile usageProfile in usageProfiles)
                {
                    if (usageProfile.GameMetrics == null)
                    {
                        continue;
                    }

                    foreach (var metric in usageProfile.GameMetrics)
                    {
                        if (metric.TabsUsed == null)
                        {
                            continue;
                        }

                        foreach (var tabName in metric.TabsUsed)
                        {
                            DataRow row = gameTabTable.NewRow();
                            row["GameMetricID"] = GetGameMetricID(conn, usageProfile.UserID, metric.GameId);
                            row["GameTabID"] = GetOrCreateGameTabID(conn, metric.GameId, tabName);
                            row["TimesSelected"] = metric.TimesSelected;
                            row["DateCreated"] = usageProfile.Timestamp;
                            row["DateModified"] = usageProfile.Timestamp;
                            gameTabTable.Rows.Add(row);
                        }
                    }
                }

                // Save out the users:
                cmd.CommandText = "CREATE TABLE #GameMetricsGameTabMapping (GameMetricID int, GameTabID int, TimesSelected int, DateCreated datetime, DateModified datetime)";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#GameMetricsGameTabMapping";
                    bulk.WriteToServer(gameTabTable);
                }

                // Update existing
                cmd.CommandText = "UPDATE GameMetricsGameTabMapping SET TimesSelected = (GameMetricsGameTabMapping.TimesSelected + 1), DateModified = tmp.DateModified FROM GameMetricsGameTabMapping, #GameMetricsGameTabMapping AS tmp WHERE GameMetricsGameTabMapping.GameMetricID = tmp.GameMetricID AND GameMetricsGameTabMapping.GameTabID = tmp.GameTabID";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();

                //Insert new
                cmd.CommandText = "INSERT INTO GameMetricsGameTabMapping (GameMetricID, GameTabID, TimesSelected, DateCreated, DateModified) " +
                    "SELECT tmp.GameMetricID, " +
                    "	tmp.GameTabID, " +
                    "	tmp.TimesSelected, " +
                    "	tmp.DateCreated, " +
                    "	tmp.DateModified " +
                    "FROM #GameMetricsGameTabMapping AS tmp " +
                    "WHERE NOT EXISTS(SELECT TOP 1 1 FROM GameMetricsGameTabMapping WHERE GameMetricsGameTabMapping.GameMetricID = tmp.GameMetricID AND GameMetricsGameTabMapping.GameTabID = tmp.GameTabID)";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
            }
        }

        private int GetGameMetricID(SqlConnection conn, int userId, int gameId)
        {
            var cmd = new SqlCommand("SELECT ID FROM GameMetrics WHERE UserID = @UserID AND GameID = @GameID", conn);
            cmd.Parameters.Add("UserID", SqlDbType.Int).Value = userId;
            cmd.Parameters.Add("GameID", SqlDbType.Int).Value = gameId;

            if (conn.State != ConnectionState.Open)
            {
                conn.Open();
            }
            object value = cmd.ExecuteScalar();
            if (value != null)
            {
                return (int)value;
            }

            return 0;
        }

        private int GetOrCreateGameTabID(SqlConnection conn, int gameId, string tabName)
        {
            var cmd = new SqlCommand("SELECT ID FROM GameTab WHERE GameID = @GameID AND TabDescription = @TabDescription", conn);
            cmd.Parameters.Add("GameID", SqlDbType.Int).Value = gameId;
            cmd.Parameters.Add("TabDescription", SqlDbType.VarChar).Value = tabName;

            if (conn.State != ConnectionState.Open)
            {
                conn.Open();
            }
            object value = cmd.ExecuteScalar();
            if (value != null)
            {
                return (int)value;
            }

            cmd.CommandText = "INSERT INTO GameTab (GameID, TabDescription) VALUES(@GameID, @TabDescription) SET @ID = SCOPE_IDENTITY()";
            cmd.Parameters.Add("ID", SqlDbType.Int);
            cmd.Parameters["ID"].Direction = ParameterDirection.Output;

            cmd.ExecuteNonQuery();
            value = cmd.Parameters["ID"].Value;
            if (value != null)
            {
                return (int)value;
            }

            return 0;
        }
    }
}
