﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using System.Text.RegularExpressions;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;

namespace Curse.ClientService.AddOns
{
    [DataContract]
    public class CAddOn
    {
        // Statics
        private readonly static string sFeedBaseUrl;
        private readonly static string sDownloadBaseUrl;
        private readonly static string sCategoryBaseUrl;
        private readonly static string sAuthorBaseUrl;
        private readonly static Regex sHtmlStripper = new Regex("<!--.*?-->|<[^>]*>", RegexOptions.Compiled);

        static CAddOn()
        {
            sFeedBaseUrl = ConfigurationManager.AppSettings["FeedBaseUrl"];
            sDownloadBaseUrl = ConfigurationManager.AppSettings["DownloadBaseUrl"];
            sCategoryBaseUrl = ConfigurationManager.AppSettings["CategoryBaseUrl"];
            sAuthorBaseUrl = ConfigurationManager.AppSettings["AuthorBaseUrl"];
        }

        // Public Members and Properties
        [DataMember]
        public int Id;

        [DataMember]
        public string Name;

        [DataMember]
        public string WebSiteURL;

        [DataMember]
        public string FeedURL;
        
        [DataMember]
        public int GameId;
        
        [DataMember]
        public string Summary;
        
        [DataMember]
        public int DefaultFileId;

        [DataMember]
        public int CommentCount;
        
        [DataMember]
        public double DownloadCount;

        [DataMember]
        public int Rating;

        [DataMember]
        public List<CAddOnFile> LatestFiles
        {
            get
            {
                return mLatestFiles;
            }
        }

        // Privates
        private string gameName;
        private string portalName;
        private string sectionName;
        private string downloadName;
        
        private string description;                
        
        private DateTime dateModified;
        private DateTime dateCreated;
        private DateTime dateReleased;
        private List<KeyValuePair<string,string>> categories;
        string primaryAuthorName;
        private List<CAddOnAuthor> authors;        
        private Dictionary<int,CAddOnFile> files;                
        private string defaultFileName;
        private List<CAddOnFile> mLatestFiles = new List<CAddOnFile>();        
        private bool isAvailable = false;
        public EProjectStatus status = EProjectStatus.Normal;
        //private string donationLink = null;


        public Dictionary<int, CAddOnFile> Files
        {
            get
            {
                return files;
            }
        }

        
       
        public bool Available
        {
            get
            {
                return isAvailable;
            }
            set
            {
                isAvailable = value;
            }
        }

        public bool IncludeInIndex
        {
            get
            {
                return status != EProjectStatus.Deleted;
            }
        }
        
                  
        public static string GetFeedUrl(SqlDataReader pReader)
        {
            return String.Format(sFeedBaseUrl, pReader.GetInt32(pReader.GetOrdinal("addon_id")));
        }

        public CAddOn()
        {
        }

        public CAddOn(SqlConnection pConn,
            SqlDataReader pReader,
            Dictionary<int, List<CIndividualFileFingerprint>> pIndividualFingerprints,
            Dictionary<int, List<CAddOnFile>> pFileArchiveCache,
            Dictionary<int, List<CAddOnFileDependency>> pFileDependencyCache)
        {
            Id = pReader.GetInt32(pReader.GetOrdinal("addon_id"));
            Name = pReader.GetString(pReader.GetOrdinal("addon_name"));
            FeedURL = GetFeedUrl(pReader);
            

            portalName = pReader.GetString(pReader.GetOrdinal("addon_portal"));
            sectionName = pReader.GetString(pReader.GetOrdinal("addon_section"));
            downloadName = pReader.GetString(pReader.GetOrdinal("addon_download_name"));
            WebSiteURL = String.Format(sDownloadBaseUrl, portalName, sectionName, downloadName);            

            gameName = pReader.GetString(pReader.GetOrdinal("addon_game"));
            GameId = pReader.GetInt32(pReader.GetOrdinal("addon_game_id"));            
            dateModified = pReader.GetDateTime(pReader.GetOrdinal("addon_last_updated"));
            dateCreated = pReader.GetDateTime(pReader.GetOrdinal("addon_date_created"));
            dateReleased = pReader.GetDateTime(pReader.GetOrdinal("addon_date_released"));
            CommentCount = pReader.GetInt32(pReader.GetOrdinal("addon_num_comments"));
            DownloadCount = pReader.GetDouble (pReader.GetOrdinal("addon_num_downloads"));
            Rating = pReader.GetInt32(pReader.GetOrdinal("addon_average_rating"));
            description = pReader.GetString(pReader.GetOrdinal("addon_description"));
            isAvailable = pReader.GetBoolean(pReader.GetOrdinal("addon_is_available"));
            status = (EProjectStatus)pReader.GetByte(pReader.GetOrdinal("addon_status"));
            Summary = sHtmlStripper.Replace(description, String.Empty);
            Summary = Summary.Replace(Environment.NewLine, " ");
            Summary = Summary.Replace("\n", " ");
            if (Summary.Length > 512)
            {
                Summary = Summary.Substring(0, 512);
            }
            primaryAuthorName = pReader.GetString(pReader.GetOrdinal("addon_primary_author"));
            populateAuthors(pConn, primaryAuthorName);
            populateFiles(pConn, pFileArchiveCache, pFileDependencyCache);            
            populateFileFingerprints(pConn);
            populateIndividualFileFingerprints(pIndividualFingerprints);
            populateCategories(pConn);                        
        }
                     
        private void populateAuthors(SqlConnection pConn, string primaryAuthor)
        {
            authors = new List<CAddOnAuthor>();
            authors.Add(new CAddOnAuthor(primaryAuthor, String.Format(sAuthorBaseUrl, primaryAuthor)));

            SqlCommand command = new SqlCommand("curseService_GetAddOnAuthors", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = Id;
            
            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    if(reader.GetString(0) != primaryAuthor)
                    {
                        authors.Add(new CAddOnAuthor(reader.GetString(0), String.Format(sAuthorBaseUrl,reader.GetString(0))));
                    }

                }
            }            
        }

        public CAddOn Clone()
        {
            return (CAddOn)MemberwiseClone();
        }

        private void populateCategories(SqlConnection pConn)
        {

            categories = new List<KeyValuePair<string, string>>();
            SqlCommand command = new SqlCommand("curseService_GetAddOnCategories", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = Id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    KeyValuePair<string, string> category = new KeyValuePair<string, string>(reader.GetString(2), reader.GetString(0));
                    categories.Add(category);
                }
            }

            if (categories.Count == 0)
            {
                KeyValuePair<string, string> category = new KeyValuePair<string, string>("17", "Miscellaneous");
                categories.Add(category);
            }

        }

        private int getDefaultFileId()
        {            
            // Release
            if (mLatestFiles.Exists(p => p.FileType == EFileType.Release && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.FileType == EFileType.Release && p.IsAlternate == false).Id;
            }

            // Beta
            if (mLatestFiles.Exists(p => p.FileType == EFileType.Beta && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.FileType == EFileType.Beta && p.IsAlternate == false).Id;
            }

            // Alpha
            if (mLatestFiles.Exists(p => p.FileType == EFileType.Alpha && p.IsAlternate == false))
            {
                return mLatestFiles.First(p => p.FileType == EFileType.Alpha && p.IsAlternate == false).Id;
            }

            return 0;
        }

        private void populateFiles(SqlConnection pConn,
                                   Dictionary<int, List<CAddOnFile>> pFileArchiveCache,
                                   Dictionary<int, List<CAddOnFileDependency>> pFileDependencyCache)
        {
            files = new Dictionary<int,CAddOnFile>();
            SqlCommand command = new SqlCommand("curseService_GetAddOnFiles", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = Id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                while (reader.Read())
                {
                    
                    CAddOnFile file = new CAddOnFile(reader, false);                                      
                    files.Add(file.Id, file);

                    // Set the dependencies:
                    if (pFileDependencyCache.ContainsKey(file.Id))
                    {
                        file.Dependencies = pFileDependencyCache[file.Id];
                    }                    
                }
            }

            setLatestFiles();
            DefaultFileId = getDefaultFileId();
            if (pFileArchiveCache.ContainsKey(Id))
            {
                foreach (CAddOnFile file in pFileArchiveCache[Id])
                {
                    if (!files.ContainsKey(file.Id))
                    {
                        files.Add(file.Id, file);
                    }
                }                
            }
            defaultFileName = files[DefaultFileId].FileName;
        }

        private void setLatestFiles()
        {

            // Set latest files:
            foreach (KeyValuePair<int, CAddOnFile> kvp in files)
            {
                CAddOnFile file = kvp.Value;
                if (!file.IsAlternate)
                {

                    if (file.AlternateFileId > 0 && !files.ContainsKey(file.AlternateFileId))
                    {
                        file.AlternateFileId = 0;
                    }

                    CAddOnFile existingFile = mLatestFiles.FirstOrDefault(p => p.FileType == file.FileType && p.IsAlternate == false);

                    if (existingFile == null || file.FileDate > existingFile.FileDate)
                    {
                        if (existingFile != null)
                        {
                            mLatestFiles.Remove(existingFile);
                            mLatestFiles.Find(p => p.Id == existingFile.AlternateFileId);
                        }

                        mLatestFiles.Add(file);
                    }

                }
            }

            // Now the alternates:
            foreach (EFileType releaseType in System.Enum.GetValues(typeof(EFileType)))
            {

                CAddOnFile primaryFile = mLatestFiles.FirstOrDefault(p => p.FileType == releaseType);

                if (primaryFile == null)
                {
                    continue;
                }

                if (primaryFile.AlternateFileId > 0)
                {
                    if (files.ContainsKey(primaryFile.AlternateFileId))
                    {
                        CAddOnFile secondaryFile = files[primaryFile.AlternateFileId];
                        if (!mLatestFiles.Exists(p => p.Id == secondaryFile.Id))
                        {
                            mLatestFiles.Add(secondaryFile);
                        }
                    }
                }
            }
        }
        
        private void populateFileFingerprints(SqlConnection pConn)
        {
            SqlCommand command = new SqlCommand("curseService_GetAddOnFileFingerprints", pConn);
            command.CommandType = CommandType.StoredProcedure;
            command.Connection = pConn;
            command.Parameters.Add("@intProjectId", SqlDbType.Int);
            command.Parameters["@intProjectId"].Value = Id;

            using (SqlDataReader reader = command.ExecuteReader())
            {
                if (!reader.HasRows)
                {
                    return;
                }

                while (reader.Read())
                {
                    int fileId = reader.GetInt32(0);
                    long fingerprint = reader.GetInt64(1);
                    if (!files.ContainsKey(fileId))
                    {
                        continue;
                    }
                    files[fileId].Fingerprints.Add(fingerprint);                                        
                }
            }
        }

        private void populateIndividualFileFingerprints(Dictionary<int, List<CIndividualFileFingerprint>> pIndividualFingerprints)
        {

            if (!pIndividualFingerprints.ContainsKey(Id))
            {
                return;
            }

            foreach (CIndividualFileFingerprint fingerprint in pIndividualFingerprints[Id])
            {

                if (!files.ContainsKey(fingerprint.FileId))
                {
                    continue;
                }
                if (!files[fingerprint.FileId].FolderFingerprints.ContainsKey(fingerprint.Folder))
                {
                    files[fingerprint.FileId].FolderFingerprints.Add(fingerprint.Folder, new List<long>());
                    files[fingerprint.FileId].FolderNames.Add(fingerprint.Folder);
                }
                files[fingerprint.FileId].FolderFingerprints[fingerprint.Folder].Add(fingerprint.Fingerprint);                
            }
            
            // Make sure the individual fingerprints are sorted:
            foreach (KeyValuePair<int, CAddOnFile> fileEntry in files)
            {
                foreach (KeyValuePair<string, List<long>> folderFingerprints in fileEntry.Value.FolderFingerprints)
                {
                    folderFingerprints.Value.Sort();
                }
            }
        }      
                               
    }
}
