﻿using System;
using Curse.AddOns;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Collections.Generic;
using Curse.ClientService.Extensions;
using System.IO;
using Curse.Extensions;
using System.Runtime.Serialization.Formatters.Binary;
using System.Linq;

namespace Curse.ClientService{
    public class CAvatarCache
    {
        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;
        private string _staticFileDestination = null;
        private bool _createFeedFiles = false;

        private static readonly CAvatarCache _instance = new CAvatarCache();
        private List<CAvatar> _avatars = new List<CAvatar>();

        public static CAvatarCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CAvatarCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());

            if (_createFeedFiles) {
                LoadFromDisk();
            }//faster loading for the feed machine
            else {
                UpdateCache();
            }//load from db

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }        

        public DateTime FileDate
        {
            get;
            set;
        }

        public List<CAvatar> Avatars { get { return _avatars; } }

        public void Initialize() {}

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }//while !aborted
        }//Cache Thread

        private void UpdateCache() {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }//try
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }//catch

                List<CAvatar> avatars = new List<CAvatar>();

                SqlCommand command = new SqlCommand("SELECT ID, Status, FileName, FileLength, Width, Height FROM Avatar with(nolock) Where EntityTypeID in (10006, 10002)" , conn);
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        CAvatar avatar = new CAvatar();
                        avatar.SetFromDataReader(reader);
                        avatars.Add(avatar);
                    } // while reading                                    
                }//using data reader

                lock (_avatars) {
                    _avatars = avatars;
                }//lock avatars

                try {
                    SaveToDisk(avatars);
                }
                catch (Exception ex) {
                    Logger.Log("Unable to create avatar feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }//try to save it to a file
            }//using connection
        }//Update Cache

        private void UpdateCacheDate() {
            string currentfile = null;
            try {
                currentfile = Path.Combine(_staticFileDestination, "Avatars.zip");
                FileDate = new FileInfo(currentfile).LastWriteTimeUtc;
            }
            catch (Exception ex) {
                Logger.Log("Unable to update avatar cache file data at '{0}'! Details: {1}", ELogLevel.Error, currentfile, ex.GetExceptionDetails());
            }
        }//UpdateCacheDate

        private void SaveToDisk(List<CAvatar> avatars)
        {
            if (_createFeedFiles) {
                SaveBinaryToDisk(avatars);
            }//if we're making the feed files

            UpdateCacheDate();
        }//Save To Disk

        private void SaveBinaryToDisk(List<CAvatar> avatars)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, avatars);
                ms.Position = 0;

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, "Avatars.zip");
                using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
                {
                    binWriter.Write(compressed);
                }

                if (File.Exists(currentfile) && FileComparer.CompareFiles(currentfile, tempfile))
                {
                    try
                    {
                        File.Delete(tempfile);
                    }
                    catch (Exception ex)
                    {
                        Logger.Log("CAvatarCache - Unable to delete temp file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                    return;
                }

                try
                {
                    if (File.Exists(currentfile))
                    {
                        File.Replace(tempfile, currentfile, backupfile);
                    }
                    else
                    {
                        File.Copy(tempfile, currentfile, true);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("CAvatarCache - SaveToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally
                {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile))
                    {
                        File.Delete(backupfile);
                    }
                }
            } //using the memory stream
        }//SaveBinary File

        private void LoadFromDisk() {
            var avatars = new List<CAvatar>();
            string currentfile = Path.Combine(_staticFileDestination, "Avatars.zip");
            if (!File.Exists(currentfile)) {
                Logger.Log("Avatar feed file has not been created", ELogLevel.Info);
                return;
            }//check that the file exists
                        
            // Read the compressed bytes out of the avatars file
            byte[] compressed, uncompressed;
            try
            {
                using (var binReader = new BinaryReader(File.Open(currentfile, FileMode.Open)))
                {
                    compressed = binReader.ReadBytes((int)binReader.BaseStream.Length);
                }//open a reader to the file
            }//try to read the file
            catch (Exception ex) {
                Logger.Log("Unable to load avatars from feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return;
            }//catch

            // Deserialize the stream
            try
            {
                uncompressed = Utility.GetDecompressedBytes(compressed);
                using (var ms = new MemoryStream())
                {
                    ms.Write(uncompressed, 0, uncompressed.Length);
                    ms.Position = 0;

                    var bf = new BinaryFormatter();
                    avatars = (List<CAvatar>)bf.Deserialize(ms);
                }//using memory stream
            }//try to deserialize the stream
            catch (Exception ex)
            {
                Logger.Log("Unable to deserialize the avatar feed! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return;
            }//catch

            lock (_avatars) {
                _avatars = avatars;
                var avatar = avatars.FirstOrDefault(p => p.Id == 6233);
                if (avatar != null)
                {
                }
            }//set the list
        }//load from disk
    }//end class
}//end namespace
