﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.AddOns;
using System.Data.SqlClient;
using System.Data;

namespace Curse.ClientService
{
    public class CFingerprintCache
    {
        private Dictionary<long, List<CFingerprintMatch>> _fingerprints;
        private Dictionary<int, Dictionary<string, List<CFuzzyMatch>>> _fuzzyIndexFingerprints;
        private bool _isCacheBuilt = false;

        private static readonly CFingerprintCache _instance = new CFingerprintCache();

        public static CFingerprintCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CFingerprintCache()
        {
            _isCacheBuilt = false;
            _fingerprints = new Dictionary<long, List<CFingerprintMatch>>();
            _fuzzyIndexFingerprints = new Dictionary<int, Dictionary<string, List<CFuzzyMatch>>>();
        }

        public void Rebuild(Dictionary<int, CAddOn> addonCache)
        {
            Dictionary<long, List<CFingerprintMatch>> fingerprints = new Dictionary<long, List<CFingerprintMatch>>();
            Dictionary<int, Dictionary<string, List<CFuzzyMatch>>> folderFingerprints = new Dictionary<int, Dictionary<string, List<CFuzzyMatch>>>();

            CAddOn addon = null;
            CAddOnFile file = null;
            string foldername = null;

            // Setup Individual Fingerprints                  
            foreach (CGame game in CGameCache.Instance.Games)
            {
                folderFingerprints.Add(game.ID, new Dictionary<string, List<CFuzzyMatch>>());
            }

            foreach (KeyValuePair<int, CAddOn> addonEntry in addonCache)
            {
                addon = addonEntry.Value;
                var game = CGameCache.Instance.Games.FirstOrDefault(p => p.ID == addon.GameId);

                if (addon.IncludeInFingerprinting)
                {
                    foreach (KeyValuePair<int, CAddOnFile> fileEntry in addon.Files)
                    {
                        file = fileEntry.Value;

                        //if (game != null && game.PackageType == EPackageType.SingleFile)
                        if(addon != null && addon.PackageType == EPackageType.SingleFile)
                        {
                            file.Fingerprints.Clear();
                            file.Fingerprints.Add(file.PackageFingerprint);
                            if ((file.PackageFingerprint != default(long)))
                            {
                                if (!fingerprints.ContainsKey(file.PackageFingerprint))
                                {
                                    fingerprints.Add(file.PackageFingerprint, new List<CFingerprintMatch>());
                                }
                                fingerprints[file.PackageFingerprint].Add(new CFingerprintMatch(addon, file));
                            }
                        }
                        else
                        {
                            // Addon-Level Fingerprints
                            foreach (long fingerprint in file.Fingerprints)
                            {
                                if (!fingerprints.ContainsKey(fingerprint))
                                {
                                    fingerprints.Add(fingerprint, new List<CFingerprintMatch>());
                                }
                                CFingerprintMatch match = new CFingerprintMatch(addon, file);
                                fingerprints[fingerprint].Add(match);
                            }

                            // Module-Level Fingerprints
                            foreach (KeyValuePair<string, List<long>> folder in file.FolderFingerprints)
                            {
                                if (game == null) break;

                                foldername = folder.Key;
                                if (!folderFingerprints[addon.GameId].ContainsKey(foldername))
                                {
                                    folderFingerprints[addon.GameId].Add(foldername, new List<CFuzzyMatch>());
                                }
                                folderFingerprints[addon.GameId][foldername].Add(new CFuzzyMatch(foldername, addon, file));
                            }
                        }
                    }
                }
            }

            lock (_fingerprints)
            {
                _fingerprints = fingerprints;
            }

            lock (_fuzzyIndexFingerprints)
            {
                _fuzzyIndexFingerprints = folderFingerprints;
            }

            _isCacheBuilt = true;
        }

        public List<CFuzzyMatch> GetFuzzyMatches(int gameID, CFolderFingerprint[] folderFingerprints)
        {
            List<CFuzzyMatch> matches = new List<CFuzzyMatch>();

            if (gameID == 0 || !_fuzzyIndexFingerprints.ContainsKey(gameID))
            {
                return matches;
            }

            CFuzzyMatch bestMatch = null;
            int matchCount = 0;
            int potentialMatchCount = 0;

            foreach (CFolderFingerprint folderFingerprint in folderFingerprints)
            {

                List<CFuzzyMatch> potentialMatches;

                if (!_fuzzyIndexFingerprints[gameID].TryGetValue(folderFingerprint.Foldername, out potentialMatches))
                {
                    continue;
                }

                int lastId = 0;
                bool singleAddonMatch = true;
                bestMatch = null;

                foreach (CFuzzyMatch potentialMatch in potentialMatches)
                {
                    if (lastId > 0 && lastId != potentialMatch.Id)
                    {
                        singleAddonMatch = false;
                    }
                    lastId = potentialMatch.Id;
                    if (potentialMatch.Fingerprints.Count != folderFingerprint.Fingerprints.Count)
                    {
                        continue;
                    }
                    potentialMatchCount = potentialMatch.FuzzyMatchCount(folderFingerprint.Fingerprints);

                    if (bestMatch == null ||
                        potentialMatchCount > matchCount ||
                        bestMatch.File.FileStatus == EFileStatus.Deleted && potentialMatch.File.FileStatus != EFileStatus.Deleted)
                    {

                        bestMatch = potentialMatch;
                        matchCount = potentialMatchCount;
                        if (potentialMatchCount == folderFingerprint.Fingerprints.Count)
                        {
                            break;
                        }
                    }
                }

                if (bestMatch == null && singleAddonMatch)
                {
                    bestMatch = potentialMatches.FirstOrDefault();
                }

                // No match found, continue to next folder
                if (bestMatch == null)
                {
                    continue;
                }
                // Ensure we don't add the same match twice
                if (matches.Exists(p => p.Id == bestMatch.Id))
                {
                    continue;
                }
                matches.Add(bestMatch);
            }
            return matches;
        }

        public CFingerprintMatchResult GetMatches(long[] pFileFingerprints)
        {
            CFingerprintMatchResult result = new CFingerprintMatchResult();
            result.IsCacheBuilt = this._isCacheBuilt;
            result.InstalledFingerprints = new List<long>(pFileFingerprints);
            if (!this._isCacheBuilt)
            {
                return result;
            }

            FindExactMatches(result);

            if (result.ExactFingerprints.Count < result.InstalledFingerprints.Count)
            {
                FindPartialMatches(result);
            }

            return result;
        }

        public void FindExactMatches(CFingerprintMatchResult pResult)
        {
            List<long> processedFingerprints = new List<long>();
            List<CFingerprintMatch> matches = new List<CFingerprintMatch>();

            foreach (long installedFingerprint in pResult.InstalledFingerprints)
            {
                if (processedFingerprints.Contains(installedFingerprint))
                {
                    continue;
                }

                if (!_fingerprints.ContainsKey(installedFingerprint))
                {
                    continue;
                }

                List<CFingerprintMatch> potentialMatches = _fingerprints[installedFingerprint];

                if (potentialMatches.Count == 0)
                {
                    continue;
                }

                CFingerprintMatch currentMatch = null;

                foreach (CFingerprintMatch potentialMatch in potentialMatches)
                {
                    // Match must be perfect, and consume more files, be the latest matching release date.
                    if (potentialMatch.IsExactMatch(pResult.InstalledFingerprints) && IsBetterMatch(currentMatch, potentialMatch))
                    {
                        currentMatch = potentialMatch;
                    }
                }

                if (currentMatch != null)
                {
                    processedFingerprints.AddRange(currentMatch.File.Fingerprints);
                    matches.Add(currentMatch);
                }
            }

            pResult.ExactMatches = matches;
            pResult.ExactFingerprints = processedFingerprints;
        }

        public void FindPartialMatches(CFingerprintMatchResult pResult)
        {
            List<long> processedFingerprints = new List<long>();
            List<CFingerprintMatch> matches = new List<CFingerprintMatch>();
            Dictionary<int, List<long>> matchFingerprints = new Dictionary<int, List<long>>();
            List<long> partialMatchFingerprints = null;

            foreach (long installedFingerprint in pResult.UnmatchedFingerprints)
            {
                if (processedFingerprints.Contains(installedFingerprint))
                {
                    continue;
                }
                if (!_fingerprints.ContainsKey(installedFingerprint))
                {
                    continue;
                }
                List<CFingerprintMatch> potentialMatches = _fingerprints[installedFingerprint];

                int matchedCount = 0;
                int potentialMatchCount = 0;
                CFingerprintMatch currentMatch = null;

                foreach (CFingerprintMatch potentialMatch in potentialMatches)
                {
                    partialMatchFingerprints = potentialMatch.GetPartialMatchFingerprints(pResult.InstalledFingerprints);

                    if (IsBetterMatch(currentMatch, potentialMatch))
                    {
                        currentMatch = potentialMatch;
                        matchedCount = potentialMatchCount;
                    }
                }

                if (currentMatch != null)
                {
                    processedFingerprints.AddRange(currentMatch.File.Fingerprints);
                    matchFingerprints[currentMatch.Id] = currentMatch.GetPartialMatchFingerprints(pResult.InstalledFingerprints);
                    matches.Add(currentMatch);
                }
            }
            pResult.PartialMatchFingerprints = matchFingerprints;
            pResult.PartialMatches = matches;
        }

        public bool IsBetterMatch(CFingerprintMatch pCurrentMatch, CFingerprintMatch pPotentialMatch)
        {
            // If there is no current match, then it is always better.
            if (pCurrentMatch == null)
            {
                return true;
            }

            // If this potential match consumes less fingerprints, then it isn't a better match
            if (pPotentialMatch.File.Fingerprints.Count < pCurrentMatch.File.Fingerprints.Count)
            {
                return false;
            }

            // if the current match isn't normal status, and the current one is, it is better
            if (pCurrentMatch.File.FileStatus != EFileStatus.Normal && pPotentialMatch.File.FileStatus == EFileStatus.Normal)
            {
                return true;
            }

            // if the potential match isn't normal status, and the current one is, discard it.
            if (pPotentialMatch.File.FileStatus != EFileStatus.Normal && pCurrentMatch.File.FileStatus == EFileStatus.Normal)
            {
                return false;
            }

            // If this is an earlier release, discard it.
            if (pPotentialMatch.File.FileDate < pCurrentMatch.File.FileDate)
            {
                return false;
            }

            return true;
        }

        public Dictionary<int, List<CIndividualFileFingerprint>> GetIndividualFingerprintCache(SqlConnection conn, DateTime changeDate)
        {
            Dictionary<int, List<CIndividualFileFingerprint>> individualFingerprintCache = new Dictionary<int, List<CIndividualFileFingerprint>>();

            // Get a cache of the individual fingerprints:                                        
            using (SqlCommand fingerprintCommand = new SqlCommand("curseService_GetAllIndividualFingerprints", conn))
            {
                fingerprintCommand.CommandType = CommandType.StoredProcedure;
                fingerprintCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fingerprintCommand.Parameters["@LastUpdated"].Value = changeDate;
                fingerprintCommand.CommandTimeout = 300;

                using (SqlDataReader fingerprintReader = fingerprintCommand.ExecuteReader())
                {
                    while (fingerprintReader.Read())
                    {
                        int id = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("addon_id"));
                        int fileId = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("file_id"));
                        long fingerprint = fingerprintReader.GetInt64(fingerprintReader.GetOrdinal("fingerprint"));
                        string folder = fingerprintReader.GetString(fingerprintReader.GetOrdinal("folder"));

                        // This is because there are paths in the DB and not Folders as previously done
                        if (folder.Contains('/'))
                        {
                            folder = folder.Substring(0, folder.IndexOf('/'));
                        }// if the folder contains a slash cut it down to it's minor component
                        else folder = string.Empty;

                        if (!individualFingerprintCache.ContainsKey(id))
                        {
                            individualFingerprintCache.Add(id, new List<CIndividualFileFingerprint>());
                        }// Add the addon to the list if it doesnt exist

                        individualFingerprintCache[id].Add(new CIndividualFileFingerprint(id, fileId, fingerprint, folder));
                    }//while there are rows to read
                }//using fingerprintReader
            }//using fingerprintCommand

            return individualFingerprintCache;
        }//Get IndividualFingerprint Cache
    }//class
}//namespace
