﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;

using Curse.Extensions;
using Curse.AddOns;
using Curse.ClientService.Extensions;
using ICSharpCode.SharpZipLib;
using ICSharpCode.SharpZipLib.BZip2;
using System.Runtime.Serialization;
using System.Web.Script.Serialization;
using System.Runtime.Serialization.Json;

namespace Curse.ClientService {
    public class CGameCache {
        public const int MinimumGameCount = 3;

        private static readonly CGameCache _instance = new CGameCache();

        private List<CGame> _games = new List<CGame>();
        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;
        private string _roamingDbConnectionString = null;
        private string _staticFileDestination = null;
        private bool _createFeedFiles = false;
        private string _versionNumber = null;

        public static CGameCache Instance {
            get {
                return _instance;
            }
        }

        private CGameCache() {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _roamingDbConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());
            _versionNumber = ConfigurationManager.AppSettings["FeedFileVersion"];

#if DEBUG
            LoadFromDisk();
#endif

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public DateTime FileDate {
            get;
            set;
        }

        public List<CGame> Games {
            get {
                return _games;
            }
        }

        public void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info,
                               null,
                               "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void UpdateCache() {

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                List<CGame> games = new List<CGame>();

                SqlCommand command = new SqlCommand("spGetGames", conn);
                command.CommandType = CommandType.StoredProcedure;

                // Populate the cache with recently modified addons:
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        CGame game = new CGame();
                        game.SetFromDataReader(conn, reader);
                        games.Add(game);
                    }
                }

                if (games.Count < MinimumGameCount) {
                    Logger.Log("Game list is incomplete. Skipping cache update, for this iteration.", ELogLevel.Info);
                    return;
                }

                lock (_games) {
                    _games = games;
                }

                try {
                    SaveToDisk(games);
                }
                catch (Exception ex) {
                    Logger.Log("Unable to create game feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }

            }
        }

        private void SaveToDisk(List<CGame> games) {
            if (_createFeedFiles) {
                SaveBinaryToDisk(games);
                SaveJsonToDisk(games);
                SaveXmlToDisk(games);
            }

            UpdateCacheDate();
        }

        private void LoadFromDisk()
        {
            List<CGame> games;
            string currentfile = Path.Combine(_staticFileDestination, "Games-v7.zip");
            if (!File.Exists(currentfile))
            {
                Logger.Log("Game feed file has not been created", ELogLevel.Info);
                return;
            }

            // Read the compressed bytes out of the avatars file
            byte[] compressed, uncompressed;
            try
            {
                using (var binReader = new BinaryReader(File.Open(currentfile, FileMode.Open)))
                {
                    compressed = binReader.ReadBytes((int)binReader.BaseStream.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to load Games from feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return;
            }

            // Deserialize the stream
            try
            {
                uncompressed = Utility.GetDecompressedBytes(compressed);
                using (var ms = new MemoryStream())
                {
                    ms.Write(uncompressed, 0, uncompressed.Length);
                    ms.Position = 0;

                    var bf = new BinaryFormatter();
                    games = (List<CGame>)bf.Deserialize(ms);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to deserialize the game feed! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return;
            }            
        }

        private void UpdateCacheDate() {
            string currentfile = null;

            try {
                currentfile = Path.Combine(_staticFileDestination, string.Format("Games{0}.zip", _versionNumber));
                FileDate = new FileInfo(currentfile).LastWriteTimeUtc;
            }
            catch (Exception ex) {
                Logger.Log("Unable to update game cache file data at '{0}'! Details: {1}", ELogLevel.Error, currentfile, ex.GetExceptionDetails());
            }
        }

        private void SaveBinaryToDisk(List<CGame> games) {
            using (MemoryStream ms = new MemoryStream()) {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, games);
                ms.Position = 0;

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, string.Format("Games{0}.zip", _versionNumber));
                try {
                    using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create))) {
                        binWriter.Write(compressed);
                    }
                }
                catch (Exception ex) {
                    throw ex;
                }

                if (File.Exists(currentfile) && FileComparer.CompareFiles(currentfile, tempfile)) {
                    try {
                        File.Delete(tempfile);
                    }
                    catch (Exception ex) {
                        Logger.Log("CGameCache - Unable to delete temp file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                    return;
                }

                try {
                    if (!File.Exists(currentfile)) {
                        File.Copy(tempfile, currentfile, true);
                    }
                    else {
                        File.Replace(tempfile, currentfile, backupfile);
                    }
                }
                catch (Exception ex) {
                    Logger.Log("CGameCache - SaveToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile)) {
                        File.Delete(backupfile);
                    }
                }
            }
        }

        private void SaveXmlToDisk(List<CGame> games)
        {
            byte[] uncompressedBytes = null;

            #region XML Serialization
            using (MemoryStream ms = new MemoryStream())
            {
                DataContractSerializer xs = new DataContractSerializer(games.GetType());
                xs.WriteObject(ms, games);
                uncompressedBytes = ms.ToArray();
            }
            #endregion

            byte[] compressedBytes = null;

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, "Games.xml.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                File.Delete(currentfile);
                File.Move(tempfile, currentfile);
            }
            catch
            {
                File.Delete(tempfile);
            }
        }

        private void SaveJsonToDisk(List<CGame> games) {
            byte[] uncompressedBytes = null;

            #region json serialization
            using (MemoryStream ms = new MemoryStream())
            {
                DataContractJsonSerializer js = new DataContractJsonSerializer(games.GetType());
                js.WriteObject(ms, games);
                uncompressedBytes = ms.ToArray();
            }
            #endregion

            byte[] compressedBytes = null;

            using (MemoryStream ms = new MemoryStream()) {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms)) {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            using (MemoryStream ms = new MemoryStream()) {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms)) {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, "Games.json.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create))) {
                binWriter.Write(compressedBytes);
            }
            try {
                File.Delete(currentfile);
                File.Move(tempfile, currentfile);
            }
            catch {
                File.Delete(tempfile);
            }
        }
    }
}
